﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/g3d/edit/detail/g3d_EditDetailDefs.h>

#if NW_G3D_CONFIG_USE_HOSTIO

#include "g3d_IEditObj.h"
#include <nw/g3d/g3d_SceneAnimObj.h>
#include "ut/g3d_FixedSizeArray.h"
#include "ut/g3d_EditWorkBuffer.h"
#include "g3d_EditAnimCurve.h"
#include "g3d_EditUtility.h"

#include <stdint.h>

namespace nw { namespace g3d {

namespace res {

class ResFile;
class ResSceneAnim;
class ResCameraAnim;
class ResLightAnim;
class ResFogAnim;

} // namespace res

namespace edit {

class IAllocator;

namespace detail {

class EditManager;
class EditCameraAnimObj;
class EditLightAnimObj;
class EditFogAnimObj;

class EditSceneAnimObj : IEditObj
{
public:
    explicit EditSceneAnimObj(IAllocator* allocator, nw::g3d::res::ResFile* resFile, ResSceneAnim* resAnim)
        : m_pAllocator(allocator)
        , m_pResFile(resFile)
        , m_pResAnim(resAnim)
        , m_IsBound(false)
        , m_IsLoopAnim(false)
    {
        NW_G3D_ASSERT_NOT_NULL(allocator);
        NW_G3D_ASSERT_NOT_NULL(resFile);
        NW_G3D_ASSERT_NOT_NULL(resAnim);
    }

    bool Setup();

    bool ReloadSceneAnimObj(nw::g3d::res::ResFile* resFile);
    bool Attach();
    void Detach();

    void EditCameraAnimCurve(
        int cameraIndex,
        int curveIndex,
        const nw::g3d::res::ResAnimCurve* resAnimCurve,
        u32 resAnimCurveSize);

    void EditLightAnimCurve(
        int lightIndex,
        int curveIndex,
        const nw::g3d::res::ResAnimCurve* resAnimCurve,
        u32 resAnimCurveSize);

    void EditFogAnimCurve(
        int fogIndex,
        int curveIndex,
        const nw::g3d::res::ResAnimCurve* resAnimCurve,
        u32 resAnimCurveSize);

    void Calc();
    void SetFrame(f32 frame);
    void SetStep(f32 step);
    void SetPlayPolicy(bool isLoopAnim);

    bool IsLoopAnim() const
    {
        return m_IsLoopAnim;
    }

    void SetBindFlag(bool enable)
    {
        m_IsBound = enable;
    }
    bool IsBound() const
    {
        return m_IsBound;
    }

    virtual bool IsResFileBound(ResFile* pResFile) const
    {
        return m_pResFile == pResFile;
    }

    ut::detail::FixedSizeArray<CameraAnimObj*>& GetCameraAnimObjPtrArray()
    {
        return m_CameraAnimObjPtrArray;
    }

    ut::detail::FixedSizeArray<LightAnimObj*>& GetLightAnimObjPtrArray()
    {
        return m_LightAnimObjPtrArray;
    }

    ut::detail::FixedSizeArray<FogAnimObj*>& GetFogAnimObjPtrArray()
    {
        return m_FogAnimObjPtrArray;
    }

    uint32_t GetResFileKey() const
    {
        return GetResFileKeyFromResFile(m_pResFile);
    }

    bool ContainsLoopAnim() const
    {
        return ContainsLoopAnimInCameraAnim()
            || ContainsLoopAnimInLightAnim()
            || ContainsLoopAnimInFogAnim();
    }

private:
    bool ContainsLoopAnimInCameraAnim() const;
    bool ContainsLoopAnimInLightAnim() const;
    bool ContainsLoopAnimInFogAnim() const;

private:
    IAllocator*                 m_pAllocator;
    nw::g3d::res::ResFile*      m_pResFile;
    nw::g3d::res::ResSceneAnim* m_pResAnim;
    bool                        m_IsBound;
    bool                        m_IsLoopAnim;

    ut::detail::FixedSizeArray<EditCameraAnimObj> m_EditCameraAnimArray;
    ut::detail::FixedSizeArray<CameraAnimObj*>    m_CameraAnimObjPtrArray;

    ut::detail::FixedSizeArray<EditLightAnimObj>  m_EditLightAnimArray;
    ut::detail::FixedSizeArray<LightAnimObj*>     m_LightAnimObjPtrArray;

    ut::detail::FixedSizeArray<EditFogAnimObj>    m_EditFogAnimArray;
    ut::detail::FixedSizeArray<FogAnimObj*>       m_FogAnimObjPtrArray;

private:
    NW_G3D_DISALLOW_COPY_AND_ASSIGN(EditSceneAnimObj);
};

class EditCameraAnimObj
{
    friend class EditSceneAnimObj;
public:
    bool Init();
    void Destroy();
    bool Setup();

    void EditCurve(
        int curveIndex,
        const nw::g3d::res::ResAnimCurve* resAnimCurve,
        u32 resAnimCurveSize);

    void SetPlayPolicy(bool isLoopAnim);
    bool IsLoopAnim() const
    {
        return m_AnimObj.GetDefaultFrameCtrl().GetPlayPolicy() == PlayPolicy_Loop;
    }

    nw::g3d::CameraAnimObj* GetAnimObj()
    {
        return &m_AnimObj;
    }
private:
    explicit EditCameraAnimObj(
        IAllocator* allocator,
        nw::g3d::res::ResCameraAnim* resAnim)
        : m_pAllocator(allocator)
        , m_pResAnim(resAnim)
        , m_WorkBuffer(allocator, CameraAnimObj::BUFFER_ALIGNMENT)
    {
        NW_G3D_ASSERT_NOT_NULL(allocator);
        NW_G3D_ASSERT_NOT_NULL(resAnim);
    }

private:
    IAllocator*                                 m_pAllocator;
    nw::g3d::CameraAnimObj                      m_AnimObj;
    nw::g3d::res::ResCameraAnim*                m_pResAnim;
    nw::g3d::AnimFrameCtrl                      m_AnimFrameCtrl;
    ut::detail::EditWorkBuffer                m_WorkBuffer;
    ut::detail::FixedSizeArray<EditAnimCurve> m_EditAnimCurveArray;
};

class EditLightAnimObj
{
    friend class EditSceneAnimObj;
public:
    bool Init();
    void Destroy();
    bool Setup();

    void EditCurve(
        int curveIndex,
        const nw::g3d::res::ResAnimCurve* resAnimCurve,
        u32 resAnimCurveSize);

    void SetPlayPolicy(bool isLoopAnim);
    bool IsLoopAnim() const
    {
        return m_AnimObj.GetDefaultFrameCtrl().GetPlayPolicy() == PlayPolicy_Loop;
    }

    nw::g3d::LightAnimObj* GetAnimObj()
    {
        return &m_AnimObj;
    }
private:
    explicit EditLightAnimObj(
        IAllocator* allocator,
        nw::g3d::res::ResLightAnim* resAnim)
        : m_pAllocator(allocator)
        , m_pResAnim(resAnim)
        , m_WorkBuffer(allocator, LightAnimObj::BUFFER_ALIGNMENT)
    {
        NW_G3D_ASSERT_NOT_NULL(allocator);
        NW_G3D_ASSERT_NOT_NULL(resAnim);
    }

private:
    IAllocator*                                 m_pAllocator;
    nw::g3d::LightAnimObj                       m_AnimObj;
    nw::g3d::res::ResLightAnim*                 m_pResAnim;
    nw::g3d::AnimFrameCtrl                      m_AnimFrameCtrl;
    ut::detail::EditWorkBuffer                m_WorkBuffer;
    ut::detail::FixedSizeArray<EditAnimCurve> m_EditAnimCurveArray;
};

class EditFogAnimObj
{
    friend class EditSceneAnimObj;
public:
    bool Init();
    void Destroy();
    bool Setup();

    void EditCurve(
        int curveIndex,
        const nw::g3d::res::ResAnimCurve* resAnimCurve,
        u32 resAnimCurveSize);

    void SetPlayPolicy(bool isLoopAnim);
    bool IsLoopAnim() const
    {
        return m_AnimObj.GetDefaultFrameCtrl().GetPlayPolicy() == PlayPolicy_Loop;
    }

    nw::g3d::FogAnimObj* GetAnimObj()
    {
        return &m_AnimObj;
    }
private:
    explicit EditFogAnimObj(
        IAllocator* allocator,
        nw::g3d::res::ResFogAnim* resAnim)
        : m_pAllocator(allocator)
        , m_pResAnim(resAnim)
        , m_WorkBuffer(allocator, FogAnimObj::BUFFER_ALIGNMENT)
    {
        NW_G3D_ASSERT_NOT_NULL_DETAIL(allocator, "%s\n", NW_G3D_RES_GET_NAME(resAnim, GetName()));
        NW_G3D_ASSERT_NOT_NULL(resAnim);
    }

private:
    IAllocator*                                 m_pAllocator;
    nw::g3d::FogAnimObj                         m_AnimObj;
    nw::g3d::res::ResFogAnim*                   m_pResAnim;
    nw::g3d::AnimFrameCtrl                      m_AnimFrameCtrl;
    ut::detail::EditWorkBuffer                m_WorkBuffer;
    ut::detail::FixedSizeArray<EditAnimCurve> m_EditAnimCurveArray;
};

}}}} // namespace nw::g3d::edit::detail

#endif // NW_G3D_CONFIG_USE_HOSTIO
