﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditModelObj.h"

#if NW_G3D_CONFIG_USE_HOSTIO

#include <nw/g3d/res/g3d_ResFile.h>
#include <nw/g3d/res/g3d_ResShader.h>
#include "g3d_EditMaterialObj.h"
#include "g3d_EditShapeObj.h"
#include <nw/g3d/edit/g3d_IAllocator.h>

#include "g3d_EditAnimObj.h"

namespace nw { namespace g3d { namespace edit { namespace detail {

using namespace ut::detail;

EditModelObj::EditModelObj(IAllocator* pAllocator, ResFile* resFile, ResModel* resModel, ResFile* pFirstLoadResFile)
    : m_pAllocator(pAllocator)
    , m_pResFileCreatedInLoadFileCallback(resFile)
    , m_pOriginalResModel(resModel)
    , m_pSecondResFile(NULL)
    , m_pSecondResModel(NULL)
    , m_pOriginalModelObj(NULL)
    , m_WorkBlockBuffer(pAllocator, ModelObj::BLOCK_BUFFER_ALIGNMENT)
    , m_Skeleton(pAllocator)
    , m_BoundAnimPtrArray(pAllocator, DEFAULT_ALIGNMENT)
    , m_HasBlinkingMaterials(false)
    , m_HasBlinkingShapes(false)
    , m_pUpdateResFile(NULL)
    , m_pFirstLoadResFile(pFirstLoadResFile)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pAllocator, "%s\n", NW_G3D_RES_GET_NAME(resModel, GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(resFile, "%s\n", NW_G3D_RES_GET_NAME(resModel, GetName()));
    NW_G3D_ASSERT_NOT_NULL(resModel);
    NW_G3D_ASSERT_NOT_NULL(pFirstLoadResFile);
    m_Key = GetResFileKeyFromResFile(resFile);
}

EditModelObj::EditModelObj(IAllocator* allocator, ResModel* resModel, ResFile* pFirstLoadResFile)
    : m_pAllocator(allocator)
    , m_pResFileCreatedInLoadFileCallback(NULL)
    , m_pOriginalResModel(resModel)
    , m_pSecondResFile(NULL)
    , m_pSecondResModel(NULL)
    , m_pOriginalModelObj(NULL)
    , m_WorkBlockBuffer(allocator, ModelObj::BLOCK_BUFFER_ALIGNMENT)
    , m_Skeleton(allocator)
    , m_BoundAnimPtrArray(allocator, DEFAULT_ALIGNMENT)
    , m_HasBlinkingMaterials(false)
    , m_ShapeLodLevel(-1)
    , m_pUpdateResFile(NULL)
    , m_pFirstLoadResFile(pFirstLoadResFile)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(allocator, "%s\n", NW_G3D_RES_GET_NAME(resModel, GetName()));
    NW_G3D_ASSERT_NOT_NULL(resModel);
    NW_G3D_ASSERT_NOT_NULL(pFirstLoadResFile);
    m_Key = 0;
}

int EditModelObj::GetMaterialCount() const
{
    return m_MaterialArray.Size();
}

int EditModelObj::GetShaderCount() const
{
    return m_ShaderPtrArray.Size();
}

const ResModel* EditModelObj::GetResModel() const
{
    return m_pOriginalResModel;
}

bool EditModelObj::AddBoundAnim(EditAnimObj* pEditAnimObj)
{
    return m_BoundAnimPtrArray.PushBack(pEditAnimObj);
}

bool EditModelObj::RemoveBoundAnim(int index)
{
    if (index < 0 || index >= m_BoundAnimPtrArray.Size())
    {
        return false;
    }

    m_BoundAnimPtrArray.Erase(index);
    return true;
}

bool EditModelObj::RemoveBoundAnim(EditAnimObj* pEditAnimObj)
{
    int index = m_BoundAnimPtrArray.IndexOf(pEditAnimObj);
    if (index < 0)
    {
        return false;
    }

    m_BoundAnimPtrArray.Erase(index);
    return true;
}

int EditModelObj::GetBoundAnimCount() const
{
    return m_BoundAnimPtrArray.Size();
}

EditAnimObj* EditModelObj::GetBoundAnimAt(int index)
{
    return m_BoundAnimPtrArray.UnsafeAt(index);
}

int EditModelObj::GetIndexOfBoundAnim(EditAnimObj* pEditAnimObj)
{
    return m_BoundAnimPtrArray.IndexOf(pEditAnimObj);
}

void
EditModelObj::UpdateRenderInfo(
    int materialIndex,
    const void* pRenderInfoDic,
    size_t dataSize)
{
    NW_G3D_EDIT_ASSERTMSG(materialIndex >= 0 && materialIndex < m_MaterialArray.Size(), "%s\n", NW_G3D_RES_GET_NAME(GetTargetModelObj()->GetResource(), GetName()));
    EditMaterialObj* materialObj = m_MaterialArray.UnsafeAt(materialIndex);
    materialObj->UpdateRenderInfo(pRenderInfoDic, dataSize);
}

bool
EditModelObj::IsSendAttachModel() const
{
    NW_G3D_ASSERT_NOT_NULL(m_pOriginalResModel);

    // SendAttachModel時に ModelObj は指定されているので、
    // NULL の時は、アタッチモデルではない。
    if (m_pOriginalModelObj == NULL)
    {
        return false;
    }

    if (m_pResFileCreatedInLoadFileCallback == NULL)
    {
        return true;
    }
    return false;
}

RuntimeErrorCode
EditModelObj::AttachEditModelObj(nw::g3d::ModelObj* pModelObj, EditCallback* pCallback)
{
    NW_G3D_ASSERT_NOT_NULL(pModelObj);
    m_pCallback = pCallback;

    {
        NW_G3D_ASSERT(m_pOriginalModelObj == NULL);
        NW_G3D_ASSERT_NOT_NULL(pModelObj);

        m_pOriginalModelObj = pModelObj;
        m_Key = reinterpret_cast<u32>(m_pOriginalModelObj);

        ResModel* pResModel = pModelObj->GetResource();
        NW_G3D_ASSERT_NOT_NULL(pResModel);
        m_pOriginalResModel = pResModel;

        memcpy(m_DestroyCheckData, m_pOriginalModelObj, s_CheckNumBytes);
        strncpy(m_ModelName, m_pOriginalResModel->GetName(), s_ModelNameBufferSize);
    }

    if (!m_Skeleton.Init(m_pOriginalModelObj, m_pOriginalModelObj->GetSkeleton()))
    {
        return RUNTIME_ERROR_CODE_UNKNOWN_ERROR;
    }

    int materialCount = m_pOriginalModelObj->GetMaterialCount();
    if (materialCount > 0)
    {
        size_t bufferSize = FixedSizeArray<EditMaterialObj>::CalcBufferSize(materialCount);
        void* buffer = m_pAllocator->Alloc(bufferSize, DEFAULT_ALIGNMENT);
        if (buffer == NULL) // バッファ確保失敗
        {
            return RUNTIME_ERROR_CODE_INSUFFICIENT_MEMORY;
        }

        m_MaterialArray.SetArrayBuffer(buffer, materialCount);
        bool isFailed = false;
        for(int i = 0, end = m_MaterialArray.Size(); i < end; ++i)
        {
            nw::g3d::MaterialObj* materialObj = m_pOriginalModelObj->GetMaterial(i);
            EditMaterialObj* editMaterialObj = m_MaterialArray.UnsafeAt(i);
            new (editMaterialObj) EditMaterialObj(m_pAllocator, i, m_pOriginalModelObj, materialObj);
            isFailed |= !editMaterialObj->Init();
        }

        // 初期化失敗のものが一つでもあった場合は、インスタンスを破棄して失敗にする。
        if (isFailed)
        {
            CleanupMaterials();
            m_Skeleton.Cleanup();
            return RUNTIME_ERROR_CODE_UNKNOWN_ERROR;
        }
    }

    ResSkeleton* resSkeleton = m_pOriginalModelObj->GetSkeleton()->GetResource();
    int shapeCount = m_pOriginalModelObj->GetShapeCount();
    if (shapeCount > 0)
    {
        size_t bufferSize = FixedSizeArray<EditShapeObj>::CalcBufferSize(shapeCount);
        void* buffer = m_pAllocator->Alloc(bufferSize, DEFAULT_ALIGNMENT);
        if (buffer == NULL) // バッファ確保失敗
        {
            CleanupMaterials();
            m_Skeleton.Cleanup();
            return RUNTIME_ERROR_CODE_INSUFFICIENT_MEMORY;
        }

        m_ShapeArray.SetArrayBuffer(buffer, shapeCount);
        bool isFailed = false;
        for(int i = 0, end = m_ShapeArray.Size(); i < end; ++i)
        {
            nw::g3d::ShapeObj* shapeObj = m_pOriginalModelObj->GetShape(i);
            EditShapeObj* editShapeObj = m_ShapeArray.UnsafeAt(i);
            new (editShapeObj) EditShapeObj(m_pAllocator, i, m_pOriginalModelObj, shapeObj);

            ResBone* resBone = resSkeleton->GetBone(shapeObj->GetBoneIndex());
            editShapeObj->SetViewDependent(resBone->GetBillboardMode() != ResBone::BILLBOARD_NONE);

            editShapeObj->ReinitBuffer(shapeObj->GetResource());
            editShapeObj->SetupShapeInstance(shapeObj->GetResource());
            isFailed |= !editShapeObj->Init();
        }

        // 初期化失敗のものが一つでもあった場合は、インスタンスを破棄して失敗にする。
        if (isFailed)
        {
            CleanupMaterials();
            m_Skeleton.Cleanup();
            return RUNTIME_ERROR_CODE_UNKNOWN_ERROR;
        }
    }

    //　インスタンス生成に成功しているので、セットアップを行う。
    for(int i = 0, end = m_MaterialArray.Size(); i < end; ++i)
    {
        EditMaterialObj* editMaterial = m_MaterialArray.UnsafeAt(i);
        editMaterial->Setup();
    }

    m_pOriginalModelObj->UpdateViewDependency();

    return ReloadResFile(m_pFirstLoadResFile, pCallback);
}

void
EditModelObj::DestroyEditModelObj()
{
    m_BoundAnimPtrArray.Destroy();

    // マテリアルの破棄
    {
        for(int i = 0, end = m_MaterialArray.Size(); i < end; ++i)
        {
            EditMaterialObj* editMaterialObj = m_MaterialArray.UnsafeAt(i);
            editMaterialObj->Destroy();
        }

        void* buffer = m_MaterialArray.GetBufferPtr();
        if (buffer != NULL)
        {
            m_MaterialArray.Clear();
            m_pAllocator->Free(buffer);
        }
    }

    // シェイプの破棄
    {
        for(int i = 0, end = m_ShapeArray.Size(); i < end; ++i)
        {
            EditShapeObj* editShapeObj = m_ShapeArray.UnsafeAt(i);
            editShapeObj->Destroy();
        }

        void* buffer = m_ShapeArray.GetBufferPtr();
        if (buffer != NULL)
        {
            m_ShapeArray.Clear();
            m_pAllocator->Free(buffer);
        }
    }

    m_Skeleton.Destroy();

    DestroyOnetimeShaders();
    DestroyShaders();
}

void
EditModelObj::ResetToOriginal(nw::g3d::edit::EditCallback* pCallback)
{
    // オリジナルモデルに変更があるかをチェックする
    // 変更があった場合はアプリケーションの実装の問題で元モデルが破棄されてしまった可能性がある
    NW_G3D_ASSERTMSG(0 == memcmp(m_DestroyCheckData, m_pOriginalModelObj, s_CheckNumBytes),
        "Model \"%s\" may have been destoyed or rewritten illegally before detatching from edit library",
        m_ModelName);

    SwapResModel(m_pSecondResModel, m_pOriginalResModel, m_pOriginalResModel, false, pCallback);
    CleanupEditModelObj();
    DestroyEditModelObj();

    if (pCallback != NULL)
    {
        if (this->IsSendAttachModel())
        {
            DetachModelArg arg;
            arg.modelObj = this->GetTargetModelObj();
            pCallback->DetachModel(arg);
        }
        else
        {
            UnloadFileArg arg;
            arg.resFile = m_pResFileCreatedInLoadFileCallback;
            pCallback->UnloadFile(arg);
        }
    }
}

RuntimeErrorCode
EditModelObj::ReloadResFile(nw::g3d::res::ResFile* resFile, nw::g3d::edit::EditCallback* callback)
{
    NW_G3D_ASSERT_NOT_NULL(resFile);
    NW_G3D_ASSERTMSG(resFile->GetModelCount() > 0, "%s\n", NW_G3D_RES_GET_NAME(resFile, GetName()));
    NW_G3D_ASSERT_NOT_NULL(m_pFirstLoadResFile);

    // 送信されていないテクスチャは初回ロード時の ResFile からバインドする。
    resFile->Bind(m_pFirstLoadResFile, nw::g3d::res::BIND_TEXTURE);

    nw::g3d::res::ResModel* resModel = resFile->GetModel(0);
    NW_G3D_ASSERT_NOT_NULL_DETAIL(resModel, "%s\n", NW_G3D_RES_GET_NAME(resFile, GetName()));

    if (m_pOriginalResModel->GetMaterialCount() != resModel->GetMaterialCount())
    {
        NW_G3D_WARNING(false, "material count is not equal.");
        return RUNTIME_ERROR_CODE_INVALID_MATERIAL_COUNT;
    }

    SwapResModel(m_pSecondResModel, resModel, resModel, false, callback);

    m_pSecondResFile = resFile;
    m_pSecondResModel = resModel;

    return RUNTIME_ERROR_CODE_NO_ERROR;
}

void
EditModelObj::CleanupEditModelObj()
{
    m_BoundAnimPtrArray.Clear();

    CleanupMaterials();
    CleanupShapes();
    m_Skeleton.Cleanup();
    CleanupShaders();
    CleanupOnetimeShaders();
}

void
EditModelObj::CleanupMaterials()
{
    for(int i = 0, end = m_MaterialArray.Size(); i < end; ++i)
    {
        EditMaterialObj* editMaterialObj = m_MaterialArray.UnsafeAt(i);
        editMaterialObj->Cleanup();
    }
}

void
EditModelObj::CleanupShapes()
{
    for(int i = 0, end = m_ShapeArray.Size(); i < end; ++i)
    {
        EditShapeObj* editShapeObj = m_ShapeArray.UnsafeAt(i);
        editShapeObj->Cleanup();
    }
}

void
EditModelObj::CleanupShaders()
{
    for (int i = 0, end = m_ShaderPtrArray.Size(); i < end; ++i)
    {
        ResShaderArchive** resShaderArchive = m_ShaderPtrArray.UnsafeAt(i);
        if (*resShaderArchive != NULL)
        {
            (*resShaderArchive)->Cleanup();
        }
    }
}

void
EditModelObj::CleanupOnetimeShaders()
{
    for (int i = 0, end = m_OnetimeShaderPtrArray.Size(); i < end; ++i)
    {
        ResShaderArchive** resShaderArchive = m_OnetimeShaderPtrArray.UnsafeAt(i);
        if (*resShaderArchive != NULL)
        {
            (*resShaderArchive)->Cleanup();
        }
    }
}

void
EditModelObj::DestroyOnetimeShaders()
{
    for (int i = 0, end = m_OnetimeShaderPtrArray.Size(); i < end; ++i)
    {
        ResShaderArchive** resShaderArchive = m_OnetimeShaderPtrArray.UnsafeAt(i);
        if (*resShaderArchive != NULL)
        {
            m_pAllocator->Free(*resShaderArchive);

            for (int j = i + 1; j < end; ++j)
            {
                ResShaderArchive** shader = m_OnetimeShaderPtrArray.UnsafeAt(j);
                if (*shader == *resShaderArchive)
                {
                    *shader = NULL;
                }
            }

            *resShaderArchive = NULL;
        }
    }

    if (void* buffer = m_OnetimeShaderPtrArray.GetBufferPtr())
    {
        m_OnetimeShaderPtrArray.Clear();
        m_pAllocator->Free(buffer);
    }
}

void
EditModelObj::DestroyShaders()
{
    for (int i = 0, end = m_ShaderPtrArray.Size(); i < end; ++i)
    {
        ResShaderArchive** resShaderArchive = m_ShaderPtrArray.UnsafeAt(i);
        if (*resShaderArchive != NULL)
        {
            m_pAllocator->Free(*resShaderArchive);

            for (int j = i + 1; j < end; ++j)
            {
                ResShaderArchive** shader = m_ShaderPtrArray.UnsafeAt(j);
                if (*shader == *resShaderArchive)
                {
                    *shader = NULL;
                }
            }

            *resShaderArchive = NULL;
        }
    }

    if (void* buffer = m_ShaderPtrArray.GetBufferPtr())
    {
        m_ShaderPtrArray.Clear();
        m_pAllocator->Free(buffer);
    }
}

void
EditModelObj::UpdateMaterialBlinking()
{
    if (!m_HasBlinkingMaterials)
    {
        return;
    }

    m_HasBlinkingMaterials = false;
    for(int i = 0, end = m_MaterialArray.Size(); i < end; ++i)
    {
        EditMaterialObj* pEditMaterialObj = m_MaterialArray.UnsafeAt(i);
        if (!pEditMaterialObj->IsBlinking())
        {
            continue;
        }

        pEditMaterialObj->UpdateBlinking();
        m_HasBlinkingMaterials = true;
    }
}

void
EditModelObj::StartMaterialBlinking(const SelectTargetArg& arg)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(arg.index, "%s\n", NW_G3D_RES_GET_NAME(GetTargetModelObj()->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(arg.indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(GetTargetModelObj()->GetResource(), GetName()));

    for (u32 i = 0; i < arg.indexSize; ++i)
    {
        EditMaterialObj* editMaterialObj = m_MaterialArray.UnsafeAt(arg.index[i]);
        if (m_pOriginalModelObj->IsMatVisible(arg.index[i]))
        {
            editMaterialObj->StartBlinking();
        }
    }

    m_HasBlinkingMaterials = true;
}

void
EditModelObj::BreakMaterialBlinking(const SelectTargetArg& arg)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(arg.index, "%s\n", NW_G3D_RES_GET_NAME(GetTargetModelObj()->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(arg.indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(GetTargetModelObj()->GetResource(), GetName()));

    for (u32 i = 0; i < arg.indexSize; ++i)
    {
        EditMaterialObj* editMaterialObj = m_MaterialArray.UnsafeAt(arg.index[i]);
        editMaterialObj->BreakBlinking();
    }
}

void EditModelObj::StartShapeBlinking(const SelectTargetArg& arg)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(arg.index, "%s\n", NW_G3D_RES_GET_NAME(GetTargetModelObj()->GetResource(), GetName()));
    NW_G3D_ASSERTMSG(arg.indexSize > 0, "%s\n", NW_G3D_RES_GET_NAME(GetTargetModelObj()->GetResource(), GetName()));

    for (u32 i = 0; i < arg.indexSize; ++i)
    {
        EditShapeObj* pEditShapeObj = m_ShapeArray.UnsafeAt(arg.index[i]);
        pEditShapeObj->StartBlinking();
    }

    m_HasBlinkingShapes = true;
}

void EditModelObj::StartModelBlinking()
{
    const ResModel* pResModel = m_pOriginalResModel;
    int shapeCount = pResModel->GetShapeCount();
    for (int i = 0; i < shapeCount; ++i)
    {
        EditShapeObj* pEditShapeObj = m_ShapeArray.UnsafeAt(i);
        pEditShapeObj->StartBlinking();
    }

    m_HasBlinkingShapes = true;
}

void
EditModelObj::UpdateShapeBlinking()
{
    if (!m_HasBlinkingShapes)
    {
        return;
    }

    m_HasBlinkingShapes = false;
    bool isShapeUpdated = false;
    for(int i = 0, end = m_ShapeArray.Size(); i < end; ++i)
    {
        EditShapeObj* pEditShapeObj = m_ShapeArray.UnsafeAt(i);
        if (!pEditShapeObj->IsBlinking())
        {
            continue;
        }

        isShapeUpdated |= pEditShapeObj->UpdateBlinking();
        m_HasBlinkingShapes = true;
    }

    if (isShapeUpdated)
    {
        CallShapeUpdatedCallback();
    }
}

void
EditModelObj::SetLodLevel(int level)
{
    if (m_ShapeLodLevel == level)
    {
        return;
    }

    m_ShapeLodLevel = level;

    for (int i = 0, end = m_ShapeArray.Size(); i < end; ++i)
    {
        EditShapeObj* editShapeObj = m_ShapeArray.UnsafeAt(i);
        editShapeObj->SetLodLevel(level);
    }

    CallShapeUpdatedCallback();
}

void EditModelObj::CallShapeUpdatedCallback()
{
    if (m_pCallback == NULL)
    {
        return;
    }

    ShapeUpdatedArg arg;
    arg.pModelObj = m_pOriginalModelObj;
    m_pCallback->ShapeUpdated(arg);
}

void
EditModelObj::ResetLodLevel()
{
    m_ShapeLodLevel = -1;

    for (int i = 0, end = m_ShapeArray.Size(); i < end; ++i)
    {
        EditShapeObj* editShapeObj = m_ShapeArray.UnsafeAt(i);
        editShapeObj->ResetLodLevel();
    }
}

void
EditModelObj::SwapResModel(
    nw::g3d::res::ResModel* oldResModel,
    nw::g3d::res::ResModel* newResModel,
    nw::g3d::res::ResModel* swapTargetResModel,
    bool useShaders,
    nw::g3d::edit::EditCallback* callback
)
{
    CleanupMaterials();
    CleanupShapes();
    m_Skeleton.Cleanup();

    {
        nw::g3d::res::ResSkeleton* resSkeleton = swapTargetResModel->GetSkeleton();

        for(int i = 0, end = m_MaterialArray.Size(); i < end; ++i)
        {
            nw::g3d::res::ResMaterial* resMaterial = swapTargetResModel->GetMaterial(i);
            EditMaterialObj* editMaterialObj = m_MaterialArray.UnsafeAt(i);
            bool result =  editMaterialObj->ReinitBuffer(resMaterial);
            NW_G3D_ASSERT(result); // 今は止める
        }

        for(int i = 0, end = m_ShapeArray.Size(); i < end; ++i)
        {
            nw::g3d::res::ResShape* resShape = swapTargetResModel->GetShape(i);
            EditShapeObj* editShapeObj = m_ShapeArray.UnsafeAt(i);

            ResBone* resBone = resSkeleton->GetBone(resShape->GetBoneIndex());
            editShapeObj->SetViewDependent(resBone->GetBillboardMode() != ResBone::BILLBOARD_NONE);

            bool result =  editShapeObj->ReinitBuffer(resShape);
            NW_G3D_ASSERT(result); // 今は止める
        }

        {
            bool result = m_Skeleton.ReinitBuffer(resSkeleton);
            NW_G3D_ASSERT(result); // 今は止める
        }

        if (callback != NULL)
        {
            UpdateShaderAssignArg arg;
            arg.modelObj = m_pOriginalModelObj;
            arg.oldResModel = oldResModel;
            arg.newResModel = newResModel;
            arg.state = UpdateShaderAssignArg::STATE_MODEL_UPDATE;
            arg.shaderArchivePtrs = NULL;
            arg.numShaderArchive = 0;

            if (useShaders)
            {
                arg.shaderArchivePtrs = m_ShaderPtrArray.GetArrayBufferPtr();
                arg.numShaderArchive = m_ShaderPtrArray.Size();
            }

            // NULL ならばオリジナルのResModelを入れる。
            // 更新前のResModelがオリジナルの場合はフラグを有効にする。
            // モデルをアタッチした場合は以下の状態にする。
            if (arg.oldResModel == NULL)
            {
                arg.oldResModel = m_pOriginalResModel;
                arg.state = UpdateShaderAssignArg::STATE_MODEL_BEGIN;
            }

            // 更新用のResModelがオリジナルの場合はフラグを有効にする。
            // モデルをデタッチした場合は以下の状態にする。
            if (m_pOriginalResModel == newResModel)
            {
                arg.state = UpdateShaderAssignArg::STATE_MODEL_END;
            }

            callback->UpdateShaderAssign(arg);
        }
    }

    {
        nw::g3d::res::ResSkeleton* resSkeleton = swapTargetResModel->GetSkeleton();
        m_Skeleton.SetupSkeletonInstance(resSkeleton);
    }

    for(int i = 0, end = m_ShapeArray.Size(); i < end; ++i)
    {
        nw::g3d::res::ResShape* resShape = swapTargetResModel->GetShape(i);
        EditShapeObj* editShapeObj = m_ShapeArray.UnsafeAt(i);
        editShapeObj->SetupShapeInstance(resShape);
    }

    for(int i = 0, end = m_MaterialArray.Size(); i < end; ++i)
    {
        nw::g3d::res::ResMaterial* resMaterial = swapTargetResModel->GetMaterial(i);
        EditMaterialObj* editMaterialObj = m_MaterialArray.UnsafeAt(i);
        editMaterialObj->SetupMaterialInstance(resMaterial);
    }

    m_pOriginalModelObj->UpdateViewDependency();
    m_pOriginalModelObj->ClearBoneVisibility();
    m_pOriginalModelObj->ClearMatVisibility();

    if (m_ShapeLodLevel >= 0 && m_pOriginalResModel != newResModel)
    {
        SetLodLevel(m_ShapeLodLevel);
    }

    if (callback != NULL)
    {
        UpdateMaterialsArg arg;
        arg.modelObj = m_pOriginalModelObj;
        arg.state = UpdateMaterialsArg::STATE_UPDATE;

        // oldResModel が NULL の場合は初回実行
        if (oldResModel == NULL)
        {
            arg.state = UpdateMaterialsArg::STATE_BEGIN;
        }

        // newResModel が m_pOriginalResModel と同一の場合は終了時実行
        if (m_pOriginalResModel == newResModel)
        {
            arg.state = UpdateMaterialsArg::STATE_END;
        }

        callback->UpdateMaterials(arg);
    }
}

void
EditModelObj::SetShaderSize(u32 size)
{
    u32 count = m_ShaderPtrArray.Size();
    if (count > 0) // 設定済みの場合は一度解放
    {
        // コールバック終了までは、シェーダアーカイブを保持するために一旦退避する。
        {
            size_t bufferSize = m_OnetimeShaderPtrArray.CalcBufferSize(count);
            void* buffer = m_pAllocator->Alloc(bufferSize, DEFAULT_ALIGNMENT);
            NW_G3D_ASSERT_NOT_NULL(buffer);
            memset(buffer, 0, bufferSize);
            m_OnetimeShaderPtrArray.SetArrayBuffer(buffer, count);
        }

        for (u32 i = 0; i < count; ++i)
        {
            ResShaderArchive** shaderArchive = m_ShaderPtrArray.UnsafeAt(i);
            ResShaderArchive** targetShaderArchive = m_OnetimeShaderPtrArray.UnsafeAt(i);
            *targetShaderArchive = *shaderArchive;
            *shaderArchive = NULL;
        }
    }

    if (size > count)
    {
        void* buffer = m_ShaderPtrArray.GetBufferPtr();
        if (buffer != NULL)
        {
            m_pAllocator->Free(buffer);
            m_ShaderPtrArray.Clear();
            buffer = NULL;
        }

        {
            size_t bufferSize = m_ShaderPtrArray.CalcBufferSize(size);
            buffer = m_pAllocator->Alloc(bufferSize, DEFAULT_ALIGNMENT);
            NW_G3D_ASSERT_NOT_NULL(buffer);
            memset(buffer, 0, bufferSize);
            m_ShaderPtrArray.SetArrayBuffer(buffer, size);
        }
    }
}

void
EditModelObj::SetShader(int index, ResShaderArchive* shaderArchive)
{
    ResShaderArchive** targetShaderArchive = m_ShaderPtrArray.UnsafeAt(index);
    NW_G3D_EDIT_ASSERT(*targetShaderArchive == NULL);
    *targetShaderArchive = shaderArchive;
}

void
EditModelObj::UpdateShaders(nw::g3d::edit::EditCallback* callback, bool useShaders)
{
    NW_G3D_EDIT_ASSERT(m_ShaderPtrArray.Size() > 0 && m_ShaderPtrArray.GetArrayBufferPtr() != NULL);

    NW_G3D_ASSERT_NOT_NULL(m_pUpdateResFile);
    NW_G3D_ASSERT(m_pUpdateResFile->GetModelCount() > 0);
    NW_G3D_ASSERT_NOT_NULL(m_pFirstLoadResFile);

    // 送信されていないテクスチャは初回ロード時の ResFile からバインドする。
    m_pUpdateResFile->Bind(m_pFirstLoadResFile, nw::g3d::res::BIND_TEXTURE);

    nw::g3d::res::ResModel* resModel = m_pUpdateResFile->GetModel(0);
    NW_G3D_ASSERT_NOT_NULL(resModel);

    if (m_pOriginalResModel->GetMaterialCount() != resModel->GetMaterialCount())
    {
        NW_G3D_WARNING(false, "material count is not equal.");
        return;
    }

    SwapResModel(m_pSecondResModel, resModel, resModel, useShaders, callback);

    m_pSecondResFile = m_pUpdateResFile;
    m_pSecondResModel = resModel;
    SetUpdateResFile(NULL);
}

}}}} // namespace nw::g3d::edit::detail

#endif // NW_G3D_CONFIG_USE_HOSTIO
