﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/account.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/fs/fs_Result.h>
#include <nn/nd/detail/nd_Log.h>
#include <nn/nd/detail/nd_Result.h>
#include <nn/nd/service/nd_UserIdManager.h>
#include <nn/os/os_Random.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_LockGuard.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_UuidApi.h>

namespace nn { namespace nd { namespace service {

namespace {

LocalUserId GenerateLocalUserId() NN_NOEXCEPT
{
    auto uuid = util::GenerateUuid();
    LocalUserId id;
    std::memcpy(id._data, uuid.data, sizeof(LocalUserId::_data));
    return id;
}
NN_STATIC_ASSERT(sizeof(LocalUserId::_data) == util::Uuid::Size);

NetworkUserId GenerateDummyNetworkUserId() NN_NOEXCEPT
{
#if 1
    NetworkUserId id;
    os::GenerateRandomBytes(&id._id, sizeof(id._id));
    return id;
#else
    return NetworkUserId{ 0 };
#endif
}

util::optional<account::NetworkServiceAccountId> GetNetworkServiceAccountId(const account::Uid& uid) NN_NOEXCEPT
{
    account::NetworkServiceAccountManager manager;
    NN_ABORT_UNLESS_RESULT_SUCCESS(account::GetNetworkServiceAccountManager(&manager, uid));
    bool isNsaAvailable;
    NN_ABORT_UNLESS_RESULT_SUCCESS(manager.IsNetworkServiceAccountAvailable(&isNsaAvailable));
    if( !isNsaAvailable )
    {
        return util::nullopt;
    }
    account::NetworkServiceAccountId nsaId;
    auto result = manager.GetNetworkServiceAccountId(&nsaId);
    if( result.IsFailure() )
    {
        NN_DETAIL_ND_WARN("[nd] GetNetworkServiceAccountId : Failed to get NetworkServiceAccountId of uid<%016llx-%016llx> : %03d-%04d (0x%08x)\n",
            uid._data[0], uid._data[1], result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
        return util::nullopt;
    }
    return nsaId;
}

} // ~nn::nd::service::<anonymous>

void UserIdManager::UserId::Initialize(const account::Uid& uid) NN_NOEXCEPT
{
    UserId u;
    auto loadResult = LoadUserId(&u, uid);
    if( fs::ResultPathNotFound::Includes(loadResult) )
    {
        m_Uid = uid;
        m_LocalUserId = GenerateLocalUserId();
        m_NsaId = GetNetworkServiceAccountId(uid);
        if( m_NsaId )
        {
            m_IsNetworkUserIdUpdateRequested = true;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(StoreUserId(*this));
        return;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(loadResult);
    *this = u;
    auto nsaId = GetNetworkServiceAccountId(uid);
    if( (m_NsaId != nsaId) || (m_NsaId && !m_NetworkUserId) )
    {
        // セーブデータから読み込んだ NetworkServiceAccountId と取得した NetworkServiceAccountId が異なる場合、
        // もしくは NetworkServiceAccountId があるにも関わらず NetworkUserId がない場合。
        // ネットワークサービスアカウントの可用性の変化があった際、ndがそれを検知して必要なデータを更新・保存する前に電源が切られた場合を想定。
        if( m_NsaId )
        {
            m_IsNetworkUserIdUpdateRequested = true;
        }
        m_NetworkUserId = util::nullopt;
        NN_ABORT_UNLESS_RESULT_SUCCESS(StoreUserId(*this));
    }
}

void UserIdManager::UserId::Reset() NN_NOEXCEPT
{
    *this = UserId{};
}

//! 未実装
/*static*/ Result UserIdManager::StoreUserId(const UserIdManager::UserId& u) NN_NOEXCEPT
{
    // TODO: アカウントセーブデータへの保存。
    NN_UNUSED(u);
    NN_RESULT_SUCCESS;
}

//! 未実装
/*static*/ Result UserIdManager::LoadUserId(UserIdManager::UserId* pOut, const account::Uid& uid) NN_NOEXCEPT
{
    // TODO: アカウントセーブデータからの読み込み。
    NN_UNUSED(pOut);
    NN_UNUSED(uid);
    return fs::ResultPathNotFound();
}

void UserIdManager::Initialize(const util::Span<const account::Uid>& uids) NN_NOEXCEPT
{
    int count = 0;
    for( const auto& uid : uids )
    {
        m_UserIds[count++].Initialize(uid);
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&m_ThreadType, UpdateNetworkUserIdThread, this, m_ThreadStack, m_ThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(nd, Network)));
    os::SetThreadNamePointer(&m_ThreadType, NN_SYSTEM_THREAD_NAME(nd, Network));
    os::StartThread(&m_ThreadType);
    // Initialize からの要求があった場合のために一度 Signal() する。
    m_ConditionalVariable.Signal();
}

void UserIdManager::Update(const util::Span<const account::Uid>& newUids) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    UpdateUnsafe(newUids);
}

LocalUserId UserIdManager::GetLocalUserId(const account::Uid& uid) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    return FindOrInitUserIdUnsafe(uid)->m_LocalUserId;
}

util::optional<NetworkUserId> UserIdManager::GetNetworkUserId(const account::Uid& uid) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    return FindOrInitUserIdUnsafe(uid)->m_NetworkUserId;
}

void UserIdManager::RefreshNetworkServiceAccountId() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    for( auto& u : m_UserIds )
    {
        if( !u.m_Uid )
        {
            continue;
        }
        auto preNsaId = u.m_NsaId;
        u.m_NsaId = GetNetworkServiceAccountId(u.m_Uid);
        if( u.m_NsaId != preNsaId )
        {
            u.m_IsNetworkUserIdUpdateRequested = u.m_NsaId;
            if( preNsaId )
            {
                // 変更前の NetworkServiceAccountId に紐付いている NetworkUserId を無効化する。
                NN_DETAIL_ND_TRACE("[nd] UserIdManager::RefreshNetworkServiceAccountId : Invalidate uid<%016llx-%016llx>'s old NetworkUserId.\n",
                    u.m_Uid._data[0], u.m_Uid._data[1]);
                u.m_NetworkUserId = util::nullopt;
                m_NetworkUserIdUpdatedEvent.Signal();
            }
            if( u.m_NsaId )
            {
                // 新しい NetworkUserId を取りにいくトリガーをかける。
                NN_DETAIL_ND_TRACE("[nd] UserIdManager::RefreshNetworkServiceAccountId : Request updating uid<%016llx-%016llx>'s NetworkUserId.\n",
                    u.m_Uid._data[0], u.m_Uid._data[1]);
                m_ConditionalVariable.Signal();
            }
            NN_ABORT_UNLESS_RESULT_SUCCESS(StoreUserId(u));
        }
    }
}

void UserIdManager::UpdateUnsafe(const util::Span<const account::Uid>& newUids) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Mutex.IsLockedByCurrentThread());

    // 削除されたユーザーを探す
    for( auto& u : m_UserIds )
    {
        if( !u.m_Uid )
        {
            continue;
        }
        if( std::find(newUids.begin(), newUids.end(), u.m_Uid) != newUids.end() )
        {
            continue;
        }
        NN_DETAIL_ND_TRACE("[nd] UserIdManager::UpdateUnsafe : uid<0x%016llx-0x%016llx> is deleted.\n", u.m_Uid._data[0], u.m_Uid._data[1]);
        u.Reset();
    }

    // 追加されたユーザーを探す
    for( auto& newU : newUids )
    {
        if( std::find_if(m_UserIds.begin(), m_UserIds.end(), [&newU](const auto& u) { return u.m_Uid == newU; }) != m_UserIds.end() )
        {
            continue;
        }
        NN_DETAIL_ND_TRACE("[nd] UserIdManager::UpdateUnsafe : uid<0x%016llx-0x%016llx> is added.\n", newU._data[0], newU._data[1]);
        auto u = std::find_if(m_UserIds.begin(), m_UserIds.end(), [](const auto& u) { return !u.m_Uid; });
        NN_ABORT_UNLESS(!(u == m_UserIds.end()), "[nd] UserIdManager::UpdateUnsafe : m_UserIds is full.");
        u->Initialize(newU);
    }
}

/*static*/ void UserIdManager::UpdateNetworkUserIdThread(void* arg) NN_NOEXCEPT
{
    auto m = static_cast<UserIdManager*>(arg);
    while( NN_STATIC_CONDITION(true) )
    {
        m->WaitUpdateNetworkUserIdRequest();
        account::Uid targetUid;
        account::NetworkServiceAccountId targetNsaId;
        {
            NN_UTIL_LOCK_GUARD(m->m_Mutex);
            auto id = std::find_if(m->m_UserIds.begin(), m->m_UserIds.end(), [](const auto& id) { return id.m_Uid && id.m_IsNetworkUserIdUpdateRequested; });
            if( id == m->m_UserIds.end() )
            {
                continue;
            }
            NN_SDK_ASSERT(id->m_NsaId);
            id->m_IsNetworkUserIdUpdateRequested = false;
            targetUid = id->m_Uid;
            targetNsaId = *id->m_NsaId;
        }
        NN_DETAIL_ND_TRACE("[nd] UserIdManager::UpdateNetworkUserIdThread : Start updating uid<0x%016llx-0x%016llx>'s NetworkUserId...\n",
            targetUid._data[0], targetUid._data[1]);

        // TODO: サーバーから取得。とりあえず適当に寝てからダミーを設定。
        os::SleepThread(TimeSpan::FromSeconds(10));
        auto newNetworkUserId = GenerateDummyNetworkUserId();

        {
            NN_UTIL_LOCK_GUARD(m->m_Mutex);
            auto id = std::find_if(m->m_UserIds.begin(), m->m_UserIds.end(), [&targetUid](const auto& id) { return id.m_Uid == targetUid; });
            if( id == m->m_UserIds.end() )
            {
                // NetworkUserId 取得中に対象のユーザーが削除された。
                NN_DETAIL_ND_TRACE("[nd] UserIdManager::UpdateNetworkUserIdThread : uid<0x%016llx-0x%016llx> is not found. Must be already deleted.\n",
                    targetUid._data[0], targetUid._data[1]);
                continue;
            }
            if( !id->m_NsaId || targetNsaId != *id->m_NsaId )
            {
                // NetworkUserId 取得中に NetworkServiceAccountId が剥がされた or 新しいものに変更された。
                NN_DETAIL_ND_TRACE("[nd] UserIdManager::UpdateNetworkUserIdThread : uid<0x%016llx-0x%016llx>'s NetworkServiceAccountId has been invalidated or changed.\n",
                    targetUid._data[0], targetUid._data[1]);
                continue;
            }
            id->m_NetworkUserId = newNetworkUserId;
            NN_DETAIL_ND_TRACE("[nd] UserIdManager::UpdateNetworkUserIdThread : uid<0x%016llx-0x%016llx>'s NetworkUserId is updated to 0x%016llx\n",
                id->m_Uid._data[0], id->m_Uid._data[1], id->m_NetworkUserId->_id);
            NN_ABORT_UNLESS_RESULT_SUCCESS(StoreUserId(*id));
        }
        m->m_NetworkUserIdUpdatedEvent.Signal();
    }
}

void UserIdManager::WaitUpdateNetworkUserIdRequest() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    while( std::find_if(m_UserIds.begin(), m_UserIds.end(), [](const auto& id) { return id.m_IsNetworkUserIdUpdateRequested; }) == m_UserIds.end() )
    {
        m_ConditionalVariable.Wait(m_Mutex);
    }
}

UserIdManager::UserId* UserIdManager::FindOrInitUserIdUnsafe(const account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Mutex.IsLockedByCurrentThread());

    auto id = std::find_if(m_UserIds.begin(), m_UserIds.end(), [&uid](const auto& u) { return u.m_Uid == uid; });
    if( id != m_UserIds.end() )
    {
        return &(*id);
    }
    // ユーザー追加時に初期化しているので基本的には到達しないはず。EventHandler に処理が回る前に API 呼び出しされた場合に到達する。。
    // ユーザーの削除と追加が連続して行われていた場合には追加されたユーザーの分の領域が空いていない可能性があるので、空き領域を探して初期化するのではなく Update を呼び出す。
    int accountCount;
    account::Uid uids[account::UserCountMax];
    NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&accountCount, uids, NN_ARRAY_SIZE(uids)));
    UpdateUnsafe({ uids, accountCount });
    id = std::find_if(m_UserIds.begin(), m_UserIds.end(), [&uid](const auto& u) { return u.m_Uid == uid; });
    NN_ABORT_UNLESS(id != m_UserIds.end());
    return &(*id);
}

}}} // ~nn::nd::service
