﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/account.h>
#include <nn/nd/detail/nd_FirmwareDebugSettings.h>
#include <nn/nd/service/nd_ServiceResource.h>
#include <nn/nd/service/nd_EventHandler.h>

namespace nn { namespace nd { namespace service {

ServiceResource::ServiceResource(nn::MemoryResource& allocator, void* networkThreadStack, size_t networkThreadStackSize) NN_NOEXCEPT
    : m_Allocator(allocator)
    , m_UserIdManager(networkThreadStack, networkThreadStackSize)
    , m_ReceiveDataManager(m_UserIdManager)
    , m_SendDataManager(m_UserIdManager)
    , m_ScanManager(m_Allocator)
{
    InitializeImpl();
}

void ServiceResource::InitializeImpl() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    detail::FirmwareDebugSettings::Initialize();
#endif

    int accountCount;
    account::Uid uids[account::UserCountMax];
    NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&accountCount, uids, NN_ARRAY_SIZE(uids)));

    m_UserIdManager.Initialize({ uids, accountCount });
    m_ReceiveDataManager.Initialize({ uids, accountCount });
    m_SendDataManager.Initialize({ uids, accountCount });

    m_pServiceForSystem = sf::CreateSharedObjectEmplaced<IServiceForSystem, ServiceForSystem>(&m_Allocator,
        m_Allocator, m_ReceiveDataManager, m_SendDataManager, m_ScanManager);
    NN_ABORT_UNLESS(m_pServiceForSystem, "[nd] Failed to create ServiceObject for System (nullptr returned)\n");

    m_pServiceForApplication = sf::CreateSharedObjectEmplaced<IServiceForApplication, ServiceForApplication>(&m_Allocator,
        m_Allocator, m_ReceiveDataManager, m_SendDataManager);
    NN_ABORT_UNLESS(m_pServiceForApplication, "[nd] Failed to create ServiceObject for Application (nullptr returned)\n");

    EventHandler::Start(m_UserIdManager, m_ReceiveDataManager, m_SendDataManager, m_ScanManager);
}

}}} // ~nn::nd::service
