﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/account.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/bgtc/bgtc_Task.h>
#include <nn/fs/fs_Result.h>
#include <nn/nd/detail/nd_DataConverter.h>
#include <nn/nd/detail/nd_FirmwareDebugSettings.h>
#include <nn/nd/detail/nd_Log.h>
#include <nn/nd/detail/nd_Result.h>
#include <nn/nd/service/nd_SendDataManager.h>
#include <nn/ndd.h>
#include <nn/os/os_Random.h>
#include <nn/pdm/pdm_QueryApiForSystem.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/system/settings_Language.h>
#include <nn/time.h>
#include <nn/time/time_StandardSteadyClock.h>
#include <nn/util/util_LockGuard.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_UuidApi.h>

namespace nn { namespace nd { namespace service {

namespace
{

ndd::DataId GenerateDataId() NN_NOEXCEPT
{
    ndd::DataId dataId;
    os::GenerateRandomBytes(&dataId.raw, sizeof(dataId.raw));
    return dataId;
}

} // ~nn::nd::service::<anonymous>


// SendDataManager::UserData

void SendDataManager::UserData::Reset() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Uid);

    *this = UserData{};
}

void SendDataManager::UserData::Initialize(const account::Uid & _uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(_uid);
    NN_SDK_REQUIRES(!m_Uid);

    m_Uid = _uid;

    auto loadResult = Load();
    if( fs::ResultPathNotFound::Includes(loadResult) )
    {
        // ndを有効にするFWアップデート後 or 本体初期化後 or ユーザー新規作成後。
        NN_DETAIL_ND_TRACE("[nd] SendDataManager::UserData::Initialize : SystemSaveData for uid<%016llx-%016llx> has not been created yet.\n",
            m_Uid._data[0], m_Uid._data[1]);


        RefreshAccountProfile();
        RefreshRecentlyPlayedApplication();

        SetDataId(GenerateDataId());
        m_SystemDataSize = 0u;
        std::memset(m_SystemData, 0, sizeof(m_SystemData));
        return;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(loadResult);
    RefreshAccountProfile();
    RefreshRecentlyPlayedApplication();
}

bool SendDataManager::UserData::IsInitialized() const NN_NOEXCEPT
{
    return static_cast<bool>(m_Uid);
}

void SendDataManager::UserData::SetDataId(const ndd::DataId& _dataId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Uid);

    m_DataId = _dataId;
}

bool SendDataManager::UserData::RefreshAccountProfile() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Uid);

    auto currentNickname = m_Nickname;
    account::Nickname newNickname;
    auto result = account::GetNickname(&newNickname, m_Uid);
    if( result.IsSuccess() )
    {
        m_Nickname = newNickname;
        return std::strncmp(m_Nickname.name, currentNickname.name, sizeof(m_Nickname.name)) != 0;
    }
    else
    {
        NN_DETAIL_ND_WARN("[nd] SendDataManager::UserData::UpdateAccountProfile : Failed to get nickname of uid<%016llx-%016llx> : %03d-%04d (0x%08x)\n",
            m_Uid._data[0], m_Uid._data[1], result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
        return false;
    }
}

bool SendDataManager::UserData::RefreshRecentlyPlayedApplication() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Uid);

    auto currentCount = m_RecentlyPlayedApplicationCount;
    ncm::ApplicationId current[RecentlyPlayedApplicationCountMax];
    std::memcpy(current, m_RecentlyPlayedApplication, sizeof(ncm::ApplicationId) * m_RecentlyPlayedApplicationCount);
    m_RecentlyPlayedApplicationCount = pdm::QueryRecentlyPlayedApplication(m_RecentlyPlayedApplication, NN_ARRAY_SIZE(m_RecentlyPlayedApplication), m_Uid);
    return !(m_RecentlyPlayedApplicationCount == currentCount && std::equal(current, current + currentCount, m_RecentlyPlayedApplication));
}

void SendDataManager::UserData::SetSystemData(const void* data, size_t dataSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Uid);
    NN_SDK_REQUIRES_NOT_NULL(data);
    NN_SDK_REQUIRES_MINMAX(dataSize, 1u, SystemDataSizeMax);

    m_SystemDataSize = dataSize;
    std::memcpy(m_SystemData, data, m_SystemDataSize);
}

void SendDataManager::UserData::ClearSystemData() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Uid);

    m_SystemDataSize = 0u;
    std::memset(m_SystemData, 0, sizeof(m_SystemData));
}

Result SendDataManager::UserData::Load() NN_NOEXCEPT
{
    // TODO: アカウントセーブデータからの読み込み。
    return fs::ResultPathNotFound();
}

Result SendDataManager::UserData::Store() NN_NOEXCEPT
{
    // TODO: アカウントセーブデータへの保存。
    NN_RESULT_SUCCESS;
}

// SendDataManager

SendDataManager::SendDataManager(UserIdManager& userIdManager) NN_NOEXCEPT
    : m_UserIdManager(userIdManager)
{
    settings::GetLanguageCode(&m_SystemLanguageCode);
    m_SendDataIdUpdateTask.Initialize();
}

void SendDataManager::Initialize(const util::Span<const account::Uid>& uids) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(static_cast<int>(uids.size()), account::UserCountMax);

    int count = 0;
    for( auto& u : uids )
    {
        m_UserDatas[count++].Initialize(u);
    }

    const auto IdRefreshIntervalSeconds = static_cast<bgtc::Interval>(detail::FirmwareDebugSettings::GetSendDataIdRefreshInterval().GetSeconds());

    auto loadSystemSaveResult = LoadSettings();
    if( fs::ResultPathNotFound::Includes(loadSystemSaveResult) )
    {
        // ndを有効にするFWアップデート後 or 本体初期化直後。
        // セーブデータ用のファイルを作成した上で RefreshSendDataId を呼んで送信データIDを更新、セーブデータを保存。
        NN_DETAIL_ND_TRACE("[nd] SendDataManager::Initialize : SystemSaveData has not been created yet.\n");
        NN_SDK_ASSERT(!m_Sender);
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_SystemSaveData.Create(SettingsFilePath, sizeof(SettingsDataLayout)));
        m_SendDataIdUpdateTask.SchedulePeriodic(IdRefreshIntervalSeconds, IdRefreshIntervalSeconds);
        RefreshSendDataId();
        return;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(loadSystemSaveResult);
    if( m_Sender )
    {
        m_SenderNetworkUserIdCache = m_UserIdManager.GetNetworkUserId(m_Sender);
    }

    time::SteadyClockTimePoint currentTime;
    NN_ABORT_UNLESS_RESULT_SUCCESS(time::StandardSteadyClock::GetCurrentTimePoint(&currentTime));
    int64_t elapsed;
    auto getSpanResult = time::GetSpanBetween(&elapsed, m_SendDataIdRefreshTimePoint, currentTime);
    if( getSpanResult.IsFailure() )
    {
        NN_DETAIL_ND_WARN("[nd] SendDataManager::Initialize : time::GetSpanBetween failed : %03d-%04d (0x%08x)\n",
            getSpanResult.GetModule(), getSpanResult.GetDescription(), getSpanResult.GetInnerValueForDebug());
        // 前回の送信データID更新時刻からの経過時間が計算できなかった場合は次回以降の更新のスケジュールだけを行う。
        m_SendDataIdUpdateTask.SchedulePeriodic(IdRefreshIntervalSeconds, IdRefreshIntervalSeconds);
        if( m_Sender )
        {
            NN_UTIL_LOCK_GUARD(m_Mutex);
            UpdateSendDataUnsafe(*FindOrInitUserDataUnsafe(m_Sender));
        }
        return;
    }
    else if( elapsed < IdRefreshIntervalSeconds )
    {
        NN_DETAIL_ND_TRACE("[nd] SendDataManager::Initialize : Next SendDataId refresh will be in %lld minutes (%lld seconds).\n",
            (IdRefreshIntervalSeconds - elapsed) / 60, IdRefreshIntervalSeconds - elapsed);
        m_SendDataIdUpdateTask.SchedulePeriodic(static_cast<bgtc::Interval>(IdRefreshIntervalSeconds - elapsed), IdRefreshIntervalSeconds);
        if( m_Sender )
        {
            NN_UTIL_LOCK_GUARD(m_Mutex);
            UpdateSendDataUnsafe(*FindOrInitUserDataUnsafe(m_Sender));
        }
        return;
    }
    NN_DETAIL_ND_TRACE("[nd] SendDataManager::Initialize : %lld minutes (%lld seconds) has passed since the last SendDataId refresh and it exceeds the refresh interval (%d seconds). Refresh now.\n",
        elapsed / 60, elapsed, IdRefreshIntervalSeconds);
    m_SendDataIdUpdateTask.SchedulePeriodic(IdRefreshIntervalSeconds, IdRefreshIntervalSeconds);
    RefreshSendDataId();
}

void SendDataManager::Update(const util::Span<const account::Uid>& newUids) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    UpdateUnsafe(newUids);
}

void SendDataManager::RefreshAccountProfile() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    for( auto& u : m_UserDatas )
    {
        if( u.IsInitialized() && u.RefreshAccountProfile() )
        {
            NN_DETAIL_ND_TRACE("[nd] SendDataManager::RefreshAccountProfile : Profile of uid<%016llx-%016llx> is updated.\n", u.m_Uid._data[0], u.m_Uid._data[1]);
            if( u.m_Uid == m_Sender )
            {
                UpdateSendDataUnsafe(u);
            }
        }
    }
}

void SendDataManager::CheckNetworkUserIdUpdate() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    for( auto& u : m_UserDatas )
    {
        if( u.IsInitialized() && u.m_Uid == m_Sender )
        {
            auto newNetworkUserId = m_UserIdManager.GetNetworkUserId(u.m_Uid);
            if( newNetworkUserId != m_SenderNetworkUserIdCache )
            {
                NN_DETAIL_ND_TRACE("[nd] SendDataManager::RefreshAccountProfile : Sender(uid<%016llx-%016llx>)'s NetworkUserId is updated.\n", u.m_Uid._data[0], u.m_Uid._data[1]);
                m_SenderNetworkUserIdCache = newNetworkUserId;
                UpdateSendDataUnsafe(u);
            }
        }
    }
}

void SendDataManager::RefreshRecentlyPlayedApplication() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    for( auto& u : m_UserDatas )
    {
        if( u.IsInitialized() && u.RefreshRecentlyPlayedApplication() )
        {
            NN_DETAIL_ND_TRACE("[nd] SendDataManager::RefreshRecentlyPlayedApplication : Recently played application of uid<%016llx-%016llx> is updated.\n",
                u.m_Uid._data[0], u.m_Uid._data[1]);
            if( u.m_Uid == m_Sender )
            {
                UpdateSendDataUnsafe(u);
            }
        }
    }
}

void SendDataManager::RefreshSendDataId() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    auto taskNotifyStartingResult = m_SendDataIdUpdateTask.NotifyStarting();
    if( taskNotifyStartingResult.IsFailure() )
    {
        NN_DETAIL_ND_WARN("[nd] SendDataManager::RefreshSendDataId : NotifyStarting failed : %03d-%04d (0x%08x)\n",
            taskNotifyStartingResult.GetModule(), taskNotifyStartingResult.GetDescription(), taskNotifyStartingResult.GetInnerValueForDebug());
    }
    NN_UTIL_SCOPE_EXIT
    {
        if( taskNotifyStartingResult.IsSuccess() )
        {
            m_SendDataIdUpdateTask.NotifyFinished();
        }
    };

    for( auto& u : m_UserDatas )
    {
        if( u.IsInitialized() )
        {
            u.SetDataId(GenerateDataId());
            if( u.m_Uid == m_Sender )
            {
                UpdateSendDataUnsafe(u);
            }
        }
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(time::StandardSteadyClock::GetCurrentTimePoint(&m_SendDataIdRefreshTimePoint));
    NN_ABORT_UNLESS_RESULT_SUCCESS(StoreSettings());
}

void SendDataManager::SetSender(const account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    m_Sender = uid;
    m_SenderNetworkUserIdCache = m_UserIdManager.GetNetworkUserId(uid);
    UpdateSendDataUnsafe(*FindOrInitUserDataUnsafe(uid));
    NN_ABORT_UNLESS_RESULT_SUCCESS(StoreSettings());
}

bool SendDataManager::GetSender(account::Uid* pOut) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    *pOut = m_Sender;
    return static_cast<bool>(m_Sender);
}

void SendDataManager::SetDataId(const account::Uid& uid, const ndd::DataId& dataId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    auto pUserData = FindOrInitUserDataUnsafe(uid);
    pUserData->SetDataId(dataId);
    if( uid == m_Sender )
    {
        UpdateSendDataUnsafe(*pUserData);
    }
}

void SendDataManager::SetSystemData(const account::Uid& uid, const void * data, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);
    NN_SDK_REQUIRES_NOT_NULL(data);
    NN_SDK_REQUIRES_MINMAX(size, 1u, SystemDataSizeMax);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    auto pUserData = FindOrInitUserDataUnsafe(uid);
    pUserData->SetSystemData(data, size);
    if( uid == m_Sender )
    {
        UpdateSendDataUnsafe(*pUserData);
    }
}

void SendDataManager::ClearSystemData(const account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    auto pUserData = FindOrInitUserDataUnsafe(uid);
    pUserData->ClearSystemData();
    if( uid == m_Sender )
    {
        UpdateSendDataUnsafe(*pUserData);
    }
}

os::SystemEvent& SendDataManager::GetSendDataIdUpdateScheduleEvent() NN_NOEXCEPT
{
    return m_SendDataIdUpdateTask.GetScheduleEvent();
}

void SendDataManager::GetSendDataForDebug(ndd::SendDataDescription *pOut, const account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_SDK_REQUIRES(uid);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    auto pUserData = FindOrInitUserDataUnsafe(uid);
    detail::MakeSendDataDescription(
        pOut,
        pUserData->m_DataId,
        m_SystemLanguageCode,
        m_UserIdManager.GetLocalUserId(uid),
        m_UserIdManager.GetNetworkUserId(uid),
        pUserData->m_Nickname,
        pUserData->m_RecentlyPlayedApplicationCount,
        pUserData->m_RecentlyPlayedApplication,
        pUserData->m_SystemDataSize,
        pUserData->m_SystemData
    );
}

void SendDataManager::UpdateSendDataUnsafe(const UserData& userData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Sender);
    NN_SDK_REQUIRES(m_Mutex.IsLockedByCurrentThread());

    ndd::SendDataDescription data;
    detail::MakeSendDataDescription(
        &data,
        userData.m_DataId,
        m_SystemLanguageCode,
        m_UserIdManager.GetLocalUserId(userData.m_Uid),
        m_UserIdManager.GetNetworkUserId(userData.m_Uid),
        userData.m_Nickname,
        userData.m_RecentlyPlayedApplicationCount,
        userData.m_RecentlyPlayedApplication,
        userData.m_SystemDataSize,
        userData.m_SystemData
    );
    ndd::AddSendData(data);
}

void SendDataManager::UpdateUnsafe(const util::Span<const account::Uid>& newUids) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Mutex.IsLockedByCurrentThread());

    // 削除されたユーザーを探す
    for( auto& u : m_UserDatas )
    {
        if( !u.IsInitialized() )
        {
            continue;
        }
        if( std::find(newUids.begin(), newUids.end(), u.m_Uid) != newUids.end() )
        {
            continue;
        }
        NN_DETAIL_ND_TRACE("[nd] SendDataManager::UpdateUnsafe : uid<0x%016llx-0x%016llx> is deleted.\n", u.m_Uid._data[0], u.m_Uid._data[1]);
        if( u.m_Uid == m_Sender )
        {
            NN_DETAIL_ND_TRACE("[nd] SendDataManager::UpdateUnsafe : Clear send data as the deleted user is the current sender.\n");
            m_Sender = account::InvalidUid;
            m_SenderNetworkUserIdCache = util::nullopt;
            NN_ABORT_UNLESS_RESULT_SUCCESS(StoreSettings());
            ndd::ClearSendData();
        }
        u.Reset();
    }

    // 追加されたユーザーを探す
    for( auto& newU : newUids )
    {
        if( std::find_if(m_UserDatas.begin(), m_UserDatas.end(), [&newU](const auto& u) { return u.m_Uid == newU; }) != m_UserDatas.end() )
        {
            continue;
        }
        NN_DETAIL_ND_TRACE("[nd] SendDataManager::UpdateUnsafe : uid<0x%016llx-0x%016llx> is added.\n", newU._data[0], newU._data[1]);
        auto u = std::find_if(m_UserDatas.begin(), m_UserDatas.end(), [](const auto& u) { return !u.IsInitialized(); });
        NN_ABORT_UNLESS(!(u == m_UserDatas.end()), "[nd] SendDataManager::UpdateUnsafe : UserDatas is full.");
        u->Initialize(newU);
    }
}

SendDataManager::UserData* SendDataManager::FindOrInitUserDataUnsafe(const account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Mutex.IsLockedByCurrentThread());

    auto u = std::find_if(m_UserDatas.begin(), m_UserDatas.end(), [&uid](const UserData& u) { return u.m_Uid == uid; });
    if( u != m_UserDatas.end() )
    {
        return &(*u);
    }
    // ユーザー追加時に初期化しているので基本的には到達しないはず。EventHandler に処理が回る前に API 呼び出しされた場合に到達する。
    // ユーザーの削除と追加が連続して行われていた場合には追加されたユーザーの分の領域が空いていない可能性があるので、空き領域を探して初期化するのではなく Update を呼び出す。
    int accountCount;
    account::Uid uids[account::UserCountMax];
    NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&accountCount, uids, NN_ARRAY_SIZE(uids)));
    UpdateUnsafe({ uids, accountCount });
    u = std::find_if(m_UserDatas.begin(), m_UserDatas.end(), [&uid](const UserData& u) { return u.m_Uid == uid; });
    NN_ABORT_UNLESS(u != m_UserDatas.end());
    return &(*u);
}

Result SendDataManager::StoreSettings() NN_NOEXCEPT
{
    SettingsDataLayout d
    {
        m_Sender,
        m_SendDataIdRefreshTimePoint,
    };
    NN_RESULT_DO(m_SystemSaveData.Write(SettingsFilePath, &d, sizeof(d)));
    NN_RESULT_DO(m_SystemSaveData.Commit());
    NN_RESULT_SUCCESS;
}

Result SendDataManager::LoadSettings() NN_NOEXCEPT
{
    size_t size;
    SettingsDataLayout d;
    NN_RESULT_DO(m_SystemSaveData.Read(&size, &d, sizeof(d), SettingsFilePath));
    m_Sender = d.sender;
    m_SendDataIdRefreshTimePoint = d.sendDataIdRefreshTimePont;
    NN_RESULT_SUCCESS;
}

}}} // ~nn::nd::service
