﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/migration/migration_UserMigrationClient.h>

#include <nn/migration/migration_Result.h>
#include <nn/migration/user/migration_Interface.sfdl.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_IServiceObject.h>

namespace nn { namespace migration {

void swap(UserMigrationClient& o0, UserMigrationClient& o1) NN_NOEXCEPT
{
    std::swap(o0.m_Ptr, o1.m_Ptr);
}

UserMigrationClient::UserMigrationClient() NN_NOEXCEPT
    : m_Ptr(nullptr)
{
}
UserMigrationClient::UserMigrationClient(user::IClient* ptr) NN_NOEXCEPT
    : m_Ptr(ptr)
{
}
UserMigrationClient::UserMigrationClient(UserMigrationClient&& rhs) NN_NOEXCEPT
    : m_Ptr(rhs.m_Ptr)
{
    rhs.m_Ptr = nullptr;
}
UserMigrationClient::~UserMigrationClient() NN_NOEXCEPT
{
    if (m_Ptr)
    {
        sf::ReleaseSharedObject(m_Ptr);
    }
}
UserMigrationClient& UserMigrationClient::operator=(UserMigrationClient&& rhs) NN_NOEXCEPT
{
    UserMigrationClient tmp(std::move(rhs));
    swap(*this, tmp);
    return *this;
}

account::SessionId UserMigrationClient::CreateLoginSession() const NN_NOEXCEPT
{
    account::SessionId sessionId;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->CreateLoginSession(&sessionId));
    return sessionId;
}
account::NetworkServiceAccountId UserMigrationClient::GetNetworkServiceAccountId() const NN_NOEXCEPT
{
    account::NetworkServiceAccountId id;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->GetNetworkServiceAccountId(&id));
    return id;
}
void UserMigrationClient::GetUserNickname(account::Nickname* pOut) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->GetUserNickname(pOut));
}
size_t UserMigrationClient::GetUserProfileImage(void* buffer, size_t bufferSize) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_GREATER(bufferSize, 0u);

    uint32_t actualSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->GetUserProfileImage(&actualSize, sf::OutBuffer(reinterpret_cast<char*>(buffer), bufferSize)));
    return static_cast<size_t>(actualSize);
}

AsyncContext UserMigrationClient::PrepareAsync() NN_NOEXCEPT
{
    sf::SharedPointer<detail::IAsyncContext> p;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->PrepareAsync(&p));
    return AsyncContext(p.Detach());
}

bool UserMigrationClient::IsConnectionRequired() const NN_NOEXCEPT
{
    bool requirement;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->GetConnectionRequirement(&requirement));
    return requirement;
}

AsyncContext UserMigrationClient::ScanServersAsync() NN_NOEXCEPT
{
    sf::SharedPointer<detail::IAsyncContext> p;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->ScanServersAsync(&p));
    return AsyncContext(p.Detach());
}
size_t UserMigrationClient::ListServers(UserMigrationServerInfo* pOutArray, size_t arraySize) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutArray);
    NN_SDK_REQUIRES_GREATER(arraySize, 0u);

    uint32_t actualCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->ListServers(&actualCount, sf::OutArray<UserMigrationServerInfo>(pOutArray, arraySize)));
    return static_cast<size_t>(actualCount);
}
AsyncContext UserMigrationClient::ConnectAsync(const util::Uuid& serverId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(serverId, util::InvalidUuid);

    sf::SharedPointer<detail::IAsyncContext> p;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->ConnectByServerIdAsync(&p, serverId));
    return AsyncContext(p.Detach());
}

account::Uid UserMigrationClient::GetImmigrantUid() const NN_NOEXCEPT
{
    account::Uid uid;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->GetImmigrantUid(&uid));
    return uid;
}
int64_t UserMigrationClient::GetStorageShortfall() const NN_NOEXCEPT
{
    int64_t shortfall;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->GetStorageShortfall(&shortfall));
    return shortfall;
}
TransferInfo UserMigrationClient::GetTotalTransferInfo() const NN_NOEXCEPT
{
    TransferInfo info;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->GetTotalTransferInfo(&info));
    return info;
}
TransferInfo UserMigrationClient::GetCurrentTransferInfo() const NN_NOEXCEPT
{
    TransferInfo info;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->GetCurrentTransferInfo(&info));
    return info;
}
size_t UserMigrationClient::GetCurrentRelatedApplications(ncm::ApplicationId* pOutArray, size_t arraySize) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutArray);
    NN_SDK_REQUIRES_GREATER(arraySize, 0u);

    uint32_t actualCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->GetCurrentRelatedApplications(&actualCount, sf::OutArray<ncm::ApplicationId>(pOutArray, arraySize)));
    return static_cast<size_t>(actualCount);
}
AsyncContext UserMigrationClient::TransferNextAsync() NN_NOEXCEPT
{
    sf::SharedPointer<detail::IAsyncContext> p;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->TransferNextAsync(&p));
    return AsyncContext(p.Detach());
}

AsyncContext UserMigrationClient::SuspendAsync() NN_NOEXCEPT
{
    sf::SharedPointer<detail::IAsyncContext> p;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->SuspendAsync(&p));
    return AsyncContext(p.Detach());
}

Result UserMigrationClient::TransferAll() NN_NOEXCEPT
{
    auto total = GetTotalTransferInfo();
    while (GetCurrentTransferInfo().count < total.count)
    {
        auto async = TransferNextAsync();
        os::SystemEvent e;
        async.GetSystemEvent(&e);
        e.Wait();

        NN_SDK_ASSERT(async.HasDone());
        NN_RESULT_DO(async.GetResult());
    }
    NN_RESULT_SUCCESS;
}

AsyncContext UserMigrationClient::CompleteAsync() NN_NOEXCEPT
{
    sf::SharedPointer<detail::IAsyncContext> p;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->CompleteAsync(&p));
    return AsyncContext(p.Detach());
}

Result UserMigrationClient::Abort() NN_NOEXCEPT
{
    return m_Ptr->Abort();
}

AsyncContext UserMigrationClient::DebugSynchronizeStateInFinalization() NN_NOEXCEPT
{
    sf::SharedPointer<detail::IAsyncContext> p;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->DebugSynchronizeStateInFinalizationAsync(&p));
    return AsyncContext(p.Detach());
}

}} // ~namespace nn::migration
