﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/migration/detail/migration_Log.h>
#include <nn/migration/idc/migration_SharedBufferConnection.h>
#include <nn/migration/idc/migration_SharedBufferConnectionManager.h>
#include <nn/os.h>

namespace nn { namespace migration { namespace idc {

SharedBufferConnectionManager::SharedBuffer::SharedBuffer() NN_NOEXCEPT
    : m_Buffer(nullptr)
    , m_BufferSize(0)
    , m_UsedBufferSize(0)
    , m_BufferMutex(false)
    , m_IsPushOpen(false)
    , m_IsPopOpen(false)
    , m_pDeallocator(nullptr)
{
}

SharedBufferConnectionManager::SharedBuffer::~SharedBuffer() NN_NOEXCEPT
{
    if( m_Buffer != nullptr )
    {
        Reset();
    }
}

void SharedBufferConnectionManager::SharedBuffer::Initialize(void* buffer, size_t bufferSize, Deallocator* pDeallocator) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_GREATER(bufferSize, 0u);
    NN_SDK_REQUIRES_NOT_NULL(pDeallocator);
    m_Buffer = buffer;
    m_BufferSize = bufferSize;
    m_UsedBufferSize = 0;
    m_IsPushOpen = false;
    m_IsPopOpen = false;
    m_pDeallocator = pDeallocator;
}

void SharedBufferConnectionManager::SharedBuffer::Reset() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_Buffer);
    NN_ABORT_UNLESS(!m_IsPushOpen);
    NN_ABORT_UNLESS(!m_IsPopOpen);
    m_pDeallocator->Deallocate(m_Buffer);
    m_Buffer = nullptr;
}

bool SharedBufferConnectionManager::SharedBuffer::IsFullUnsafe() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Buffer);
    NN_SDK_REQUIRES(m_BufferMutex.IsLockedByCurrentThread());
    return (m_UsedBufferSize == m_BufferSize);
}

bool SharedBufferConnectionManager::SharedBuffer::IsEmptyUnsafe() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Buffer);
    NN_SDK_REQUIRES(m_BufferMutex.IsLockedByCurrentThread());
    return (m_UsedBufferSize == 0);
}

bool SharedBufferConnectionManager::SharedBuffer::TimedWaitWhileImpl(bool (SharedBuffer::*wait)() const, os::ConditionVariable& conditionVariable, int timeoutMilliSeconds) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Buffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(timeoutMilliSeconds, 0);
    std::lock_guard<os::Mutex> lock(m_BufferMutex);
    auto limitTick = nn::os::GetSystemTick() + nn::os::ConvertToTick(nn::TimeSpan::FromMilliSeconds(timeoutMilliSeconds));
    while( (this->*wait)() )
    {
        // spurious wakeup 対策。
        auto timespan = (limitTick - nn::os::GetSystemTick()).ToTimeSpan();
        if( timespan < 0 )
        {
            return true;
        }
        conditionVariable.TimedWait(*(m_BufferMutex.GetBase()), timespan);
    }
    return false;
}

bool SharedBufferConnectionManager::SharedBuffer::WaitPushable(int timeoutMilliSeconds) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Buffer);
    NN_SDK_REQUIRES(m_IsPushOpen);
    NN_SDK_REQUIRES_GREATER_EQUAL(timeoutMilliSeconds, 0);
    return !TimedWaitWhileImpl(&SharedBuffer::IsFullUnsafe, m_PopCondition, timeoutMilliSeconds);
}

bool SharedBufferConnectionManager::SharedBuffer::WaitPoppable(int timeoutMilliSeconds) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Buffer);
    NN_SDK_REQUIRES(m_IsPopOpen);
    NN_SDK_REQUIRES_GREATER_EQUAL(timeoutMilliSeconds, 0);
    return !TimedWaitWhileImpl(&SharedBuffer::IsEmptyUnsafe, m_PushCondition, timeoutMilliSeconds);
}

size_t SharedBufferConnectionManager::SharedBuffer::Push(const void* stream, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Buffer);
    std::lock_guard<os::Mutex> lock(m_BufferMutex);
    NN_SDK_REQUIRES(!IsFullUnsafe());
    NN_SDK_REQUIRES(m_IsPushOpen);
    auto pushSize = std::min(size, m_BufferSize - m_UsedBufferSize);
    std::memcpy(reinterpret_cast<Bit8*>(m_Buffer) + m_UsedBufferSize, stream, pushSize);
    m_UsedBufferSize += pushSize;
    m_PushCondition.Signal();
    return pushSize;
}

size_t SharedBufferConnectionManager::SharedBuffer::Pop(void* pOutStream, size_t outStreamSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Buffer);
    std::lock_guard<os::Mutex> lock(m_BufferMutex);
    NN_SDK_REQUIRES(!IsEmptyUnsafe());
    NN_SDK_REQUIRES(m_IsPopOpen);
    auto popSize = std::min(outStreamSize, m_UsedBufferSize);
    std::memcpy(pOutStream, m_Buffer, popSize);
    std::memmove(m_Buffer, reinterpret_cast<Bit8*>(m_Buffer) + popSize, m_BufferSize - popSize);
    m_UsedBufferSize -= popSize;
    m_PopCondition.Signal();
    return popSize;
}

void SharedBufferConnectionManager::SharedBuffer::BeginPush() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Buffer);
    NN_SDK_REQUIRES(!m_IsPushOpen);
    m_IsPushOpen = true;
}

void SharedBufferConnectionManager::SharedBuffer::BeginPop() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Buffer);
    NN_SDK_REQUIRES(!m_IsPopOpen);
    m_IsPopOpen = true;
}

void SharedBufferConnectionManager::SharedBuffer::EndPush() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Buffer);
    NN_SDK_REQUIRES(m_IsPushOpen);
    m_IsPushOpen = false;
    if( !m_IsPopOpen )
    {
        Reset();
    }
}

void SharedBufferConnectionManager::SharedBuffer::EndPop() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Buffer);
    NN_SDK_REQUIRES(m_IsPopOpen);
    m_IsPopOpen = false;
    if( !m_IsPushOpen )
    {
        Reset();
    }
}

bool SharedBufferConnectionManager::SharedBuffer::IsPushOpen() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Buffer);
    return m_IsPushOpen;
}

bool SharedBufferConnectionManager::SharedBuffer::IsPopOpen() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Buffer);
    return m_IsPopOpen;
}

SharedBufferConnectionManager::Connection::Connection() NN_NOEXCEPT
    : serverId(-1)
    , clientId(-1)
{
}

void SharedBufferConnectionManager::Connection::Reset() NN_NOEXCEPT
{
    serverId = clientId = -1;
    serverSendBuffer.Reset();
    serverReceiveBuffer.Reset();
}

void SharedBufferConnectionManager::Connection::OpenServerEnd(int id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(id, 0);
    serverId = id;
    serverSendBuffer.BeginPush();
    serverReceiveBuffer.BeginPop();
}

void SharedBufferConnectionManager::Connection::OpenClientEnd(int id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(id, 0);
    clientId = id;
    serverSendBuffer.BeginPop();
    serverReceiveBuffer.BeginPush();
}

void SharedBufferConnectionManager::Connection::CloseServerEnd() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(serverId, 0);
    serverSendBuffer.EndPush();
    serverReceiveBuffer.EndPop();
    serverId = -1;
}

void SharedBufferConnectionManager::Connection::CloseClientEnd() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(clientId, 0);
    serverSendBuffer.EndPop();
    serverReceiveBuffer.EndPush();
    clientId = -1;
}

//////////////////////////////////////////////////

SharedBufferConnectionManager::SharedBufferConnectionManager(void* buffer, size_t bufferSize) NN_NOEXCEPT
    : m_Mutex(false)
    , m_Deallocator(&m_Allocator)
{
    m_Allocator.Initialize(buffer, bufferSize);
}

SharedBufferConnectionManager::~SharedBufferConnectionManager() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    m_Allocator.Finalize();
}

bool SharedBufferConnectionManager::CreateConnection(SharedBufferConnection* pOutServer, SharedBufferConnection* pOutClient, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutServer);
    NN_SDK_REQUIRES_NOT_NULL(pOutClient);
    NN_SDK_REQUIRES_GREATER(bufferSize, 0u);
    NN_SDK_REQUIRES_EQUAL(pOutServer->m_Id, -1);
    NN_SDK_REQUIRES_EQUAL(pOutClient->m_Id, -1);

    std::lock_guard<os::Mutex> lock(m_Mutex);

    auto pConnection = AllocateConnectionUnsafe(bufferSize);
    pConnection->OpenServerEnd(m_IdGenerator.Generate());
    pConnection->OpenClientEnd(m_IdGenerator.Generate());

    *pOutServer = SharedBufferConnection(this, pConnection->serverId, &pConnection->serverSendBuffer, &pConnection->serverReceiveBuffer);
    *pOutClient = SharedBufferConnection(this, pConnection->clientId, &pConnection->serverReceiveBuffer, &pConnection->serverSendBuffer);

    NN_DETAIL_MIGRATION_TRACE("[SharedBufferConnectionManager::CreateConnection] created server(%d), client(%d).\n", pConnection->serverId, pConnection->clientId);

    return true;
}

bool SharedBufferConnectionManager::CreateServer(SharedBufferConnection* pOutServer, int* pOutServerId, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutServer);
    NN_SDK_REQUIRES_GREATER(bufferSize, 0u);
    NN_SDK_REQUIRES_EQUAL(pOutServer->m_Id, -1);

    std::lock_guard<os::Mutex> lock(m_Mutex);

    auto pConnection = AllocateConnectionUnsafe(bufferSize);
    pConnection->OpenServerEnd(m_IdGenerator.Generate());

    *pOutServerId = pConnection->serverId;
    *pOutServer = SharedBufferConnection(this, pConnection->serverId, &pConnection->serverSendBuffer, &pConnection->serverReceiveBuffer);

    NN_DETAIL_MIGRATION_TRACE("[SharedBufferConnectionManager::CreateServer] created server(%d)\n", pConnection->serverId);

    return true;
}

bool SharedBufferConnectionManager::WaitConnection(int serverId, int timeoutSeconds) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(serverId, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(timeoutSeconds, 0);

    std::lock_guard<os::Mutex> lock(m_Mutex);
    auto connection = FindConnectionUnsafe(serverId);
    if( !(connection.first == true && connection.second != nullptr) )
    {
        NN_DETAIL_MIGRATION_TRACE("[SharedBufferConnectionManager::WaitConnection] server(%d) is not found\n", serverId);
        return false;
    }
    auto pConnection = connection.second;
    NN_DETAIL_MIGRATION_TRACE("[SharedBufferConnectionManager::WaitConnection] wait for connection to server(%d)...\n", serverId);
    auto limitTick = nn::os::GetSystemTick() + nn::os::ConvertToTick(nn::TimeSpan::FromSeconds(timeoutSeconds));
    while( pConnection->clientId == -1 )
    {
        // spurious wakeup 対策。
        auto timespan = (limitTick - nn::os::GetSystemTick()).ToTimeSpan();
        if( timespan < 0 )
        {
            NN_DETAIL_MIGRATION_TRACE("[SharedBufferConnectionManager::WaitConnection] connection to server(%d) is not established in time.\n", serverId);
            return false;
        }
        m_ConditionVariable.TimedWait(*(m_Mutex.GetBase()), timespan);
    }
    // 既に接続済みでもここにくる。
    NN_DETAIL_MIGRATION_TRACE("[SharedBufferConnectionManager::WaitConnection] connection to server(%d) is established.\n", serverId);
    return true;
}

bool SharedBufferConnectionManager::Connect(SharedBufferConnection* pOut, int serverId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_SDK_REQUIRES_GREATER_EQUAL(serverId, 0);

    std::lock_guard<os::Mutex> lock(m_Mutex);
    auto connection = FindConnectionUnsafe(serverId);
    if( !(connection.first == true && connection.second != nullptr) )
    {
        NN_DETAIL_MIGRATION_TRACE("[SharedBufferConnectionManager::Connect] server(%d) is not found\n", serverId);
        return false;
    }
    auto pConnection = connection.second;
    if( pConnection->clientId >= 0 )
    {
        NN_DETAIL_MIGRATION_TRACE("[SharedBufferConnectionManager::Connect] server(%d) is already connected to client(%d).\n", serverId, pConnection->clientId);
        return false;
    }
    pConnection->OpenClientEnd(m_IdGenerator.Generate());
    *pOut = SharedBufferConnection(this, pConnection->clientId, &pConnection->serverReceiveBuffer, &pConnection->serverSendBuffer);
    m_ConditionVariable.Signal();
    NN_DETAIL_MIGRATION_TRACE("[SharedBufferConnectionManager::Connect] client(%d) is created and connected to server(%d).\n", pConnection->clientId, serverId);
    return true;
}

void SharedBufferConnectionManager::Close(SharedBufferConnection* pC) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pC);
    NN_SDK_REQUIRES_GREATER_EQUAL(pC->m_Id, 0);
    std::lock_guard<os::Mutex> lock(m_Mutex);
    auto id = pC->m_Id;
    auto connection = FindConnectionUnsafe(id);
    bool isServer = connection.first;
    auto pConnection = connection.second;
    NN_ABORT_UNLESS_NOT_NULL(pConnection);
    if( isServer )
    {
        pConnection->CloseServerEnd();
    }
    else
    {
        pConnection->CloseClientEnd();
    }
}

std::pair<bool, SharedBufferConnectionManager::Connection*> SharedBufferConnectionManager::FindConnectionUnsafe(int id) NN_NOEXCEPT
{
    for( auto& c : m_ConnectionPool )
    {
        if( c.serverId == id )
        {
            return std::pair<bool, SharedBufferConnectionManager::Connection*>(true, &c);
        }
        if( c.clientId == id )
        {
            return std::pair<bool, SharedBufferConnectionManager::Connection*>(false, &c);
        }
    }
    return std::pair<bool, SharedBufferConnectionManager::Connection*>(false, nullptr);
}

SharedBufferConnectionManager::Connection* SharedBufferConnectionManager::AllocateConnectionUnsafe(size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Mutex.IsLockedByCurrentThread());
    for( auto& c : m_ConnectionPool )
    {
        if( c.serverId == -1 && c.clientId == -1 )
        {
            auto bufferA = m_Allocator.Allocate(bufferSize);
            NN_ABORT_UNLESS_NOT_NULL(bufferA);
            auto bufferB = m_Allocator.Allocate(bufferSize);
            NN_ABORT_UNLESS_NOT_NULL(bufferB);
            c.serverId = -1;
            c.clientId = -1;
            c.serverSendBuffer.Initialize(bufferA, bufferSize, &m_Deallocator);
            c.serverReceiveBuffer.Initialize(bufferB, bufferSize, &m_Deallocator);
            return &c;
        }
    }
    NN_ABORT("[SharedBufferConnectionManager::AllocateConnectionUnsafe] failed to allocate connection. connection pool is full.\n");
}

}}};
