﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_SdkAssert.h>
#include <nn/migration/detail/migration_AbstractStorage.h>
#include <nn/migration/detail/migration_Diagnosis.h>

namespace nn { namespace migration { namespace detail {

struct Writer
{
    const detail::AbstractStorage* pStorage;
    const char* path;
    size_t offset;

    void Reset(const detail::AbstractStorage& storage, const char* p, size_t initialOffset) NN_NOEXCEPT
    {
        pStorage = &storage;
        path = p;
        offset = initialOffset;
    }

    void Flush(void* data, size_t dataSize) NN_NOEXCEPT
    {
        auto lock = pStorage->AcquireWriteLock();
        NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(pStorage->Append(path, offset, data, dataSize));
        NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(pStorage->Commit());
        offset += dataSize;
    }
};

template <typename T>
struct BufferedWriter
{
    T* ptr;
    int64_t capacity;
    int64_t filled;
    Writer writer {};

    void Reset(
        const detail::AbstractStorage& storage, const char* path, size_t initialOffset,
        void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(reinterpret_cast<uintptr_t>(buffer) % std::alignment_of<decltype(*ptr)>::value, 0u);
        ptr = reinterpret_cast<decltype(ptr)>(buffer);
        capacity = bufferSize / sizeof(decltype(*ptr));
        NN_SDK_ASSERT_GREATER(capacity, 0);
        filled = 0;

        writer.Reset(storage, path, initialOffset);
    }

    void Update(const T& data) NN_NOEXCEPT
    {
        if (!(filled < capacity))
        {
            writer.Flush(ptr, static_cast<size_t>(filled) * sizeof(*ptr));
            filled = 0;
        }
        ptr[filled++] = data;
    }

    void Flush() NN_NOEXCEPT
    {
        if (filled > 0)
        {
            writer.Flush(ptr, static_cast<size_t>(filled) * sizeof(*ptr));
            filled = 0;
        }
    }
};

}}} // ~namespace nn::account::detail
