﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/migration/detail/migration_Authenticator.h>

#include <nn/nn_SdkAssert.h>
#include <nn/account/account_TypesForSystemServices.h>
#include <nn/migration/detail/migration_Log.h>
#include <nn/migration/detail/migration_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include "migration_MigrationAuthenticator.h"

namespace nn { namespace migration { namespace detail {
namespace {

const account::SystemProgramIdentification Id = {{0x010000000000003Aull}, 0x00u};

} // ~namespace nn::migration::detail

Result Authenticator::PrepareNsaIdToken(LoginSessionBase& login, const Cancellable* pCancellable) NN_NOEXCEPT
{
    return login.AcquireNetworkServiceAccountIdToken(nullptr, nullptr, 0u, Id, pCancellable);
}

Result Authenticator::Authenticate(
    FsKeySeed* pOutFsKeySeed, FsKeySeedDigest* pOutDigest, OnetimeToken* pOutOtt,
    LoginSessionBase& login, const FsChallenge& fsChallenge,
    void* workBuffer, size_t workBufferSize, const Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutFsKeySeed);
    NN_SDK_ASSERT_NOT_NULL(pOutDigest);
    NN_SDK_ASSERT_NOT_NULL(pOutOtt);
    NN_SDK_ASSERT_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(reinterpret_cast<uintptr_t>(workBuffer) % std::alignment_of<Resource>::value, 0u);
    NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, sizeof(Resource));
    NN_UNUSED(workBufferSize);

    auto* pResource = reinterpret_cast<Resource*>(workBuffer);

    size_t actualSize;
    NN_RESULT_DO(login.AcquireNetworkServiceAccountIdToken(
        &actualSize, pResource->nsaIdToken.data, sizeof(pResource->nsaIdToken.data),
        Id, pCancellable));
    NN_RESULT_THROW_UNLESS(actualSize < sizeof(pResource->nsaIdToken.data), ResultDataUnexpectedSize());
    NN_UNUSED(actualSize);

    NN_RESULT_DO(AuthenticateMigrationSource(
        &pResource->u.source.authData, &pResource->u.source.onetimeToken,
        pResource->nsaIdToken, fsChallenge,
        &pResource->workBuffer, sizeof(pResource->workBuffer), pCancellable));

    *pOutFsKeySeed = pResource->u.source.authData.fsKeySeed;
    *pOutDigest = pResource->u.source.authData.fsKeySeedDigest;
    *pOutOtt = pResource->u.source.onetimeToken;
    NN_RESULT_SUCCESS;
}

Result Authenticator::AuthenticateWithOneTimeCode(
    FsKeySeed* pOutFsKeySeed, FsKeySeedDigest* pOutDigest,
    LoginSessionBase& login, const FsChallenge& fsChallenge, const OnetimeToken& ott,
    void* workBuffer, size_t workBufferSize, const Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutFsKeySeed);
    NN_SDK_ASSERT_NOT_NULL(pOutDigest);
    NN_SDK_ASSERT_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(reinterpret_cast<uintptr_t>(workBuffer) % std::alignment_of<std::max_align_t>::value, 0u);
    NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, sizeof(Resource));
    NN_UNUSED(workBufferSize);

    auto* pResource = reinterpret_cast<Resource*>(workBuffer);

    size_t actualSize;
    NN_RESULT_DO(login.AcquireNetworkServiceAccountIdToken(
        &actualSize, pResource->nsaIdToken.data, sizeof(pResource->nsaIdToken.data),
        Id, pCancellable));
    NN_RESULT_THROW_UNLESS(actualSize < sizeof(pResource->nsaIdToken.data), ResultDataUnexpectedSize());
    NN_UNUSED(actualSize);

    NN_RESULT_DO(AuthenticateMigrationDestination(
        &pResource->u.destination.authData,
        pResource->nsaIdToken, fsChallenge, ott,
        &pResource->workBuffer, sizeof(pResource->workBuffer), pCancellable));

    *pOutFsKeySeed = pResource->u.source.authData.fsKeySeed;
    *pOutDigest = pResource->u.source.authData.fsKeySeedDigest;
    NN_RESULT_SUCCESS;
}

}}} // ~namespace nn::migration::detail
