﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/nn_Result.h>
#include <nn/crypto/crypto_EcdsaP256Sha256Verifier.h>
#include <nn/crypto/crypto_RsaOaepEncryptor.h>
#include <nn/crypto/crypto_RsaPkcs1Sha256Verifier.h>
#include <nn/csl/csl.h>
#include <nn/manu/manu_Api.h>
#include <nn/manu/server/manu_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/spl/spl_Api.h>
#include <nn/util/util_Endian.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <cal_Settings.h>

#define NN_MANU_KEY_VALIDATOR_LOG(...) Log(__VA_ARGS__)

namespace nn { namespace manu {

namespace {

LogFunctionPointer g_pLog = nullptr;

void Log(const char* format, ...) NN_NOEXCEPT
{
    if(g_pLog)
    {
        g_pLog(format);
    }
}

struct DataRange
{
    uint8_t* bytes;
    uint64_t size;
};

const uint32_t sizeRsa = 2048 / 8;
const uint32_t sizePublicExponent = 32 / 8;
const uint32_t sizeEccB233 = (233 + 7) / 8;
const uint32_t sizeEccP256 = 256 / 8;

const uint8_t testdata[] =
{
    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF
};

const uint8_t publicExponentCa[] =
{
    0x00, 0x01, 0x00, 0x01
};

const uint8_t modulusCa2Dev1[] =
{
    0x9e, 0x16, 0x85, 0xd7, 0x7b, 0x2a, 0x4f, 0x9a, 0xb4, 0x1a, 0x51, 0x02, 0x2b, 0x11, 0xba, 0xdf,
    0x40, 0x66, 0x2a, 0xf1, 0x61, 0x81, 0xed, 0xe4, 0x82, 0x72, 0xac, 0x8c, 0xb1, 0x89, 0x48, 0xf9,
    0xfa, 0x25, 0x28, 0x4b, 0x40, 0x6e, 0x8e, 0xe8, 0x10, 0xd7, 0xef, 0xfc, 0x53, 0x22, 0x23, 0xa5,
    0x98, 0x29, 0xe5, 0x31, 0x4c, 0xeb, 0x30, 0x67, 0x30, 0x03, 0xfd, 0x72, 0x36, 0xfb, 0x62, 0xca,
    0xa9, 0x03, 0x64, 0xbe, 0x73, 0xb2, 0x86, 0xf6, 0x8f, 0x95, 0x32, 0xc3, 0xb6, 0x6c, 0x06, 0xae,
    0x56, 0x3c, 0x8e, 0x9c, 0x4f, 0x39, 0x58, 0x19, 0x54, 0x34, 0xa0, 0x1f, 0x39, 0x67, 0x13, 0x9a,
    0xd8, 0xe2, 0x96, 0x45, 0x8e, 0x1d, 0x5c, 0xe7, 0xe2, 0x59, 0x2b, 0x72, 0xaf, 0xb1, 0xde, 0xe5,
    0x32, 0x05, 0x60, 0x3d, 0x45, 0x70, 0x8f, 0xc1, 0x8b, 0xba, 0x88, 0xc9, 0x31, 0xc9, 0x5e, 0xa7,
    0x5a, 0x7f, 0x10, 0x03, 0xeb, 0xbd, 0x5c, 0xa6, 0xd4, 0x89, 0x73, 0x38, 0xe0, 0xfc, 0x62, 0x9a,
    0x3c, 0x0f, 0x77, 0xbc, 0xf7, 0x08, 0xdf, 0xdc, 0xdf, 0x2d, 0x04, 0xb6, 0xc2, 0xef, 0x87, 0x0c,
    0x66, 0x65, 0x9a, 0x1b, 0xcb, 0x9c, 0x8d, 0x7b, 0xc0, 0xd0, 0xa5, 0xd9, 0x34, 0x02, 0xe0, 0xaf,
    0x94, 0x51, 0x9b, 0x4c, 0xa5, 0xef, 0x6c, 0x5a, 0xb2, 0xe8, 0xbb, 0xbd, 0x2f, 0x13, 0x7d, 0xf6,
    0x69, 0x78, 0x0b, 0x07, 0x4c, 0xd6, 0x85, 0xac, 0x02, 0xdf, 0x4a, 0xf0, 0x2b, 0xd2, 0xeb, 0xbe,
    0x84, 0xd1, 0xec, 0xd2, 0x04, 0xe7, 0x77, 0xf2, 0x7a, 0x87, 0xcd, 0xc3, 0x74, 0x6e, 0xcc, 0xa1,
    0x18, 0xc3, 0x76, 0x27, 0x79, 0x50, 0x6d, 0xb8, 0xe9, 0x5d, 0x37, 0xe2, 0x6a, 0xce, 0x1e, 0x9a,
    0x56, 0x07, 0xda, 0xb5, 0x5c, 0x18, 0xed, 0xad, 0x10, 0xd9, 0x8f, 0x71, 0x2c, 0xfa, 0xe9, 0xf9
};

const uint8_t modulusCa2Prod1[] =
{
    0xb7, 0xbf, 0x3c, 0xae, 0xeb, 0x85, 0xa6, 0xd6, 0x45, 0x95, 0xbf, 0xb3, 0x4f, 0x83, 0xe5, 0xd5,
    0xc8, 0x90, 0x11, 0xb7, 0xb4, 0x5a, 0xf6, 0x08, 0x2e, 0x6f, 0xb0, 0x90, 0xd9, 0x88, 0x3c, 0x01,
    0xe3, 0x2c, 0x6a, 0x94, 0xd8, 0xc9, 0x2e, 0x4e, 0x50, 0x84, 0xf2, 0x4c, 0xef, 0xc2, 0xa7, 0x79,
    0x1e, 0xf3, 0xeb, 0x2f, 0x7b, 0xd8, 0x03, 0x66, 0xc9, 0x09, 0xf5, 0x31, 0xa6, 0x58, 0x77, 0xa8,
    0x27, 0xcc, 0x34, 0x83, 0xa1, 0x60, 0x6f, 0x22, 0x95, 0x63, 0xba, 0x00, 0x9c, 0x85, 0x70, 0x0c,
    0x37, 0x43, 0x9b, 0xe3, 0x88, 0xad, 0x89, 0x90, 0x04, 0x38, 0x5a, 0x06, 0xbf, 0x9c, 0x45, 0x14,
    0x98, 0x69, 0x1c, 0xbe, 0x82, 0x70, 0x1f, 0x9b, 0x96, 0x20, 0xd0, 0xe5, 0x42, 0x6a, 0x1d, 0x17,
    0x28, 0xdb, 0xf5, 0x81, 0x49, 0x55, 0x16, 0x59, 0x0b, 0x18, 0x36, 0x85, 0x0f, 0xc9, 0x8b, 0xe9,
    0x70, 0xd0, 0x74, 0x9a, 0x2f, 0xfc, 0x0c, 0x02, 0xf8, 0x07, 0x10, 0x1b, 0x55, 0x16, 0x2e, 0xee,
    0x85, 0xe7, 0x79, 0xc2, 0xb6, 0xd3, 0x7b, 0x47, 0x9a, 0x5c, 0xc8, 0x53, 0x15, 0x31, 0xfb, 0xa4,
    0x4f, 0x2f, 0xae, 0xac, 0x9f, 0x7d, 0x8f, 0xce, 0x24, 0xb1, 0x30, 0x14, 0x92, 0xe3, 0x26, 0xef,
    0xab, 0x7c, 0xa0, 0xb1, 0x64, 0xf4, 0xc8, 0x5f, 0x14, 0x71, 0xd7, 0x04, 0x88, 0x71, 0xeb, 0x37,
    0x27, 0xee, 0x66, 0x51, 0x79, 0x8d, 0xa7, 0x03, 0x9d, 0x2d, 0x28, 0x76, 0xc6, 0x5f, 0xf5, 0xc4,
    0x00, 0x2b, 0x24, 0xd8, 0x8b, 0xbe, 0x06, 0x29, 0x7a, 0xe3, 0x5c, 0xc1, 0x4c, 0x5e, 0x48, 0x74,
    0x9b, 0xd9, 0x75, 0xf1, 0x97, 0x1a, 0x6b, 0x7d, 0x29, 0x05, 0xb7, 0x4a, 0x7e, 0x86, 0x0a, 0xdf,
    0xe8, 0x17, 0x8e, 0xfe, 0xb5, 0xbc, 0xb6, 0x8a, 0xe9, 0xdd, 0xe2, 0xb6, 0x3f, 0x25, 0x4c, 0xef
};
static_assert(sizeof(modulusCa2Dev1) == sizeof(modulusCa2Prod1), "");

const uint8_t pubkeyCa1Dev1[] =
{
    // x
    0x00, 0x46, 0x91, 0x3d, 0xf8, 0xbb, 0x8d, 0x1c, 0x74, 0x5e, 0x26, 0x91, 0x50, 0xfb, 0xa7, 0x7e,
    0xe3, 0xf8, 0x2e, 0x4f, 0x3f, 0x87, 0x69, 0x1b, 0xc4, 0x0a, 0x52, 0x94, 0x76, 0xe9,
    // y
    0x01, 0x7e, 0xb9, 0x5d, 0xe1, 0xb1, 0x28, 0xe4, 0xf8, 0x26, 0xb8, 0x4f, 0xa6, 0x0a, 0xd8, 0xd8,
    0x8d, 0x6b, 0x17, 0x01, 0x3c, 0x2e, 0x43, 0x67, 0xf2, 0xe7, 0xbb, 0x45, 0x40, 0x17
};

const uint8_t pubkeyCa1Prod1[] =
{
    // x
    0x01, 0x5a, 0x32, 0xc7, 0x6d, 0xe9, 0xee, 0x97, 0xc1, 0x6f, 0xd6, 0xcd, 0x26, 0x16, 0x34, 0x4c,
    0x53, 0x65, 0xcd, 0x61, 0x67, 0x5d, 0xeb, 0xb5, 0xc8, 0x75, 0x61, 0xe6, 0x11, 0xd6,
    // y
    0x00, 0xe1, 0x2b, 0xa3, 0xdc, 0xb6, 0xaf, 0xbc, 0x0b, 0x0e, 0x15, 0xa1, 0x4d, 0xb5, 0x99, 0x4b,
    0x17, 0xdc, 0x9b, 0x0e, 0xdb, 0xe8, 0xa2, 0x6f, 0x19, 0xab, 0x2d, 0x1a, 0xf3, 0x00
};
static_assert(sizeof(pubkeyCa1Dev1) == sizeof(pubkeyCa1Prod1), "");

const uint8_t pubkeyCaAmiiboTest[] =
{
    // x
    0x6c, 0x34, 0xbe, 0xea, 0xd0, 0x7a, 0x25, 0x57, 0xc1, 0xb9, 0x0d, 0xac, 0xde, 0x73, 0x0d, 0x03,
    0xeb, 0x69, 0x27, 0xed, 0xd6, 0x02, 0x8e, 0xc9, 0xa7, 0xb6, 0x41, 0xad, 0xad, 0xf2, 0x5d, 0x9c,
    // y
    0x2d, 0xda, 0xa2, 0x40, 0x11, 0x84, 0x1c, 0x77, 0x17, 0x8e, 0x03, 0x62, 0xd9, 0x52, 0xac, 0x86,
    0x42, 0x18, 0x75, 0x59, 0xbf, 0x2a, 0x7f, 0x9c, 0x06, 0x6e, 0x84, 0xa9, 0x73, 0xb8, 0xe0, 0x3a
};

const uint8_t pubkeyCaAmiiboProd[] =
{
    // x
    0xa6, 0xc9, 0x94, 0x47, 0x76, 0xa1, 0x12, 0xf2, 0xe3, 0xfe, 0xc5, 0xc1, 0x33, 0x08, 0xf1, 0xfe,
    0x49, 0xe8, 0x25, 0x9a, 0xf2, 0xf1, 0xd1, 0x3b, 0x5d, 0x7f, 0xa8, 0xab, 0xb1, 0xa2, 0xef, 0x68,
    // y
    0xd5, 0x0b, 0x7c, 0x42, 0x0b, 0x8a, 0x54, 0x77, 0x65, 0x84, 0x23, 0xf0, 0xdf, 0x8d, 0x53, 0x06,
    0x24, 0xfe, 0xe9, 0xad, 0xad, 0x92, 0xd3, 0x72, 0x02, 0x1f, 0xae, 0x8c, 0xa2, 0xae, 0x6f, 0xdf
};
static_assert(sizeof(pubkeyCaAmiiboTest) == sizeof(pubkeyCaAmiiboProd), "");

void concatKeyGeneration(
        ::nn::Bit8* pOut,
        const void* pKey,
        size_t keySize,
        const void* pGen,
        size_t genSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOut);
    NN_SDK_ASSERT_NOT_NULL(pKey);
    NN_SDK_ASSERT_NOT_NULL(pGen);
    ::std::memcpy(pOut, pKey, keySize);
    ::std::memcpy(pOut + keySize, pGen, genSize);
}

bool extractContent(
        const DataRange* pData, DataRange* pDataContent) NN_NOEXCEPT
{
    uint32_t sizeSuppliment = pData->bytes[0] == 0x03 ? 1 : 0;  // 1 if BIT STRING
    uint32_t sizeSize = 0;
    uint32_t sizeContent = 0;

    if (pData->bytes[1] < 128)
    {
        sizeContent = pData->bytes[1];
    }
    else
    {
        sizeSize = pData->bytes[1] - 128;
        for (int i = 0; i < sizeSize; i++)
        {
            sizeContent = sizeContent * 256 + pData->bytes[2 + i];
        }
    }
    if (pData->size < 2 + sizeSize + sizeContent)  // note: sizeContent includes sizeSuppliment
    {
        NN_MANU_KEY_VALIDATOR_LOG(
                "[keypair, ASN.1] cannot extract content: data shortage\n");
        return false;
    }
    pDataContent->bytes = pData->bytes + 2 + sizeSize + sizeSuppliment;
    pDataContent->size  = sizeContent - sizeSuppliment;

    return true;
}

bool stepin(DataRange* pData) NN_NOEXCEPT
{
    return extractContent(pData, pData);
}

bool stepover(DataRange* pData) NN_NOEXCEPT
{
    DataRange dataChild;
    uint32_t offset = 0;

    if (!extractContent(pData, &dataChild))
    {
        return false;
    }
    offset = dataChild.bytes + dataChild.size - pData->bytes;
    if (pData->size <= offset)
    {
        NN_MANU_KEY_VALIDATOR_LOG("[keypair, ASN.1] invalid stepover: data shortage\n");
        return false;
    }
    pData->bytes += offset;
    pData->size -= offset;

    return true;
}

bool extractFirst(const DataRange* pData, DataRange* pDataFirst) NN_NOEXCEPT
{
    DataRange second = *pData;

    if (!stepover(&second))
    {
        return false;
    }
    pDataFirst->bytes = pData->bytes;
    pDataFirst->size = second.bytes - pData->bytes;

    return true;
}

bool truncateBigint(DataRange* pBigint, uint32_t sizeMax)
{
    // remove addtional 0x00 prefix in signed big integer
    if (pBigint->size == sizeMax + 1 && pBigint->bytes[0] == 0x00)
    {
        pBigint->bytes++;
        pBigint->size--;
    }
    return pBigint->size <= sizeMax;
}

bool verifyCertEshop(uint8_t* bytesCert, uint32_t sizePayload) NN_NOEXCEPT
{
    uint8_t* pubkeyCa = nullptr;
    uint8_t* payload = nullptr;
    uint8_t* signature = nullptr;

    nn::spl::InitializeForEs();
    NN_UTIL_SCOPE_EXIT
    {
        nn::spl::Finalize();
    };

    if (nn::spl::IsDevelopment())
    {
        pubkeyCa = (uint8_t*)pubkeyCa1Dev1;
    }
    else
    {
        pubkeyCa = (uint8_t*)pubkeyCa1Prod1;
    }

    // get payload and signature
    payload   = bytesCert + 0x0080;
    signature = bytesCert + 0x0004;

    // verify signature
    uint8_t digest[nn::crypto::Sha256Generator::HashSize] = { 0 };
    nn::crypto::GenerateSha256Hash(
            digest, sizeof(digest), payload, sizePayload);
    return CSL_VerifyEccSig(
            digest, sizeof(digest), pubkeyCa, signature) == CSL_OK;
}

bool verifyRsaFlag(const uint8_t* bytesCert, uint32_t expectation) NN_NOEXCEPT
{
    uint32_t flag = nn::util::LoadBigEndian(
            reinterpret_cast<const uint32_t*>(bytesCert + 0x0144));
    return flag == expectation;
}

bool verifyRsaCertIdentifier(const uint8_t* bytesCert, uint32_t expectation) NN_NOEXCEPT
{
    uint32_t identifier = nn::util::LoadBigEndian(
            reinterpret_cast<const uint32_t*>(bytesCert + 0x020C));
    return identifier == expectation;
}

bool verifyDeviceCertIssuer(const uint8_t* bytesCert, const char* expectation) NN_NOEXCEPT
{
    const char* issuer = reinterpret_cast<const char*>(bytesCert + 0x80);
    return nn::util::Strncmp(issuer, expectation, 64) == 0;
}

bool verifySslCertIssuer(
        const uint8_t* bytesCert, uint64_t size, const char* expectation) NN_NOEXCEPT
{
    uint8_t cert[size];
    std::memcpy(cert, bytesCert, size);

    DataRange asnWork;
    asnWork.bytes = cert;
    asnWork.size = size;

    DataRange commonName;
    auto isParseOk = stepin(&asnWork)
        && stepin(&asnWork)   // cert payload
        && stepover(&asnWork) // cert version
        && stepover(&asnWork) // cert serial number
        && stepover(&asnWork) // signature algorithm
        && stepin(&asnWork)   // issuer
        && stepover(&asnWork) // issuer country name
        && stepover(&asnWork) // issuer state
        && stepover(&asnWork) // issuer locality name
        && stepover(&asnWork) // issuer organization name
        && stepin(&asnWork)   // issuer common name
        && stepin(&asnWork)   // issuer common name payload
        && stepover(&asnWork) // issuer common name payload oid
        && extractContent(&asnWork, &commonName); // issuer common name payload string
    if (!isParseOk)
    {
        NN_MANU_KEY_VALIDATOR_LOG(
                "[keytype, SSL] failed to extract issuer name from cert.\n");
        return false;
    }

    const char* issuer = reinterpret_cast<const char*>(commonName.bytes);
    return nn::util::Strncmp(issuer, expectation, commonName.size) == 0;
}

class SslKeypairVerifier final
{
    NN_DISALLOW_COPY(SslKeypairVerifier);
    NN_DISALLOW_MOVE(SslKeypairVerifier);

private:
    DataRange m_ModulusCa;
    DataRange m_Payload;
    DataRange m_Modulus;
    uint8_t   m_PublicExponent[sizePublicExponent];
    nn::cal::SslCertificateBlock     m_CertBlock;
    nn::cal::SslCertificateSizeBlock m_CertSizeBlock;
    nn::cal::ExtendedSslKeyBlock     m_KeyBlock;

public:
    explicit SslKeypairVerifier(
            const nn::cal::CalibrationInfo& info) NN_NOEXCEPT
        : m_CertBlock(info.body.sslCertificateBlock)
        , m_CertSizeBlock(info.body.sslCertificateSizeBlock)
        , m_KeyBlock(info.body.extendedSslKeyBlock)
    {
        ::std::memset(m_PublicExponent, 0, sizeof(m_PublicExponent));
    }

    bool Run() NN_NOEXCEPT;

private:
    void GetModulusCa() NN_NOEXCEPT;
    bool GetPayloadAndSignature() NN_NOEXCEPT;
    bool GetModulusAndExponent() NN_NOEXCEPT;
    bool CheckKeypair() NN_NOEXCEPT;
};

bool SslKeypairVerifier::Run() NN_NOEXCEPT
{
    this->GetModulusCa();

    if (!this->GetPayloadAndSignature())
    {
        return false;
    }

    if (!this->GetModulusAndExponent())
    {
        return false;
    }

    if (!this->CheckKeypair())
    {
        return false;
    }

    return true;
}

void SslKeypairVerifier::GetModulusCa() NN_NOEXCEPT
{
    nn::spl::InitializeForEs();
    NN_UTIL_SCOPE_EXIT
    {
        nn::spl::Finalize();
    };

    if (nn::spl::IsDevelopment())
    {
        m_ModulusCa.bytes = (uint8_t*)modulusCa2Dev1;
        m_ModulusCa.size = sizeof(modulusCa2Dev1);
    }
    else
    {
        m_ModulusCa.bytes = (uint8_t*)modulusCa2Prod1;
        m_ModulusCa.size = sizeof(modulusCa2Prod1);
    }
}

bool SslKeypairVerifier::GetPayloadAndSignature() NN_NOEXCEPT
{
    DataRange asnWork;
    asnWork.bytes = m_CertBlock.sslCertificate.data;
    asnWork.size = m_CertSizeBlock.sslCertificateSize;

    DataRange signature;
    bool isParseOk = stepin(&asnWork)
        && extractFirst(&asnWork, &m_Payload)
        && stepover(&asnWork)
        && stepover(&asnWork)
        && extractContent(&asnWork, &signature)
        && truncateBigint(&signature, sizeRsa);
    if (!isParseOk)
    {
        NN_MANU_KEY_VALIDATOR_LOG(
                "[keypair, SSL] failed to extract payload/signature from cert.\n");
        return false;
    }

    // verify signature
    if (!nn::crypto::Rsa2048Pkcs1Sha256Verifier::Verify(
                signature.bytes, signature.size,
                m_ModulusCa.bytes, m_ModulusCa.size,
                publicExponentCa, sizeof(publicExponentCa),
                m_Payload.bytes, m_Payload.size))
    {
        NN_MANU_KEY_VALIDATOR_LOG("[keypair, SSL] failed to verify cert.\n");
        return false;
    }

    return true;
}

bool SslKeypairVerifier::GetModulusAndExponent() NN_NOEXCEPT
{
    DataRange asnWork = m_Payload;
    DataRange publicExponentOnCert;

    bool isParseOk = stepin(&asnWork)  // cert payload
        && stepover(&asnWork)     // cert version
        && stepover(&asnWork)     // cert serial number
        && stepover(&asnWork)     // signature algorithm
        && stepover(&asnWork)     // issuer
        && stepover(&asnWork)     // validity
        && stepover(&asnWork)     // subject
        && stepin(&asnWork)       // subject public key
        && stepover(&asnWork)     // subject public key data bitstring
        && stepin(&asnWork)       // subject public key data
        && stepin(&asnWork)       // subject public key data modulus
        && extractContent(&asnWork, &m_Modulus)
        && truncateBigint(&m_Modulus, sizeRsa)
        && stepover(&asnWork)     // subject public key data public exponent
        && extractContent(&asnWork, &publicExponentOnCert)
        && truncateBigint(&publicExponentOnCert, sizePublicExponent);
    if (!isParseOk)
    {
        NN_MANU_KEY_VALIDATOR_LOG(
                "[keypair, SSL] failed to extract device pubkey from cert.\n");
        return false;
    }

    // make publicExponent 4-byte because nn::spl::ModularExponentiate() doesn't accept 3-byte one
    ::std::memcpy(
            m_PublicExponent + sizeof(m_PublicExponent) - publicExponentOnCert.size,
            publicExponentOnCert.bytes,
            publicExponentOnCert.size);

    return true;
}

bool SslKeypairVerifier::CheckKeypair() NN_NOEXCEPT
{
    uint8_t blockIn[sizeRsa] = { 0 };
    uint8_t blockOut[sizeRsa] = { 0 };

    // setup input
    auto offset = sizeof(blockIn) - sizeof(testdata);
    ::std::memcpy(blockIn + offset, testdata, sizeof(testdata));

    nn::spl::InitializeForSsl();
    NN_UTIL_SCOPE_EXIT
    {
        nn::spl::Finalize();
    };

    nn::crypto::RsaCalculator<sizeRsa, sizePublicExponent> publicKey;
    if (!publicKey.Initialize(
                m_Modulus.bytes, m_Modulus.size, m_PublicExponent, sizeof(m_PublicExponent)))
    {
        NN_MANU_KEY_VALIDATOR_LOG("[keypair, SSL] failed to encrypt with device pubkey\n");
        return false;
    }
    if (!publicKey.ModExp(blockOut, sizeof(blockOut), blockIn, sizeof(blockIn)))
    {
        NN_MANU_KEY_VALIDATOR_LOG("[keypair, SSL] failed to encrypt testdata\n");
        return false;
    }

    const auto SslKeyEncSize =
        sizeof(m_KeyBlock.extendedSslKey) +
        sizeof(m_KeyBlock.encryptionKeyGeneration);
    ::nn::Bit8 sslKeyEnc[SslKeyEncSize] = { 0 };
    concatKeyGeneration(
            sslKeyEnc,
            &m_KeyBlock.extendedSslKey,
            sizeof(m_KeyBlock.extendedSslKey),
            &m_KeyBlock.encryptionKeyGeneration,
            sizeof(m_KeyBlock.encryptionKeyGeneration));

    auto result = nn::Result();
    uint8_t privateExponent[sizeRsa] = { 0 };
    if (m_KeyBlock.encryptionKeyGeneration.IsZero())
    {
        result = nn::spl::ExtractSslClientCertKey(
                privateExponent, sizeof(privateExponent),
                sslKeyEnc, sizeof(sslKeyEnc));
        if (result.IsFailure())
        {
            NN_MANU_KEY_VALIDATOR_LOG(
                    "[keypair, SSL] failed to decrypt device privkey: desc=spl.%d\n",
                    result.GetDescription());
            return false;
        }

        nn::crypto::RsaCalculator<sizeRsa, sizeRsa> privateKey;
        if (!privateKey.Initialize(
                    m_Modulus.bytes, m_Modulus.size, privateExponent, sizeof(privateExponent)))
        {
            NN_MANU_KEY_VALIDATOR_LOG("[keypair, SSL] failed to decrypt with device privkey\n");
            return false;
        }
        if (!privateKey.ModExp(blockOut, sizeof(blockOut), blockOut, sizeof(blockOut)))
        {
            NN_MANU_KEY_VALIDATOR_LOG("[keypair, SSL] failed to decrypt testdata\n");
            return false;
        }
    }
    else
    {
        result = nn::spl::ExtractSslClientCertKey(
                privateExponent, sizeof(privateExponent),
                sslKeyEnc, sizeof(sslKeyEnc));
        if (result.IsSuccess())
        {
            NN_MANU_KEY_VALIDATOR_LOG("[keypair, SSL] failed to confirm that legacy api returns failure\n");
            return false;
        }

        // setup privkey
        result = nn::spl::DecryptAndStoreSslClientCertKey(
                sslKeyEnc, sizeof(sslKeyEnc));
        if (result.IsFailure())
        {
            NN_MANU_KEY_VALIDATOR_LOG("[keypair, SSL] failed to decrypt and store device privkey(encrypted): desc=spl.%d\n",
                    result.GetDescription());
            return false;
        }

        result = nn::spl::ModularExponentiateWithSslClientCertKey(
                blockOut, sizeof(blockOut),
                blockOut, sizeof(blockOut),
                m_Modulus.bytes, m_Modulus.size);
        if (result.IsFailure())
        {
            NN_MANU_KEY_VALIDATOR_LOG("[keypair, SSL] failed to decrypt with device privkey: desc=spl.%d\n",
                    result.GetDescription());
            return false;
        }
    }

    // check encrypted-and-decrypted input
    if (::std::memcmp(blockOut, blockIn, sizeof(blockOut)) == 0)
    {
        NN_MANU_KEY_VALIDATOR_LOG("[keypair, SSL] device keypair [OK]\n");
    }
    else
    {
        NN_MANU_KEY_VALIDATOR_LOG("[keypair, SSL] device keypair not matched.\n");
        return false;
    }

    return true;
}

class EccDeviceKeypairVerifier final
{
    NN_DISALLOW_COPY(EccDeviceKeypairVerifier);
    NN_DISALLOW_MOVE(EccDeviceKeypairVerifier);

private:
    DataRange m_Pubkey;
    DataRange m_Privkey;
    uint8_t m_PrivkeyFromSpl[256 / 8];
    nn::cal::ExtendedEccB233DeviceKeyBlock m_KeyBlock;
    nn::cal::EccB233DeviceCertificateBlock m_CertBlock;
    uint32_t m_Version;

public:
    EccDeviceKeypairVerifier(
            const nn::cal::CalibrationInfo& info, uint32_t version) NN_NOEXCEPT
        : m_KeyBlock(info.body.extendedEccB233DeviceKeyBlock)
        , m_CertBlock(info.body.eccB233DeviceCertificateBlock)
        , m_Version(version)
    {
        ::std::memset(m_PrivkeyFromSpl, 0, sizeof(m_PrivkeyFromSpl));
    }

    bool Run() NN_NOEXCEPT;

private:
    bool GetPubkey() NN_NOEXCEPT;
    bool GetPrivkey() NN_NOEXCEPT;
    bool CheckKeypair() NN_NOEXCEPT;
};

bool EccDeviceKeypairVerifier::Run() NN_NOEXCEPT
{
    if (!this->GetPubkey())
    {
        return false;
    }

    if (!this->GetPrivkey())
    {
        return false;
    }

    if (!this->CheckKeypair())
    {
        return false;
    }

    return true;
}

bool EccDeviceKeypairVerifier::GetPubkey() NN_NOEXCEPT
{
    if (!verifyCertEshop(m_CertBlock.deviceCertificate.data, 256))
    {
        NN_MANU_KEY_VALIDATOR_LOG("[keypair, eShop Device ECC] failed to verify cert.\n");
        return false;
    }

    if (m_Version >= 11)
    {
        if (!verifyRsaFlag(m_CertBlock.deviceCertificate.data, 1))
        {
            NN_MANU_KEY_VALIDATOR_LOG(
                    "[keypair, eShop Device ECC] failed to verify RSA-2048 flag on cert\n");
        }
    }

    m_Pubkey.bytes = m_CertBlock.deviceCertificate.data + 0x0108;
    m_Pubkey.size = sizeEccB233 * 2;

    return true;
}

bool EccDeviceKeypairVerifier::GetPrivkey() NN_NOEXCEPT
{
    const auto PrivkeyEncSize =
        sizeof(m_KeyBlock.extendedDeviceKey) +
        sizeof(m_KeyBlock.encryptionKeyGeneration);
    ::nn::Bit8 privkeyEnc[PrivkeyEncSize] ={ 0 };
    concatKeyGeneration(
            privkeyEnc,
            &m_KeyBlock.extendedDeviceKey,
            sizeof(m_KeyBlock.extendedDeviceKey),
            &m_KeyBlock.encryptionKeyGeneration,
            sizeof(m_KeyBlock.encryptionKeyGeneration));

    nn::spl::InitializeForEs();
    NN_UTIL_SCOPE_EXIT{ nn::spl::Finalize(); };

    auto result = nn::spl::ExtractDrmDeviceCertEccKey(
            m_PrivkeyFromSpl, sizeof(m_PrivkeyFromSpl),
            privkeyEnc, sizeof(privkeyEnc));
    if (result.IsFailure())
    {
        NN_MANU_KEY_VALIDATOR_LOG(
                "[keypair, eShop Device ECC] failed to decrypt device privkey: desc=spl.%d\n",
                result.GetDescription());
        return false;
    }

    m_Privkey.bytes = m_PrivkeyFromSpl + sizeof(m_PrivkeyFromSpl) - sizeEccB233;
    m_Privkey.size = sizeEccB233;

    return true;
}

bool EccDeviceKeypairVerifier::CheckKeypair() NN_NOEXCEPT
{
    uint8_t pubkeyCalc[sizeEccB233 * 2] = { 0 };

    CSL_error resultCsl = CSL_GenerateEccPublicKey(m_Privkey.bytes, pubkeyCalc);
    if (resultCsl != CSL_OK)
    {
        NN_MANU_KEY_VALIDATOR_LOG(
                "[keypair, eShop Device ECC] failed to calculate device pubkey from privkey: desc=csl.%d\n",
                resultCsl);
        return false;
    }

    if (::std::memcmp(pubkeyCalc, m_Pubkey.bytes, sizeEccB233 * 2) == 0)
    {
        NN_MANU_KEY_VALIDATOR_LOG("[keypair, eShop Device ECC] device keypair [OK]\n");
    }
    else
    {
        NN_MANU_KEY_VALIDATOR_LOG("[keypair, eShop Device ECC] device keypair not matched.\n");
        return false;
    }

    return true;
}

class RsaDeviceKeypairVerifier final
{
    NN_DISALLOW_COPY(RsaDeviceKeypairVerifier);
    NN_DISALLOW_MOVE(RsaDeviceKeypairVerifier);

private:
    DataRange m_Modulus;
    DataRange m_PublicExponent;
    nn::cal::ExtendedRsa2048DeviceKeyBlock m_KeyBlock;
    nn::cal::Rsa2048DeviceCertificateBlock m_CertBlock;
    uint32_t m_Version;

public:
    RsaDeviceKeypairVerifier(
            const nn::cal::CalibrationInfo& info, uint32_t version) NN_NOEXCEPT
        : m_KeyBlock(info.body.extendedRsa2048DeviceKeyBlock)
        , m_CertBlock(info.body.rsa2048DeviceCertificateBlock)
        , m_Version(version)
    {
    }

    bool Run() NN_NOEXCEPT;

private:
    bool GetModulusAndExponent() NN_NOEXCEPT;
    bool CheckKeypair() NN_NOEXCEPT;
};

bool RsaDeviceKeypairVerifier::Run() NN_NOEXCEPT
{
    if (m_Version < 8)
    {
        return true;
    }

    if (!this->GetModulusAndExponent())
    {
        return false;
    }

    if (!this->CheckKeypair())
    {
        return false;
    }

    return true;
}

bool RsaDeviceKeypairVerifier::GetModulusAndExponent() NN_NOEXCEPT
{
    // verify cert
    if (!verifyCertEshop(m_CertBlock.rsa2048DeviceCertificate.data, 448))
    {
        NN_MANU_KEY_VALIDATOR_LOG("[keypair, eShop Device RSA] failed to verify cert.\n");
        return false;
    }

    if (!verifyRsaCertIdentifier(m_CertBlock.rsa2048DeviceCertificate.data, 1))
    {
        NN_MANU_KEY_VALIDATOR_LOG("[keypair, eShop Device RSA] failed to verify identifier on cert.");
        return false;
    }

    // get modulus, public exponent from cert
    m_Modulus.bytes = m_CertBlock.rsa2048DeviceCertificate.data + 0x0108;
    m_Modulus.size = sizeRsa;
    m_PublicExponent.bytes = m_CertBlock.rsa2048DeviceCertificate.data + 0x0208;
    m_PublicExponent.size = sizePublicExponent;

    return true;
}

bool RsaDeviceKeypairVerifier::CheckKeypair() NN_NOEXCEPT
{
    uint8_t blockIn[sizeRsa] = { 0 };
    uint8_t blockOut[sizeRsa] = { 0 };

    auto offset = sizeof(blockIn) - sizeof(testdata);
    ::std::memcpy(blockIn + offset, testdata, sizeof(testdata));

    nn::spl::InitializeForEs();
    NN_UTIL_SCOPE_EXIT
    {
        nn::spl::Finalize();
    };

    nn::crypto::RsaCalculator<sizeRsa, sizePublicExponent> publicKey;
    if (!publicKey.Initialize(
                m_Modulus.bytes, m_Modulus.size, m_PublicExponent.bytes, m_PublicExponent.size))
    {
        NN_MANU_KEY_VALIDATOR_LOG(
                "[keypair, eShop Device RSA] failed to encrypt with device pubkey\n");
        return false;
    }
    if (!publicKey.ModExp(blockOut, sizeof(blockOut), blockIn, sizeof(blockIn)))
    {
        NN_MANU_KEY_VALIDATOR_LOG(
                "[keypair, eShop Device RSA] failed to encrypt testdata\n");
        return false;
    }

    const auto PrivkeyEncSize =
        sizeof(m_KeyBlock.extendedDeviceKey) +
        sizeof(m_KeyBlock.encryptionKeyGeneration);
    ::nn::Bit8 privkeyEnc[PrivkeyEncSize] = { 0 };
    concatKeyGeneration(
            privkeyEnc,
            &m_KeyBlock.extendedDeviceKey,
            sizeof(m_KeyBlock.extendedDeviceKey),
            &m_KeyBlock.encryptionKeyGeneration,
            sizeof(m_KeyBlock.encryptionKeyGeneration));

    auto result = nn::spl::DecryptAndStoreDrmDeviceCertKey(
            privkeyEnc, sizeof(privkeyEnc));
    if (result.IsFailure())
    {
        NN_MANU_KEY_VALIDATOR_LOG("[keypair, eShop Device RSA] failed to decrypt and store device privkey: desc=spl.%d\n",
                result.GetDescription());
        return false;
    }

    result = nn::spl::ModularExponentiateWithDrmDeviceCertKey(
            blockOut, sizeof(blockOut),
            blockOut, sizeof(blockOut),
            m_Modulus.bytes, m_Modulus.size);
    if (result.IsFailure())
    {
        NN_MANU_KEY_VALIDATOR_LOG(
                "[keypair, eShop Device RSA] failed to decrypt with device privkey: desc=spl.%d\n",
                result.GetDescription());
        return false;
    }

    // check encrypted-and-decrypt input
    if (::std::memcmp(blockOut, blockIn, sizeof(blockOut)) == 0)
    {
        NN_MANU_KEY_VALIDATOR_LOG(
                "[keypair, eShop Device RSA] device keypair [OK]\n");
    }
    else
    {
        NN_MANU_KEY_VALIDATOR_LOG(
                "[keypair, eShop Device RSA] device keypair not matched.\n");
        return false;
    }

    return true;
}

class RsaEticketKeypairVerifier final
{
    NN_DISALLOW_COPY(RsaEticketKeypairVerifier);
    NN_DISALLOW_MOVE(RsaEticketKeypairVerifier);

private:
    DataRange m_Modulus;
    DataRange m_PublicExponent;
    nn::cal::ExtendedRsa2048ETicketKeyBlock m_KeyBlock;
    nn::cal::Rsa2048ETicketCertificateBlock m_CertBlock;

public:
    explicit RsaEticketKeypairVerifier(
            const nn::cal::CalibrationInfo& info) NN_NOEXCEPT
        : m_KeyBlock(info.body.extendedRsa2048ETicketKeyBlock)
        , m_CertBlock(info.body.rsa2048ETicketCertificateBlock)
    {
    }

    bool Run() NN_NOEXCEPT;

private:
    bool GetModulusAndExponent() NN_NOEXCEPT;
    bool CheckKeypair() NN_NOEXCEPT;
};

bool RsaEticketKeypairVerifier::Run() NN_NOEXCEPT
{
    if (!this->GetModulusAndExponent())
    {
        return false;
    }

    if (!this->CheckKeypair())
    {
        return false;
    }

    return true;
}

bool RsaEticketKeypairVerifier::GetModulusAndExponent() NN_NOEXCEPT
{
    if (!verifyCertEshop(m_CertBlock.rsa2048ETicketCertificate.data, 448))
    {
        NN_MANU_KEY_VALIDATOR_LOG("[keypair, eShop eTicket RSA] failed to verify cert.\n");
        return false;
    }

    if (!verifyRsaCertIdentifier(m_CertBlock.rsa2048ETicketCertificate.data, 0))
    {
        NN_MANU_KEY_VALIDATOR_LOG("[keypair, eShop eTicket RSA] failed to verify identifier on cert.");
        return false;
    }

    // get modulus, public exponent from cert
    m_Modulus.bytes = m_CertBlock.rsa2048ETicketCertificate.data + 0x0108;
    m_Modulus.size = sizeRsa;
    m_PublicExponent.bytes = m_CertBlock.rsa2048ETicketCertificate.data + 0x0208;
    m_PublicExponent.size = sizePublicExponent;

    return true;
}

bool RsaEticketKeypairVerifier::CheckKeypair() NN_NOEXCEPT
{
    nn::spl::InitializeForEs();
    NN_UTIL_SCOPE_EXIT
    {
        nn::spl::Finalize();
    };

    // encrypt testdata with device pubkey
    uint8_t testdataEnc[sizeRsa] = { 0 };
    const uint8_t seed[nn::crypto::Sha256Generator::HashSize] = { 0 };
    nn::crypto::RsaOaepEncryptor<sizeRsa, nn::crypto::Sha256Generator>::Encrypt(
            testdataEnc, sizeof(testdataEnc),
            m_Modulus.bytes, m_Modulus.size,
            m_PublicExponent.bytes, m_PublicExponent.size,
            testdata, sizeof(testdata),
            seed, sizeof(seed),
            "", 0);

    // setup privkey
    const auto PrivkeyEncSize =
        sizeof(m_KeyBlock.extendedRsa2048ETicketKey) +
        sizeof(m_KeyBlock.encryptionKeyGeneration);
    ::nn::Bit8 privkeyEnc[PrivkeyEncSize] = { 0 };
    concatKeyGeneration(
            privkeyEnc,
            &m_KeyBlock.extendedRsa2048ETicketKey,
            sizeof(m_KeyBlock.extendedRsa2048ETicketKey),
            &m_KeyBlock.encryptionKeyGeneration,
            sizeof(m_KeyBlock.encryptionKeyGeneration));

    auto result = nn::spl::LoadEsDeviceKey(privkeyEnc, sizeof(privkeyEnc));
    if (result.IsFailure())
    {
        NN_MANU_KEY_VALIDATOR_LOG("[keypair, eShop eTicket RSA] failed to decrypt device privkey: desc=spl.%d\n",
                result.GetDescription());
        return false;
    }

    // decrypt testdata with device privkey
    const uint8_t sha256ofZeroByte[] =
    {
        0xE3, 0xB0, 0xC4, 0x42, 0x98, 0xFC, 0x1C, 0x14, 0x9A, 0xFB, 0xF4, 0xC8, 0x99, 0x6F, 0xB9, 0x24,
        0x27, 0xAE, 0x41, 0xE4, 0x64, 0x9B, 0x93, 0x4C, 0xA4, 0x95, 0x99, 0x1B, 0x78, 0x52, 0xB8, 0x55,
    };
    nn::spl::AccessKey accessKeyForTitleKey;

    result = nn::spl::PrepareEsTitleKey(
            &accessKeyForTitleKey,
            &testdataEnc, sizeof(testdataEnc),
            m_Modulus.bytes, m_Modulus.size,
            sha256ofZeroByte, sizeof(sha256ofZeroByte), 0);
    if (result.IsFailure())
    {
        // if keypair is invalid, OAEP Hash(Label) etc. are broken, then RSA-OAEP decryption fails.
        NN_MANU_KEY_VALIDATOR_LOG(
                "[keypair, eShop eTicket RSA] failed to decrypt with device privkey: desc=spl.%d\n",
                result.GetDescription());
        return false;
    }

    NN_MANU_KEY_VALIDATOR_LOG("[keypair, eShop eTicket RSA] device keypair [OK]\n");

    return true;
}

class AmiiboEccP160KeypairVerifier final
{
    NN_DISALLOW_COPY(AmiiboEccP160KeypairVerifier);
    NN_DISALLOW_MOVE(AmiiboEccP160KeypairVerifier);

private:
    DataRange m_Privkey;
    uint8_t m_PrivkeyFromSpl[256 / 8];
    nn::cal::AmiiboKeyBlock m_KeyBlock;
    nn::cal::AmiiboEcqvCertificateBlock m_CertBlock;
    nn::cal::AmiiboEcdsaCertificateBlock m_RootCertBlock;
    DataRange m_PubkeyCa;

public:
    explicit AmiiboEccP160KeypairVerifier(
            const nn::cal::CalibrationInfo& info) NN_NOEXCEPT
        : m_KeyBlock(info.body.amiiboKeyBlock)
        , m_CertBlock(info.body.amiiboEcqvCertificateBlock)
        , m_RootCertBlock(info.body.amiiboEcdsaCertificateBlock)
    {
        ::std::memset(m_PrivkeyFromSpl, 0, sizeof(m_PrivkeyFromSpl));
    }

    bool Run() NN_NOEXCEPT;

private:
    bool GetPrivkey() NN_NOEXCEPT;
    bool VerifyRootCertificate() NN_NOEXCEPT;
};

bool AmiiboEccP160KeypairVerifier::Run() NN_NOEXCEPT
{
    if (!this->GetPrivkey())
    {
        return false;
    }

    if (!this->VerifyRootCertificate())
    {
        return false;
    }

    // TODO: 鍵ペア検証

    NN_MANU_KEY_VALIDATOR_LOG("[keypair, amiibo p160] device key and root cert [OK]\n");
    return true;
}

bool AmiiboEccP160KeypairVerifier::GetPrivkey() NN_NOEXCEPT
{
    const auto PrivkeyEncSize =
        sizeof(m_KeyBlock.amiiboKey) +
        sizeof(m_KeyBlock.encryptionKeyGeneration);
    ::nn::Bit8 privkeyEnc[PrivkeyEncSize] = { 0 };
    concatKeyGeneration(
            privkeyEnc,
            &m_KeyBlock.amiiboKey,
            sizeof(m_KeyBlock.amiiboKey),
            &m_KeyBlock.encryptionKeyGeneration,
            sizeof(m_KeyBlock.encryptionKeyGeneration));

    nn::spl::InitializeForEs();
    NN_UTIL_SCOPE_EXIT
    {
        nn::spl::Finalize();
    };
    auto result = nn::spl::ExtractNfpEccP160Key(
            m_PrivkeyFromSpl, sizeof(m_PrivkeyFromSpl),
            privkeyEnc, sizeof(privkeyEnc));
    if (result.IsFailure())
    {
        NN_MANU_KEY_VALIDATOR_LOG(
                "[keypair, amiibo p160] failed to decrypt privkey: desc=spl.%d\n",
                result.GetDescription());
        return false;
    }

    m_Privkey.bytes = m_PrivkeyFromSpl;
    m_Privkey.size = sizeof(m_PrivkeyFromSpl);

    return true;
}

bool AmiiboEccP160KeypairVerifier::VerifyRootCertificate() NN_NOEXCEPT
{
    nn::spl::InitializeForEs();
    NN_UTIL_SCOPE_EXIT
    {
        nn::spl::Finalize();
    };

    if (nn::spl::IsDevelopment())
    {
        m_PubkeyCa.bytes = (uint8_t*)pubkeyCaAmiiboTest;
        m_PubkeyCa.size = sizeof(pubkeyCaAmiiboTest);
    }
    else
    {
        m_PubkeyCa.bytes = (uint8_t*)pubkeyCaAmiiboProd;
        m_PubkeyCa.size = sizeof(pubkeyCaAmiiboProd);
    }

    DataRange signature;
    DataRange payload;
    signature.bytes = m_RootCertBlock.amiiboEcdsaCertificate.data;
    signature.size = sizeEccP256 * 2;
    payload.bytes = signature.bytes + signature.size;
    payload.size = 0x30;

    if (!nn::crypto::VerifyEcdsaP256Sha256(
                signature.bytes, signature.size,
                m_PubkeyCa.bytes, m_PubkeyCa.size,
                payload.bytes, payload.size))
    {
        NN_MANU_KEY_VALIDATOR_LOG(
                "[keypair, amiibo p160] failed to verify root cert.\n");
        return false;
    }

    return true;
}

class AmiiboEccBn128KeypairVerifier final
{
    NN_DISALLOW_COPY(AmiiboEccBn128KeypairVerifier);
    NN_DISALLOW_MOVE(AmiiboEccBn128KeypairVerifier);

private:
    DataRange m_Privkey;
    uint8_t m_PrivkeyFromSpl[256 / 8];
    nn::cal::AmiiboEcqvBlsKeyBlock m_KeyBlock;
    nn::cal::AmiiboEcqvBlsCertificateBlock m_CertBlock;
    nn::cal::AmiiboEcqvBlsRootCertificateBlock m_RootCertBlock;
    DataRange m_PubkeyCa;

public:
    explicit AmiiboEccBn128KeypairVerifier(
            const nn::cal::CalibrationInfo& info) NN_NOEXCEPT
        : m_KeyBlock(info.body.amiiboEcqvBlsKeyBlock)
        , m_CertBlock(info.body.amiiboEcqvBlsCertificateBlock)
        , m_RootCertBlock(info.body.amiiboEcqvBlsRootCertificateBlock)
    {
        ::std::memset(m_PrivkeyFromSpl, 0, sizeof(m_PrivkeyFromSpl));
    }

    bool Run() NN_NOEXCEPT;

private:
    bool GetPrivkey() NN_NOEXCEPT;
    bool VerifyRootCertificate() NN_NOEXCEPT;
};

bool AmiiboEccBn128KeypairVerifier::Run() NN_NOEXCEPT
{
    if (!this->GetPrivkey())
    {
        return false;
    }

    if (!this->VerifyRootCertificate())
    {
        return false;
    }

    // TODO: 鍵ペア検証

    NN_MANU_KEY_VALIDATOR_LOG("[keypair, amiibo bn128] device key and root cert [OK]\n");
    return true;
}

bool AmiiboEccBn128KeypairVerifier::GetPrivkey() NN_NOEXCEPT
{
    const auto PrivkeyEncSize =
        sizeof(m_KeyBlock.amiiboEcqvBlsKey) +
        sizeof(m_KeyBlock.encryptionKeyGeneration);
    ::nn::Bit8 privkeyEnc[PrivkeyEncSize] = { 0 };
    concatKeyGeneration(
            privkeyEnc,
            &m_KeyBlock.amiiboEcqvBlsKey,
            sizeof(m_KeyBlock.amiiboEcqvBlsKey),
            &m_KeyBlock.encryptionKeyGeneration,
            sizeof(m_KeyBlock.encryptionKeyGeneration));

    nn::spl::InitializeForEs();
    NN_UTIL_SCOPE_EXIT
    {
        nn::spl::Finalize();
    };
    auto result = nn::spl::ExtractNfpEccBn128Key(
            m_PrivkeyFromSpl, sizeof(m_PrivkeyFromSpl),
            privkeyEnc, sizeof(privkeyEnc));
    if (result.IsFailure())
    {
        NN_MANU_KEY_VALIDATOR_LOG(
                "[keypair, amiibo bn128] failed to decrypt privkey: desc=spl.%d\n",
                result.GetDescription());
        return false;
    }

    m_Privkey.bytes = m_PrivkeyFromSpl;
    m_Privkey.size = sizeof(m_PrivkeyFromSpl);

    return true;
}

bool AmiiboEccBn128KeypairVerifier::VerifyRootCertificate() NN_NOEXCEPT
{
    nn::spl::InitializeForEs();
    NN_UTIL_SCOPE_EXIT
    {
        nn::spl::Finalize();
    };

    if (nn::spl::IsDevelopment())
    {
        m_PubkeyCa.bytes = (uint8_t*)pubkeyCaAmiiboTest;
        m_PubkeyCa.size = sizeof(pubkeyCaAmiiboTest);
    }
    else
    {
        m_PubkeyCa.bytes = (uint8_t*)pubkeyCaAmiiboProd;
        m_PubkeyCa.size = sizeof(pubkeyCaAmiiboProd);
    }

    DataRange signature;
    DataRange payload;
    signature.bytes = m_RootCertBlock.amiiboEcqvBlsRootCertificate.data;
    signature.size = sizeEccP256 * 2;
    payload.bytes = signature.bytes + signature.size;
    payload.size = 0x50;

    if (!nn::crypto::VerifyEcdsaP256Sha256(
                signature.bytes, signature.size,
                m_PubkeyCa.bytes, m_PubkeyCa.size,
                payload.bytes, payload.size))
    {
        NN_MANU_KEY_VALIDATOR_LOG(
                "[keypair, amiibo bn128] failed to verify root cert.\n");
        return false;
    }

    return true;
}

bool VerifyKeyEncryption(const nn::cal::CalibrationInfo& info, uint32_t version) NN_NOEXCEPT
{
    // バージョン 9 より前は各秘密鍵が固有鍵で暗号化されていない
    if (version >= 9)
    {
        if (info.body.extendedRsa2048DeviceKeyBlock.encryptionKeyGeneration.IsZero())
        {
            NN_MANU_KEY_VALIDATOR_LOG(
                    "[encryption, eShop Device RSA] failed to verify encryption generation.\n");
            return false;
        }
        else if (info.body.extendedRsa2048ETicketKeyBlock.encryptionKeyGeneration.IsZero())
        {
            NN_MANU_KEY_VALIDATOR_LOG(
                    "[encryption, eShop eTicket RSA] failed to verify encryption generation.\n");
            return false;
        }
        else if (info.body.extendedSslKeyBlock.encryptionKeyGeneration.IsZero())
        {
            NN_MANU_KEY_VALIDATOR_LOG(
                    "[encryption, SSL] failed to verify encryption generation.\n");
            return false;
        }
        else if (info.body.extendedGameCardKeyBlock.encryptionKeyGeneration.IsZero())
        {
            NN_MANU_KEY_VALIDATOR_LOG(
                    "[encryption, GameCard] failed to verify encryption generation.\n");
            return false;
        }
        else if (info.body.amiiboKeyBlock.encryptionKeyGeneration.IsZero())
        {
            NN_MANU_KEY_VALIDATOR_LOG(
                    "[encryption, amiibo] failed to verify encryption generation.\n");
            return false;
        }
        else if (info.body.amiiboEcqvBlsKeyBlock.encryptionKeyGeneration.IsZero())
        {
            NN_MANU_KEY_VALIDATOR_LOG(
                    "[encryption, amiibo EcqvBls] failed to verify encryption generation.\n");
            return false;
        }
    }

    // バージョン 11 より前は ECC-B233 秘密鍵が固有鍵で暗号化されていない
    if (version >= 11)
    {
        if (info.body.extendedEccB233DeviceKeyBlock.encryptionKeyGeneration.IsZero())
        {
            NN_MANU_KEY_VALIDATOR_LOG(
                    "[encryption, eShop Device ECC] failed to verify encryption generation.\n");
            return false;
        }
    }

    NN_MANU_KEY_VALIDATOR_LOG("[encryption] key encryption [OK]\n");
    return true;
}

bool VerifyKeypairSsl(const nn::cal::CalibrationInfo& info, uint32_t version) NN_NOEXCEPT
{
    NN_UNUSED(version);
    SslKeypairVerifier verifier(info);
    return verifier.Run();
}

bool VerifyKeypairEccDevice(const nn::cal::CalibrationInfo& info, uint32_t version) NN_NOEXCEPT
{
    EccDeviceKeypairVerifier verifier(info, version);
    return verifier.Run();
}

bool VerifyKeypairRsaDevice(const nn::cal::CalibrationInfo& info, uint32_t version) NN_NOEXCEPT
{
    RsaDeviceKeypairVerifier verifier(info, version);
    return verifier.Run();
}

bool VerifyKeypairRsaEticket(const nn::cal::CalibrationInfo& info, uint32_t version) NN_NOEXCEPT
{
    NN_UNUSED(version);
    RsaEticketKeypairVerifier verifier(info);
    return verifier.Run();
}

bool VerifyKeypairAmiiboEccP160(const nn::cal::CalibrationInfo& info, uint32_t version) NN_NOEXCEPT
{
    NN_UNUSED(version);
    AmiiboEccP160KeypairVerifier verifier(info);
    return verifier.Run();
}

bool VerifyKeypairAmiiboEccBn128(const nn::cal::CalibrationInfo& info, uint32_t version) NN_NOEXCEPT
{
    NN_UNUSED(version);
    AmiiboEccBn128KeypairVerifier verifier(info);
    return verifier.Run();
}

bool VerifyKeyType(const nn::cal::CalibrationInfo& info, uint32_t version) NN_NOEXCEPT
{
    nn::spl::InitializeForEs();
    NN_UTIL_SCOPE_EXIT
    {
        nn::spl::Finalize();
    };
    const char* issuer =
        nn::spl::IsDevelopment() ? "NintendoNXCA1Dev1": "NintendoNXCA1Prod1";

    if (!verifyDeviceCertIssuer(
                info.body.eccB233DeviceCertificateBlock.deviceCertificate.data, issuer))
    {
        NN_MANU_KEY_VALIDATOR_LOG("[keytype, eShop Device ECC] failed to verify key type.\n");
        return false;
    }
    else if (!verifyDeviceCertIssuer(
                info.body.rsa2048ETicketCertificateBlock.rsa2048ETicketCertificate.data, issuer))
    {
        NN_MANU_KEY_VALIDATOR_LOG("[keytype, eShop Device RSA] failed to verify key type.\n");
        return false;
    }

    // バージョン 8 より前は RSA-2048 版のデバイス登録用秘密鍵が書かれていない
    if (version >= 8)
    {
        if (!verifyDeviceCertIssuer(
                    info.body.rsa2048DeviceCertificateBlock.rsa2048DeviceCertificate.data, issuer))
        {
            NN_MANU_KEY_VALIDATOR_LOG("[keytype, eShop Device RSA] failed to verify key type.\n");
            return false;
        }
    }

    const char* sslIssuer =
        nn::spl::IsDevelopment() ? "NintendoNXCA2Dev1": "NintendoNXCA2Prod1";
    if (!verifySslCertIssuer(
                info.body.sslCertificateBlock.sslCertificate.data,
                info.body.sslCertificateSizeBlock.sslCertificateSize,
                sslIssuer))
    {
        NN_MANU_KEY_VALIDATOR_LOG("[keytype, SSL] failed to verify key type.\n");
        return false;
    }

    NN_MANU_KEY_VALIDATOR_LOG("[keytype] key type [OK]\n");
    return true;
}

} // namespace

nn::Result ValidatePrivateKey(bool& isSuccess) NN_NOEXCEPT
{
    NN_RESULT_DO(ValidatePrivateKey(isSuccess, nullptr));
    NN_RESULT_SUCCESS;
}

nn::Result ValidatePrivateKey(
        bool& isSuccess, const LogFunctionPointer pLog) NN_NOEXCEPT
{
    g_pLog = pLog;
    NN_UTIL_SCOPE_EXIT
    {
        g_pLog = nullptr;
    };

    std::unique_ptr<nn::cal::CalibrationInfo> calibrationInfo(new nn::cal::CalibrationInfo());
    NN_RESULT_DO(GetCalibrationInfo(calibrationInfo.get()));

    const auto& header = calibrationInfo->header;
    const auto& version = header.version;

    bool isOk = true;
    isOk &= VerifyKeyEncryption(*calibrationInfo, version);
    isOk &= VerifyKeypairSsl(*calibrationInfo, version);
    isOk &= VerifyKeypairEccDevice(*calibrationInfo, version);
    isOk &= VerifyKeypairRsaDevice(*calibrationInfo, version);
    isOk &= VerifyKeypairRsaEticket(*calibrationInfo, version);

    nn::spl::InitializeForEs();
    auto isDevelopment = nn::spl::IsDevelopment();
    nn::spl::Finalize();
    if (!isDevelopment)
    {
        isOk &= VerifyKeypairAmiiboEccBn128(*calibrationInfo, version);
        isOk &= VerifyKeypairAmiiboEccP160(*calibrationInfo, version);
    }

    isOk &= VerifyKeyType(*calibrationInfo, version);

    isSuccess = isOk;

    NN_RESULT_SUCCESS;
}

}} // namespace nn::manu

