﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/ldn/detail/ldn_Context.h>
#include <nn/ldn/detail/ldn_LocalCommunicationServiceManager.h>
#include <nn/ldn/detail/Debug/ldn_Assert.h>
#include <nn/ldn/detail/Debug/ldn_Log.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace ldn { namespace detail { namespace
{

}}}} // namespace nn::ldn::detail::<unnamed>

namespace nn { namespace ldn { namespace detail
{
    LocalCommunicationServiceManager::LocalCommunicationServiceManager(
        void* buffer, size_t bufferSize, Core* pCore) NN_NOEXCEPT
        : m_Mutex(true),
          m_StartLocalCommunicationEvent(nn::os::EventClearMode_ManualClear),
          m_StopLocalCommunicationEvent(nn::os::EventClearMode_AutoClear),
          m_ReleaseNetworkInterfaceEvent(nn::os::EventClearMode_ManualClear),
          m_CancelEvent(nn::os::EventClearMode_ManualClear),
          m_Buffer(static_cast<impl::LocalCommunicationServiceManagerBuffer*>(buffer)),
          m_pCore(pCore),
          m_ActiveContextId(InvalidContextId)
    {
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_ALIGNED(buffer, nn::os::ThreadStackAlignment);
        NN_SDK_ASSERT(RequiredBufferSize <= bufferSize);
        NN_SDK_ASSERT_NOT_NULL(pCore);
        NN_UNUSED(bufferSize);
        m_StopLocalCommunicationEvent.Signal();
        nn::os::CreateThread(
            &m_NetworkInterfaceMonitorThread, NetworkInterfaceMonitorThread, this,
            m_Buffer->networkInterfaceMonitorThreadStack,
            sizeof(m_Buffer->networkInterfaceMonitorThreadStack),
            NN_SYSTEM_THREAD_PRIORITY(ldn, NetworkInterfaceMonitor));
        nn::os::SetThreadNamePointer(
            &m_NetworkInterfaceMonitorThread, NN_SYSTEM_THREAD_NAME(ldn, NetworkInterfaceMonitor));
        nn::os::StartThread(&m_NetworkInterfaceMonitorThread);
    }

    LocalCommunicationServiceManager::~LocalCommunicationServiceManager() NN_NOEXCEPT
    {
        m_CancelEvent.Signal();
        nn::os::WaitThread(&m_NetworkInterfaceMonitorThread);
        nn::os::DestroyThread(&m_NetworkInterfaceMonitorThread);
    }

    int LocalCommunicationServiceManager::CreateContext() NN_NOEXCEPT
    {
        return nn::ldn::detail::CreateContext();
    }

    Result LocalCommunicationServiceManager::DestroyContext(int contextId) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        Context* pContext = GetContext(contextId);
        NN_LDN_REQUIRES_NOT_NULL(pContext);
        if (m_ActiveContextId == contextId)
        {
            StopLocalCommunicationImpl(contextId, true);
        }
        nn::ldn::detail::DestroyContext(contextId);
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceManager::StartLocalCommunication(
        int contextId, Bit64 pid, const NetworkInterfaceRequestParameter& param) NN_NOEXCEPT
    {
        NN_LDN_REQUIRES_MINMAX(param.priority, PriorityMin, PriorityMax);
        NN_LDN_REQUIRES_NOT_EQUAL(contextId, m_ActiveContextId);

        // コンテキストを取得します。
        Context* pContext = GetContext(contextId);
        NN_LDN_REQUIRES_NOT_NULL(pContext);
        NN_RESULT_DO(pContext->discardReason);
        pContext->pid = pid;

        // ネットワーク・インタフェースの占有を要求します。
        NetworkInterfaceRequest request;
        Result rejectReason = request.Submit(param);
        if (rejectReason.IsFailure())
        {
            NN_LDN_LOG_INFO("failed to start local communication (context=%d, reason=0x%08X)\n",
                contextId, rejectReason.GetInnerValueForDebug());
            return rejectReason;
        }
        m_StopLocalCommunicationEvent.Wait();

        // ネットワーク・インタフェースの監視を開始します。
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        Result result = m_pCore->StartLocalCommunication(request.GetNetworkInterface());
        if (result.IsSuccess())
        {
            NN_LDN_LOG_INFO("local communication started (context=%d)\n", contextId);
            m_ActiveContextId = contextId;
            m_ActiveRequest = std::move(request);
            m_StartLocalCommunicationEvent.Signal();
            NN_RESULT_SUCCESS;
        }
        else
        {
            NN_LDN_LOG_ERROR("failed to start local communication (context=%d, reason=0x%08X)\n",
                contextId, result.GetInnerValueForDebug());
            request.Release();
            return result;
        }
    }

    Result LocalCommunicationServiceManager::StopLocalCommunication(int contextId) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_RESULT_DO(IsActiveContext(contextId));
        NN_RESULT_DO(StopLocalCommunicationImpl(contextId, false));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceManager::StopLocalCommunicationImpl(
        int contextId, bool isTriggeredBySystem) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(contextId, m_ActiveContextId);
        NN_UNUSED(contextId);
        m_ActiveContextId = InvalidContextId;
        m_pCore->StopLocalCommunication(isTriggeredBySystem);
        m_ActiveRequest.Release();
        NN_LDN_LOG_INFO("local communication finished (context=%d)\n", contextId);
        NN_RESULT_SUCCESS;
    }

    State LocalCommunicationServiceManager::GetState(int contextId) NN_NOEXCEPT
    {
        return contextId == m_ActiveContextId ? m_pCore->GetState() : State_Error;
    }

    Result LocalCommunicationServiceManager::Scan(
        int contextId, NetworkInfo* pOutScanResultArray, int* pOutCount, int bufferCount,
        const ScanFilter& filter, int channel) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_RESULT_DO(IsActiveContext(contextId));

        // デフォルトのローカル通信識別子を取得します。
        Context* pContext = GetContext(contextId);
        ScanFilter completedFilter = filter;
        if ((filter.flag & ScanFilterFlag_LocalCommunicationId) != 0 &&
            (filter.networkId.intentId.localCommunicationId == DefaultLocalCommunicationId))
        {
            GetDefaultLocalCommunicationId(
                &completedFilter.networkId.intentId.localCommunicationId, pContext->pid);
        }

        // ネットワークを探索します。
        NN_RESULT_DO(m_pCore->Scan(
            pOutScanResultArray, pOutCount, bufferCount, completedFilter, channel));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceManager::GetNetworkInfo(
        int contextId, NetworkInfo* pOutNetwork,
        NodeLatestUpdate* outUpdates, int bufferCount) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_RESULT_DO(IsActiveContext(contextId));
        NN_RESULT_DO(m_pCore->GetNetworkInfo(pOutNetwork, outUpdates, bufferCount));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceManager::GetNetworkInfo(
        int contextId, NetworkInfo* pOutNetwork) const NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_RESULT_DO(IsActiveContext(contextId));
        NN_RESULT_DO(m_pCore->GetNetworkInfo(pOutNetwork));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceManager::GetIpv4Address(
        int contextId, Ipv4Address* pOutAddress, SubnetMask* pOutMask) const NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_RESULT_DO(IsActiveContext(contextId));
        NN_RESULT_DO(m_pCore->GetIpv4Address(pOutAddress, pOutMask));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceManager::GetSecurityParameter(
        int contextId, SecurityParameter* pOutParam) const NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_RESULT_DO(IsActiveContext(contextId));
        NN_RESULT_DO(m_pCore->GetSecurityParameter(pOutParam));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceManager::GetNetworkConfig(
        int contextId, NetworkConfig* pOutNetworkConfig) const NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_RESULT_DO(IsActiveContext(contextId));
        NN_RESULT_DO(m_pCore->GetNetworkConfig(pOutNetworkConfig));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceManager::OpenAccessPoint(
        int contextId, nn::os::SystemEventType* pStateChangeEvent) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_RESULT_DO(IsActiveContext(contextId));
        NN_RESULT_DO(m_pCore->OpenAccessPoint(pStateChangeEvent));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceManager::CloseAccessPoint(int contextId) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_RESULT_DO(IsActiveContext(contextId));
        NN_RESULT_DO(m_pCore->CloseAccessPoint(false));
        NN_RESULT_SUCCESS;
    }


    Result LocalCommunicationServiceManager::CreateNetwork(
        int contextId,
        const NetworkConfig& network,
        const SecurityConfig& securityConfig,
        const SecurityParameter& securityParam,
        const UserConfig& user,
        int addressEntryCount,
        const AddressEntry* addressEntries) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        // アクティブなコンテキストでなければ実行できません。
        NN_RESULT_DO(IsActiveContext(contextId));

        // ローカル通信識別子の検証です。
        Context* pContext = GetContext(contextId);
        auto completedNetwork = network;
        if (network.intentId.localCommunicationId == DefaultLocalCommunicationId)
        {
            GetDefaultLocalCommunicationId(
                &completedNetwork.intentId.localCommunicationId, pContext->pid);
        }
        else if (!HasPermission(pContext->pid, network.intentId.localCommunicationId))
        {
            return ResultUnauthorizedLocalCommunicationId();
        }

        // ネットワークを構築します。
        nn::Result result;
        const int retryCountMax = 5;
        for (int i = 0; i < retryCountMax; ++i)
        {
            result = m_pCore->CreateNetwork(
                completedNetwork, securityConfig, securityParam, user,
                addressEntryCount, addressEntries);
            if (nn::ldn::ResultL2NotSupportedChannel::Includes(result) ||
                nn::ldn::ResultL2InvalidState::Includes(result))
            {
                // ネットワークを破棄してからリトライします。
                DestroyNetwork(contextId);
            }
            else
            {
                break;
            }
        }

        // リトライしても成功しなかった場合は Abort します。
        if (nn::ldn::ResultL2NotSupportedChannel::Includes(result) ||
            nn::ldn::ResultL2InvalidState::Includes(result))
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
        return result;
    }

    Result LocalCommunicationServiceManager::DestroyNetwork(int contextId) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_RESULT_DO(IsActiveContext(contextId));
        NN_RESULT_DO(m_pCore->DestroyNetwork(false));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceManager::SetAdvertiseData(
        int contextId, const void* data, size_t dataSize) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_RESULT_DO(IsActiveContext(contextId));
        NN_RESULT_DO(m_pCore->SetAdvertiseData(data, dataSize));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceManager::SetStationAcceptPolicy(
        int contextId, AcceptPolicy policy) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_RESULT_DO(IsActiveContext(contextId));
        NN_RESULT_DO(m_pCore->SetStationAcceptPolicy(policy));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceManager::Reject(
        int contextId, Ipv4Address ipv4Address) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_RESULT_DO(IsActiveContext(contextId));
        NN_RESULT_DO(m_pCore->Reject(ipv4Address));
        NN_RESULT_SUCCESS;
    }


    Result LocalCommunicationServiceManager::AddAcceptFilterEntry(
        int contextId, MacAddress macAddress) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_RESULT_DO(IsActiveContext(contextId));
        NN_RESULT_DO(m_pCore->AddAcceptFilterEntry(macAddress));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceManager::ClearAcceptFilter(int contextId) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_RESULT_DO(IsActiveContext(contextId));
        NN_RESULT_DO(m_pCore->ClearAcceptFilter());
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceManager::OpenStation(
        int contextId, nn::os::SystemEventType* pStateChangeEvent) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_RESULT_DO(IsActiveContext(contextId));
        NN_RESULT_DO(m_pCore->OpenStation(pStateChangeEvent));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceManager::CloseStation(int contextId) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_RESULT_DO(IsActiveContext(contextId));
        NN_RESULT_DO(m_pCore->CloseStation(false));
        NN_RESULT_SUCCESS;
    }


    Result LocalCommunicationServiceManager::Connect(
        int contextId,
        const NetworkConfig& network,
        const SecurityConfig& securityConfig,
        const SecurityParameter& securityParam,
        const UserConfig& user,
        int version,
        ConnectOption option) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        // アクティブなコンテキストでなければ実行できません。
        NN_RESULT_DO(IsActiveContext(contextId));

        // ローカル通信識別子の検証です。
        Context* pContext = GetContext(contextId);
        auto completedNetwork = network;
        if (network.intentId.localCommunicationId == DefaultLocalCommunicationId)
        {
            GetDefaultLocalCommunicationId(
                &completedNetwork.intentId.localCommunicationId, pContext->pid);
        }
        else if (!HasPermission(pContext->pid, network.intentId.localCommunicationId))
        {
            return ResultUnauthorizedLocalCommunicationId();
        }

        // ネットワークに接続します。
        NN_RESULT_DO(m_pCore->Connect(
            completedNetwork, securityConfig, securityParam, user, version, option));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceManager::Disconnect(int contextId) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_RESULT_DO(IsActiveContext(contextId));
        NN_RESULT_DO(m_pCore->Disconnect(false));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceManager::GetDisconnectReason(
        int contextId, DisconnectReason* pOutReason) const NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        const Context* pContext = GetContext(contextId);
        if (IsActiveContext(contextId).IsSuccess())
        {
            return m_pCore->GetDisconnectReason(pOutReason);
        }
        else if (pContext != nullptr)
        {
            *pOutReason = static_cast<DisconnectReason>(pContext->disconnectReason);
        }
        else
        {
            *pOutReason = DisconnectReason_None;
        }
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceManager::SetOperationMode(int contextId, OperationMode mode) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_RESULT_DO(IsActiveContext(contextId));
        NN_RESULT_DO(m_pCore->SetOperationMode(mode));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceManager::SetWirelessControllerRestriction(
        int contextId, WirelessControllerRestriction restriction) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_RESULT_DO(IsActiveContext(contextId));
        NN_RESULT_DO(m_pCore->SetWirelessControllerRestriction(restriction));
        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationServiceManager::IsActiveContext(int contextId) const NN_NOEXCEPT
    {
        Context* pContext = GetContext(contextId);
        NN_LDN_REQUIRES_NOT_NULL(pContext);
        NN_RESULT_DO(pContext->discardReason);
        NN_LDN_REQUIRES_EQUAL(contextId, m_ActiveContextId);
        NN_RESULT_SUCCESS;
    }

    void LocalCommunicationServiceManager::NetworkInterfaceMonitorThread(void* pArg) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pArg);
        NN_LDN_LOG_DEBUG("NetworkInterfaceMonitorThread: started\n");
        auto& manager = *static_cast<LocalCommunicationServiceManager*>(pArg);
        auto& cancelEvent = manager.m_CancelEvent;
        auto& startEvent = manager.m_StartLocalCommunicationEvent;
        auto& stopEvent = manager.m_StopLocalCommunicationEvent;
        auto& activeRequest = manager.m_ActiveRequest;

        // スレッドの終了要求を受け取れるようにしておきます。
        nn::os::MultiWaitType       multiWait;
        nn::os::MultiWaitHolderType cancelHolder;
        nn::os::InitializeMultiWait(&multiWait);
        nn::os::InitializeMultiWaitHolder(&cancelHolder, manager.m_CancelEvent.GetBase());
        nn::os::LinkMultiWaitHolder(&multiWait, &cancelHolder);

        // ローカル通信の開始要求を待ち受けるようにしておきます。
        nn::os::MultiWaitHolderType startLocalCommunicationHolder;
        nn::os::InitializeMultiWaitHolder(&startLocalCommunicationHolder, startEvent.GetBase());
        nn::os::LinkMultiWaitHolder(&multiWait, &startLocalCommunicationHolder);

        // ネットワーク・インタフェースの監視イベントはローカル通信が開始されるまで生成されません。
        nn::os::MultiWaitHolderType requestStateChangeHolder;

        // スレッドの終了要求を受け取るまで待機します。
        nn::os::MultiWaitHolderType* pHolder = nullptr;
        while ((pHolder = nn::os::WaitAny(&multiWait)) != &cancelHolder)
        {
            std::lock_guard<nn::os::Mutex> lock(manager.m_Mutex);
            if (pHolder == &startLocalCommunicationHolder)
            {
                startEvent.Clear();
                auto& requestStateChangeEvent = activeRequest.GetRequestStateChangeEvent();
                nn::os::InitializeMultiWaitHolder(
                    &requestStateChangeHolder, requestStateChangeEvent.GetBase());
                nn::os::LinkMultiWaitHolder(&multiWait, &requestStateChangeHolder);
            }
            else if (pHolder == &requestStateChangeHolder)
            {
                Result result = activeRequest.GetResult();
                if (result.IsSuccess())
                {
                    auto& requestStateChangeEvent = activeRequest.GetRequestStateChangeEvent();
                    requestStateChangeEvent.Clear();
                }
                else
                {
                    int contextId = manager.m_ActiveContextId;
                    if (contextId != InvalidContextId)
                    {
                        NN_LDN_LOG_DEBUG("discarded the active context\n");

                        // 切断理由を取得します。
                        Context* pActiveContext = GetContext(manager.m_ActiveContextId);
                        if (manager.GetState(contextId) == State_StationConnected)
                        {
                            pActiveContext->disconnectReason =
                                DisconnectReason_DisconnectedBySystem;
                        }
                        else
                        {
                            DisconnectReason reason;
                            if (manager.GetDisconnectReason(contextId, &reason).IsSuccess())
                            {
                                pActiveContext->disconnectReason = static_cast<int8_t>(reason);
                            }
                        }

                        // ローカル通信を中断します。
                        pActiveContext->discardReason = result;
                        manager.StopLocalCommunicationImpl(manager.m_ActiveContextId, true);
                    }
                    nn::os::UnlinkMultiWaitHolder(&requestStateChangeHolder);
                    nn::os::FinalizeMultiWaitHolder(&requestStateChangeHolder);
                    stopEvent.Signal();
                }
            }
        }

        // 終了処理です。
        nn::os::UnlinkAllMultiWaitHolder(&multiWait);
        nn::os::FinalizeMultiWaitHolder(&startLocalCommunicationHolder);
        nn::os::FinalizeMultiWaitHolder(&cancelHolder);
        nn::os::FinalizeMultiWait(&multiWait);
        cancelEvent.Clear();
        NN_LDN_LOG_DEBUG("NetworkInterfaceMonitorThread: finished\n");
    }

}}} // namespace nn::ldn::detail
