﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/ldn/ldn_Result.h>
#include <nn/ldn/detail/Debug/ldn_Log.h>
#include <nn/ldn/detail/NetworkInterface/ldn_NintendoEthernet.h>
#include <nn/ldn/detail/NetworkInterface/ldn_FrameReceiver.h>
#include <nn/os/os_EventApi.h>
#include <nn/os/os_ThreadApi.h>

namespace nn { namespace ldn { namespace detail { namespace
{
    const size_t ReceiveThreadStackSize = 4 * 1024;

}}}} // namespace nn::ldn::detail::<unnamed>

namespace nn { namespace ldn { namespace detail
{
    FrameReceiver::FrameReceiver() NN_NOEXCEPT
        : m_pNetworkInterface(nullptr),
          m_ReceiveBuffer(nullptr),
          m_ReceiveBufferSize(0),
          m_IsRunning(false),
          m_ProtocolMutex(false)
    {
    }

    FrameReceiver::~FrameReceiver() NN_NOEXCEPT
    {
        if (m_IsRunning)
        {
            Finalize();
        }
    }

    size_t FrameReceiver::GetRequiredBufferSize(size_t frameSizeMax, int protocolCount) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(protocolCount <= ProtocolCountMax);
        NN_UNUSED(protocolCount);
        return ReceiveThreadStackSize + frameSizeMax;
    }

    void FrameReceiver::Initialize(
        INetworkInterface* pInterface, void* buffer, size_t bufferSize,
        size_t frameSizeMax, int protocolCount) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!m_IsRunning, "Receiver has already been initialized");
        NN_SDK_ASSERT_NOT_NULL(pInterface);
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_ALIGNED(buffer, nn::os::ThreadStackAlignment);
        NN_SDK_ASSERT(GetRequiredBufferSize(frameSizeMax, protocolCount) <= bufferSize);
        NN_UNUSED(protocolCount);
        std::memset(buffer, 0, bufferSize);
        std::memset(m_QueueList, 0, sizeof(m_QueueList));
        m_pNetworkInterface = pInterface;
        m_ReceiveBuffer = static_cast<Bit8*>(buffer) + ReceiveThreadStackSize;
        m_ReceiveBufferSize = frameSizeMax;
        m_IsRunning = true;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
            &m_ReceiveThread, ReceiveThread, this,
            buffer, ReceiveThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(ldn, Receiver)));
        nn::os::SetThreadNamePointer(&m_ReceiveThread, NN_SYSTEM_THREAD_NAME(ldn, Receiver));
        nn::os::StartThread(&m_ReceiveThread);
    }

    void FrameReceiver::Finalize() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsRunning, "Receiver has not been initialized yet");
        m_IsRunning = false;
        m_pNetworkInterface->Cancel();
        nn::os::WaitThread(&m_ReceiveThread);
        nn::os::DestroyThread(&m_ReceiveThread);
        m_ReceiveBufferSize = 0;
        m_ReceiveBuffer = nullptr;
        m_pNetworkInterface = nullptr;
    }

    void FrameReceiver::Register(
        ProtocolId protocol, nn::os::EventType* pEvent, FrameQueue* pQueue) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsRunning, "Receiver has not been initialized yet");
        NN_SDK_ASSERT_NOT_NULL(pEvent);
        NN_SDK_ASSERT_NOT_NULL(pQueue);
        std::lock_guard<nn::os::Mutex> lock(m_ProtocolMutex);
        for (int i = 0; i < ProtocolCountMax; ++i)
        {
            auto& info = m_QueueList[i];
            if (info.pQueue == nullptr)
            {
                info.protocol = protocol;
                info.pEvent = pEvent;
                info.pQueue = pQueue;
                nn::os::ClearEvent(pEvent);
                return;
            }
        }
        NN_SDK_ASSERT(false);
    }

    void FrameReceiver::Unregister(ProtocolId protocol) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsRunning, "Receiver has not been initialized yet");
        std::lock_guard<nn::os::Mutex> lock(m_ProtocolMutex);
        for (int i = 0; i < ProtocolCountMax; ++i)
        {
            auto& info = m_QueueList[i];
            if (info.protocol == protocol)
            {
                std::memset(&info, 0, sizeof(info));
                return;
            }
        }
        NN_SDK_ASSERT(false);
    }

    void FrameReceiver::ReceiveThread(void* pArg) NN_NOEXCEPT
    {
        NN_LDN_LOG_DEBUG("ReceiveThread: started\n");
        auto& receiver = *static_cast<FrameReceiver*>(pArg);
        while (receiver.m_IsRunning)
        {
            // フレームを受信します。
            size_t receivedSize;
            auto result = receiver.m_pNetworkInterface->Receive(
                receiver.m_ReceiveBuffer, &receivedSize, receiver.m_ReceiveBufferSize);

            // 受信に失敗した場合には受信スレッドを終了します。
            if (ResultCancelled::Includes(result))
            {
                break;
            }
            else if (ResultInvalidState::Includes(result))
            {
                break;
            }
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            // 受信サイズが異常な場合は無視します。
            if (receivedSize < NintendoEthernetFrameSizeMin ||
                sizeof(NintendoEthernetFrame) < receivedSize)
            {
                NN_LDN_LOG_WARN("received invalid frame: %u Bytes\n", receivedSize);
                continue;
            }

            // 受信したフレームのプロトコルに応じてキューに振り分けます。
            // 対応するプロトコルが登録されていない場合には無視します。
            const auto& frame = *static_cast<NintendoEthernetFrame*>(receiver.m_ReceiveBuffer);
            const auto& nintendoEthernetHeader = frame.nintendoEthernetHeader;
            for (int i = 0; i < ProtocolCountMax; ++i)
            {
                auto& item = receiver.m_QueueList[i];
                if (item.pQueue != nullptr && item.protocol == nintendoEthernetHeader.protocol)
                {
                    if (item.pQueue->TryEnqueue(&frame, receivedSize))
                    {
                        // キューにデータを入れたことを通知します。
                        nn::os::SignalEvent(item.pEvent);

                        // キューに入れたデータは即座に処理して貰います。
                        nn::os::YieldThread();
                    }
                    else
                    {
                        // キューに空きがない場合には受信データを破棄します。
                        NN_LDN_LOG_WARN("discarded a frame for protocol %u\n", item.protocol.code);
                    }
                    break;
                }
            }
        }
        NN_LDN_LOG_DEBUG("ReceiveThread: finished\n");
    }

}}} // namespace nn::ldn::detail
