﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/lcs/lcs_Result.h>
#include <nn/lcs/detail/lcs_Util.h>
#include <nn/lcs/detail/Debug/lcs_Log.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ContentDeliveryApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace lcs { namespace detail
{

    bool HasApplication(Bit64 id) NN_NOEXCEPT
    {
        nn::ncm::ApplicationId appId;
        appId.value = id;

        int offset = 0;
        int count = 0;
        nn::ns::ApplicationRecord recordList[ApplicationRecordListSize] = {};

        do {
            count = nn::ns::ListApplicationRecord(recordList, ApplicationRecordListSize, offset);
            for (int i = 0; i < count; ++i)
            {
                if (recordList[i].id.value == id)
                {
                    return true;
                }
            }
            offset += ApplicationRecordListSize;
        } while (count == ApplicationRecordListSize);

        NN_LCS_LOG_ERROR("Application %x is not found\n", id);

        return false;
    }

    Result GetApplicationDetailInfo(
        ApplicationDetailInfo* pOutInfo, void* buffer, size_t bufferSize, Bit64 id) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutInfo);
        NN_SDK_ASSERT_NOT_NULL(buffer);


        ::std::memset(pOutInfo, 0, sizeof(ApplicationDetailInfo));
        ::std::memset(buffer, 0, bufferSize);

        Result result;
        nn::ncm::ApplicationId appId;
        appId.value = id;

        // id を設定
        pOutInfo->id = id;

        // attributeFlag と displayVersion を取得
        size_t size = 0;
        result = nn::ns::GetApplicationControlData(&size, buffer, bufferSize,
            nn::ns::ApplicationControlSource::Storage, appId);
        if (nn::ns::ResultApplicationControlDataNotFound::Includes(result))
        {
            return ResultApplicationNotFound();
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        nn::ns::ApplicationControlDataAccessor accessor(buffer, size);
        auto& acProperty = accessor.GetProperty();

        pOutInfo->attributeFlag = acProperty.attributeFlag;
        ::std::memcpy(pOutInfo->displayVersion, acProperty.displayVersion, DisplayVersionSizeMax);

        // requiredSize を取得
        pOutInfo->requiredSize = GetRequiredSize(id);

        NN_RESULT_SUCCESS;
    }

    Result GetApplicationControlData(size_t* pOutSize, void* buffer, size_t bufferSize, Bit64 id) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutSize);
        NN_SDK_ASSERT_NOT_NULL(buffer);

        ::std::memset(buffer, 0, bufferSize);

        Result result;
        nn::ncm::ApplicationId appId;
        appId.value = id;

        nn::ns::GetApplicationControlData(pOutSize, buffer, bufferSize,
            nn::ns::ApplicationControlSource::Storage, appId);
        if (nn::ns::ResultApplicationControlDataNotFound::Includes(result))
        {
            return ResultApplicationNotFound();
        }
        else if (nn::ns::ResultBufferNotEnough::Includes(result))
        {
            return ResultBufferNotEnough();
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        NN_RESULT_SUCCESS;
    }

    int64_t GetRequiredSize(Bit64 id) NN_NOEXCEPT
    {
        Result result;
        nn::ncm::ApplicationId appId;
        nn::ns::ApplicationDeliveryInfo appInfo[ApplicationDeliveryInfoCountMax] = {};
        nn::ncm::ContentMetaKey key[ContentMetaKeyMax] = {};
        int appInfoCount = 0;
        int keyCount = 0;
        nn::ns::ApplicationDeliveryAttribute attribute = {};
        int64_t size = 0;

        appId.value = id;

        attribute.Reset();
        attribute.Set<nn::ns::ApplicationDeliveryAttribute_RequestPatch>(true);

        result = nn::ns::GetApplicationDeliveryInfo(
            &appInfoCount, appInfo, ApplicationDeliveryInfoCountMax, appId, attribute);
        if (result.IsFailure())
        {
            NN_LCS_LOG_ERROR("Failed to GetApplicationDeliveryInfo : 0x%08X\n",
                result.GetInnerValueForDebug());
            return 0;
        }

        result = nn::ns::ListContentMetaKeyToDeliverApplication(
            &keyCount, key, ContentMetaKeyMax, 0, appInfo, appInfoCount);
        if (result.IsFailure())
        {
            NN_LCS_LOG_ERROR("Failed to list content meta key: 0x%08X\n",
                result.GetInnerValueForDebug());
            return 0;
        }

        result = nn::ns::EstimateRequiredSize(&size, key, keyCount);
        if (result.IsFailure())
        {
            NN_LCS_LOG_ERROR("Failed to estimate required size: 0x%08X\n",
                result.GetInnerValueForDebug());
            return 0;
        }
        NN_LCS_LOG_DEBUG("RequiredSize : %lld\n", size);

        return size;
    }

    bool HasFreeSpaceSize(int64_t needSize) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_GREATER_EQUAL(needSize, 0);

        Result result;
        int64_t freeSize = 0;

        // SD の空きを確認
        if (nn::ncm::IsInstallableStorage(nn::ncm::StorageId::SdCard))
        {
            result = nn::ns::GetFreeSpaceSize(&freeSize, nn::ncm::StorageId::SdCard);
            if (result.IsSuccess())
            {
                NN_LCS_LOG_DEBUG("SD FreeSpace : %lld\n", freeSize);
                if (freeSize > needSize)
                {
                    return true;
                }
            }
            else if (nn::ns::ResultStorageAccessFailed::Includes(result))
            {
                NN_LCS_LOG_DEBUG("Failed to access to sdcard\n");
            }
            else
            {
                NN_LCS_LOG_ERROR("Failed to get free space size on sdcard: 0x%08X\n",
                    result.GetInnerValueForDebug());
            }
        }

        // NAND の空きを確認
        if (nn::ncm::IsInstallableStorage(nn::ncm::StorageId::BuiltInUser))
        {
            result = nn::ns::GetFreeSpaceSize(&freeSize, nn::ncm::StorageId::BuiltInUser);
            if (result.IsSuccess())
            {
                NN_LCS_LOG_DEBUG("NAND FreeSpace : %lld\n", freeSize);
                if (freeSize > needSize)
                {
                    return true;
                }
            }
            else
            {
                NN_LCS_LOG_ERROR("Failed to get free space size on built in storage: 0x%08X\n",
                    result.GetInnerValueForDebug());
            }
        }
        return false;
    }

    void ConvertApplicationInfoToContetsInfo(
        ContentsInfo *pOutInfo, const ApplicationDetailInfo& appInfo) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutInfo);

        // ToDo : アプリ本体が配信できるならそのフラグも
        pOutInfo->contentsFlag.Reset();
        pOutInfo->contentsFlag.Set<ContentsType::Patch>(true);

        pOutInfo->applicationId.value = appInfo.id;
        pOutInfo->attributeFlag = appInfo.attributeFlag;
        ::std::memcpy(pOutInfo->displayVersion, appInfo.displayVersion, DisplayVersionSizeMax);
    }

    Result GetApplicationDeliveryInfoHash(
        void *pHash, size_t bufferSize, const ApplicationInfo &appInfo) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pHash);
        NN_SDK_ASSERT_GREATER_EQUAL(bufferSize, ApplicationDeliveryInfoHashSize);

        nn::ns::ApplicationDeliveryInfoHash appHash = {};
        nn::ns::ApplicationDeliveryInfo appDeliveryInfo[ApplicationDeliveryInfoCountMax] = {};

        for (int i = 0; i < appInfo.deliveryInfoCount; ++i)
        {
            ::std::memcpy(&appDeliveryInfo[i], appInfo.deliveryInfo[i].appDeliveryInfo,
                NsApplicationDeliveryInfoSize);
        }

        ::std::memset(pHash, 0x00, bufferSize);

        NN_RESULT_DO(nn::ns::GetApplicationDeliveryInfoHash(
            &appHash, appDeliveryInfo, appInfo.deliveryInfoCount));

        ::std::memcpy(pHash, appHash.data, ApplicationDeliveryInfoHashSize);

        NN_RESULT_SUCCESS;
    }

    bool IsApplicationDeliveryInfoHash(const ApplicationDeliveryInfoHash& appHash) NN_NOEXCEPT
    {
        for (int i = 0; i < ApplicationDeliveryInfoHashSize; ++i)
        {
            if (appHash.hash[i] != 0)
            {
                return true;
            }
        }
        return false;
    }

}}} // end of namespace nn::lcs
