﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/lcs/lcs_Result.h>
#include <nn/lcs/lcs_PrivateResult.h>
#include <nn/lcs/detail/lcs_NetworkInfo.h>
#include <nn/lcs/detail/lcs_SystemShareApi.h>
#include <nn/lcs/detail/Debug/lcs_Log.h>
#include <nn/ns/ns_ContentDeliveryApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_SystemUpdateSystemApi.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace lcs { namespace detail
{

    Result GetSystemDeliveryInfo(SystemDeliveryInfo *pOutInfo) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutInfo);

        nn::ns::SystemDeliveryInfo sysInfo = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ns::GetSystemDeliveryInfo(&sysInfo));

        ::std::memcpy(pOutInfo, &sysInfo, sizeof(SystemDeliveryInfo));

        NN_RESULT_SUCCESS;
    }

    Result VerifyDeliveryProtocolVersion(const SystemDeliveryInfo& sysInfo) NN_NOEXCEPT
    {
        Result result;
        nn::ns::SystemDeliveryInfo verifyInfo;
        ::std::memcpy(&verifyInfo, sysInfo.sysDeliveryInfo, NsSystemDeliveryInfoSize);
        return nn::ns::VerifyDeliveryProtocolVersion(verifyInfo);
    }

    Result CompareSystemDeliveryInfo(int* pOutValue,
        const SystemDeliveryInfo& lhs, const SystemDeliveryInfo& rhs) NN_NOEXCEPT
    {
        nn::ns::SystemDeliveryInfo lInfo = {};
        nn::ns::SystemDeliveryInfo rInfo = {};

        ::std::memcpy(&lInfo, lhs.sysDeliveryInfo, NsSystemDeliveryInfoSize);
        ::std::memcpy(&rInfo, rhs.sysDeliveryInfo, NsSystemDeliveryInfoSize);

        // CompareSystemDeliveryInfo を使う時は
        // VerifyDeliveryProtocolVersion でシステムの互換性は取れているものなので、エラーなら Abort
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::ns::CompareSystemDeliveryInfo(pOutValue, lInfo, rInfo));

        //NN_LCS_LOG_DEBUG("Value : %d\n", *pOutValue);
        NN_RESULT_SUCCESS;
    }

    Result SelectLatestSystemDeliveryInfo(
        int* outValue, const SystemDeliveryInfo* list, int listCount,
        const SystemDeliveryInfo& receiverSystemInfo,
        const ApplicationDeliveryInfo* receiveAppInfoList, int appInfoListCount) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(outValue);
        NN_SDK_ASSERT_NOT_NULL(list);
        NN_SDK_ASSERT_NOT_NULL(receiveAppInfoList);

        nn::ns::SystemDeliveryInfo receiverSysInfo;
        nn::ns::SystemDeliveryInfo sysInfo[NodeCountMax] = {};
        nn::ns::ApplicationDeliveryInfo appInfo[ApplicationDeliveryInfoCountMax] = {};

        for (int i = 0; i < listCount; ++i)
        {
            ::std::memcpy(
                &sysInfo[i], list[i].sysDeliveryInfo, NsSystemDeliveryInfoSize);
        }
        ::std::memcpy(&receiverSysInfo, &receiverSystemInfo, sizeof(nn::ns::SystemDeliveryInfo));
        for (int i = 0; i < appInfoListCount; ++i)
        {
            ::std::memcpy(
                &appInfo[i], receiveAppInfoList[i].appDeliveryInfo, NsApplicationDeliveryInfoSize);
        }

        // SelectLatestSystemDeliveryInfo を使う時は
        // VerifyDeliveryProtocolVersion でシステムの互換性は取れているものなので、エラーなら Abort
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ns::SelectLatestSystemDeliveryInfo(
            outValue, sysInfo, listCount, receiverSysInfo, appInfo, appInfoListCount));

        NN_RESULT_SUCCESS;
    }

    Result SelectLatestSystemDeliveryInfo(
        int* outValue, const SystemDeliveryInfo* list, int listCount,
        const SystemDeliveryInfo& receiverSystemInfo) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(outValue);
        NN_SDK_ASSERT_NOT_NULL(list);

        nn::ns::SystemDeliveryInfo receiverSysInfo;
        nn::ns::SystemDeliveryInfo sysInfo[NodeCountMax] = {};

        for (int i = 0; i < listCount; ++i)
        {
            ::std::memcpy(
                &sysInfo[i], list[i].sysDeliveryInfo, NsSystemDeliveryInfoSize);
        }
        ::std::memcpy(&receiverSysInfo, &receiverSystemInfo, sizeof(nn::ns::SystemDeliveryInfo));

        // SelectLatestSystemDeliveryInfo を使う時は
        // VerifyDeliveryProtocolVersion でシステムの互換性は取れているものなので、エラーなら Abort
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::ns::SelectLatestSystemDeliveryInfo(outValue, sysInfo, listCount, receiverSysInfo));

        NN_RESULT_SUCCESS;
    }

    Result RequestReceiveSystemUpdate(
        nn::ns::AsyncResult* asyncResult, Ipv4Address fromAddress, uint16_t port,
        const SystemDeliveryInfo& info,
        nn::ns::SystemUpdateControl* systemUpdateControl) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(asyncResult);

        nn::ns::SystemDeliveryInfo systemDeliveryInfo = {};
        ::std::memcpy(&systemDeliveryInfo, &info.sysDeliveryInfo, NsSystemDeliveryInfoSize);

        // タスクを複数作った時だけ発生するので、Abort させる
        NN_ABORT_UNLESS_RESULT_SUCCESS(systemUpdateControl->RequestReceiveSystemUpdate(
            asyncResult, fromAddress, port, systemDeliveryInfo));

        NN_RESULT_SUCCESS;
    }

    void GetReceiveSystemUpdateProgress(uint64_t* downloadedSize, uint64_t* totalSize,
        nn::ns::SystemUpdateControl* systemUpdateControl) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(downloadedSize);
        NN_SDK_ASSERT_NOT_NULL(totalSize);

        nn::ns::SystemUpdateProgress progress = systemUpdateControl->GetReceiveProgress();

        *downloadedSize = progress.loaded;
        *totalSize = progress.total;
    }

    Result GetDownloadedEulaDataSize(size_t* size, const char* eulaDataPath,
        nn::ns::SystemUpdateControl* systemUpdateControl) NN_NOEXCEPT
    {
        Result result = systemUpdateControl->GetReceivedEulaDataSize(size, eulaDataPath);

        if (nn::ns::ResultEulaSystemDataNotFound().Includes(result))
        {
            return ResultNotExsistEula();
        }
        else if (nn::ns::ResultEulaDataPathNotFound().Includes(result))
        {
            return ResultBadDataPath();
        }
        NN_RESULT_SUCCESS;
    }

    Result GetDownloadedEulaData(size_t* size, void* buffer, size_t bufferSize,
        const char* eulaDataPath,
        nn::ns::SystemUpdateControl* systemUpdateControl) NN_NOEXCEPT
    {
        Result result = systemUpdateControl->GetReceivedEulaData(size, buffer, bufferSize, eulaDataPath);

        if (nn::ns::ResultEulaSystemDataNotFound().Includes(result))
        {
            return ResultNotExsistEula();
        }
        else if (nn::ns::ResultEulaDataPathNotFound().Includes(result))
        {
            return ResultBadDataPath();
        }
        else if (nn::ns::ResultBufferNotEnough().Includes(result))
        {
            return ResultBufferNotEnough();
        }
        NN_RESULT_SUCCESS;
    }

    bool ApplyReceivedUpdate(nn::ns::SystemUpdateControl* systemUpdateControl) NN_NOEXCEPT
    {
        if (!systemUpdateControl->HasReceived())
        {
            return false;
        }
        systemUpdateControl->ApplyReceivedUpdate();
        return true;
    }

    Result RequestSendSystemUpdate(nn::ns::AsyncResult* asyncResult,
        Ipv4Address toAddress, uint16_t port, const SystemDeliveryInfo& info) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(asyncResult);

        nn::ns::SystemDeliveryInfo systemDeliveryInfo = {};
        ::std::memcpy(&systemDeliveryInfo, &info.sysDeliveryInfo, NsSystemDeliveryInfoSize);

        // タスクを複数作った時だけ発生するので、Abort させる
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::ns::RequestSendSystemUpdate(asyncResult, toAddress, port, systemDeliveryInfo));

        NN_RESULT_SUCCESS;
    }

    void GetSendSystemUpdateProgress(uint64_t* deliveredSize, uint64_t* totalSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(deliveredSize);
        NN_SDK_ASSERT_NOT_NULL(totalSize);

        nn::ns::SystemUpdateProgress progress = {};

        progress = nn::ns::GetSendSystemUpdateProgress();

        *deliveredSize = progress.loaded;
        *totalSize = progress.total;
    }

    void ShowSystemDeliveryInfo(const SystemDeliveryInfo& info) NN_NOEXCEPT
    {
        nn::ns::SystemDeliveryInfo sysInfo;
        ::std::memcpy(&sysInfo, &info, sizeof(nn::ns::SystemDeliveryInfo));
        NN_LCS_LOG_DEBUG("SystemUpdateVersion : %x\n", sysInfo.systemUpdateVersion);
        NN_LCS_LOG_DEBUG("SystemUpdateId : %llx\n", sysInfo.systemUpdateId);
        NN_LCS_LOG_DEBUG("SystemDeliveryProtocolVersion : %x\n", sysInfo.systemDeliveryProtocolVersion);
        NN_LCS_LOG_DEBUG("ApplicationDeliveryProtocolVersion : %x\n", sysInfo.applicationDeliveryProtocolVersion);
    }

}}} // end of namespace nn::lcs
