﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/lcs/lcs_Result.h>
#include <nn/lcs/lcs_Result.private.h>
#include <nn/lcs/detail/lcs_Resource.h>
#include <nn/lcs/detail/lcs_ApplicationShareApi.h>
#include <nn/lcs/detail/Debug/lcs_Log.h>
#include <nn/lcs/detail/Packet/lcs_PacketManager.h>
#include <nn/lcs/detail/Packet/lcs_PacketReceiver.h>
#include <nn/lcs/detail/Packet/lcs_PacketSender.h>
#include <nn/lcs/detail/Packet/lcs_PacketUtilty.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/socket.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace lcs { namespace detail { namespace {

    nn::os::Mutex g_SendMutex(false);

}}}}

namespace nn { namespace lcs { namespace detail
{

    Result DownloadAppControlDataSize(size_t *pOutAppDataSize,
        int socket, const nn::ncm::ApplicationId& id,
        void* packetBuffer, size_t packetBufferSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutAppDataSize);
        NN_SDK_ASSERT_NOT_NULL(packetBuffer);
        NN_SDK_ASSERT_GREATER_EQUAL(packetBufferSize, PacketSize);

        ::std::lock_guard<nn::os::Mutex> guard(g_SendMutex);

        NN_RESULT_THROW_UNLESS(
            SendApplicationControlDataReq(socket, id, true) > 0, ResultCommunicationError());

        // タイムアウトタイマーセット
        nn::os::TimerEventType timeoutEvent;
        nn::os::InitializeTimerEvent(&timeoutEvent, nn::os::EventClearMode_AutoClear);
        nn::os::ClearTimerEvent(&timeoutEvent);
        nn::TimeSpan interval = nn::TimeSpan::FromMilliSeconds(TimeoutMilliSeconds);
        nn::os::StartOneShotTimerEvent(&timeoutEvent, interval);

        Result result = ResultCommunicationError();
        while (!nn::os::TryWaitTimerEvent(&timeoutEvent))
        {
            CommonHeader header;
            ssize_t recvSize = RecvCommonHeaderWithPeek(socket, &header);
            if (recvSize > 0)
            {
                uint32_t len = nn::socket::InetNtohl(header.length);
                NN_LCS_LOG_DEBUG("Type : %d / Length : %d\n", header.type, len);
                if (header.type == Type_ApplicationControlDataSizeRes)
                {
                    AppControlDataResPacket packet;
                    recvSize = RecvData(socket, &packet,
                        sizeof(AppControlDataResPacket), sizeof(AppControlDataResPacket));
                    Bit64 recvId = GetMerged(
                        nn::socket::InetNtohl(packet.appUpperId), nn::socket::InetNtohl(packet.appLowerId));
                    if (recvId == id.value)
                    {
                        *pOutAppDataSize = nn::socket::InetNtohl(packet.dataSize);
                        result = ResultSuccess();
                        break;
                    }
                }
                else
                {
                    recvSize = RecvData(socket, packetBuffer, packetBufferSize, len);
                }
            }
            else if (recvSize == 0)
            {
                break;
            }
        }
        nn::os::FinalizeTimerEvent(&timeoutEvent);
        return result;
    }

    Result DownloadAppControlData(
        void* pOutappDataBuffer, size_t appDataBufferSize, size_t *pOutAppDataSize,
        int socket, const nn::ncm::ApplicationId& id,
        void* packetBuffer, size_t packetBufferSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutappDataBuffer);
        NN_SDK_ASSERT_NOT_NULL(pOutAppDataSize);
        NN_SDK_ASSERT_NOT_NULL(packetBuffer);
        NN_SDK_ASSERT_GREATER_EQUAL(packetBufferSize, PacketSize);

        ::std::lock_guard<nn::os::Mutex> guard(g_SendMutex);

        NN_RESULT_THROW_UNLESS(
            SendApplicationControlDataReq(socket, id, false) > 0, ResultCommunicationError());

        // タイムアウトタイマーセット
        nn::os::TimerEventType timeoutEvent;
        nn::os::InitializeTimerEvent(&timeoutEvent, nn::os::EventClearMode_AutoClear);
        nn::os::ClearTimerEvent(&timeoutEvent);
        nn::TimeSpan interval = nn::TimeSpan::FromMilliSeconds(TimeoutMilliSeconds);
        nn::os::StartOneShotTimerEvent(&timeoutEvent, interval);

        ssize_t recvSize = 0;
        size_t downloadedSize = 0;
        size_t dataSize = 0;
        bool recvFlag = false;

        Result result = ResultCommunicationError();
        while (!nn::os::TryWaitTimerEvent(&timeoutEvent))
        {
            if (!recvFlag)
            {
                CommonHeader header;
                recvSize = RecvCommonHeaderWithPeek(socket, &header);
                if (recvSize > 0)
                {
                    uint32_t len = nn::socket::InetNtohl(header.length);
                    NN_LCS_LOG_DEBUG("Type : %d / Length : %d\n", header.type, len);
                    if (header.type == Type_ApplicationControlDataRes)
                    {
                        AppControlDataResPacket packet;
                        recvSize = RecvData(socket, &packet,
                            sizeof(AppControlDataResPacket), sizeof(AppControlDataResPacket));
                        dataSize = nn::socket::InetNtohl(packet.dataSize);
                        NN_LCS_LOG_DEBUG("DataSize : %d\n", dataSize);
                        recvFlag = true;
                    }
                    else
                    {
                        recvSize = RecvData(socket, packetBuffer, packetBufferSize, len);
                    }
                }
            }
            else
            {
                uint8_t* data = reinterpret_cast<uint8_t*>(pOutappDataBuffer);
                size_t readSize = dataSize - downloadedSize;
                recvSize = RecvData(socket, data + downloadedSize, readSize, readSize);
                //NN_LCS_LOG_DEBUG("recvSize : %d\n", recvSize);
                if (recvSize > 0)
                {
                    downloadedSize += recvSize;
                }
                if (downloadedSize == dataSize)
                {
                    NN_LCS_LOG_DEBUG("RecvApplicationControlData Success\n");
                    *pOutAppDataSize = dataSize;
                    result = ResultSuccess();
                    break;
                }
            }
        }
        nn::os::FinalizeTimerEvent(&timeoutEvent);
        return result;
    }

    Result SendAppControlDataRes(int socket, const nn::ncm::ApplicationId& id,
        void* buffer, size_t size, bool isSizeOnly) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_GREATER_EQUAL(socket, 0);
        NN_SDK_ASSERT_GREATER_EQUAL(size, TcpDataSizeMax);

        ::std::lock_guard<nn::os::Mutex> guard(g_SendMutex);

        Result result;
        size_t dataSize = 0;
        ::std::memset(buffer, 0, size);
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetApplicationControlData(&dataSize, buffer, size, id.value));

        NN_LCS_LOG_DEBUG("AppControlDataSize : %d\n", dataSize);

        NN_RESULT_THROW_UNLESS(
            SendApplicationControlDataRes(socket, id, dataSize, isSizeOnly) > 0, ResultSendError());

        if (!isSizeOnly)
        {
            NN_RESULT_THROW_UNLESS(
                SendApplicationControlData(socket, buffer, dataSize) > 0, ResultSendError());
        }
        NN_RESULT_SUCCESS;
    }

    Result SendAcceptNode(int socket, uint32_t index) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(g_SendMutex);

        NN_RESULT_THROW_UNLESS(SendAccept(socket, index) > 0, ResultSendError());
        NN_RESULT_SUCCESS;
    }

    Result SendRejectNode(int socket, RejectReason reason) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(g_SendMutex);

        NN_RESULT_THROW_UNLESS(SendReject(socket, reason) > 0, ResultSendError());
        NN_RESULT_SUCCESS;
    }

    Result SendLeaveSession(int socket, LeaveReason reason) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(g_SendMutex);

        NN_RESULT_THROW_UNLESS(SendLeave(socket, reason) > 0, ResultSendError());
        NN_RESULT_SUCCESS;
    }

    Result SendShareStart(int socket) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(g_SendMutex);

        NN_RESULT_THROW_UNLESS(SendStart(socket) > 0, ResultSendError());
        NN_RESULT_SUCCESS;
    }

    Result SendShareEnd(int socket) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(g_SendMutex);

        NN_RESULT_THROW_UNLESS(SendEnd(socket) > 0, ResultSendError());
        NN_RESULT_SUCCESS;
    }

    Result SendShareEndWithProgress(int socket,
        const AdvertiseData& advertiseData) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(g_SendMutex);

        NN_RESULT_THROW_UNLESS(SendEndWithProgress(socket, advertiseData) > 0, ResultSendError());
        NN_RESULT_SUCCESS;
    }

    Result SendHostMigration(int socket, const MigrationInfo& info) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(g_SendMutex);

        NN_RESULT_THROW_UNLESS(SendMigration(socket, info) > 0, ResultSendError());
        NN_RESULT_SUCCESS;
    }

    Result SendResumeSession(int socket) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(g_SendMutex);

        NN_RESULT_THROW_UNLESS(SendResume(socket) > 0, ResultSendError());
        NN_RESULT_SUCCESS;
    }

    Result SendNodeDetailInfo(
        int socket, char* name, const Version& lcsVersion, const SystemDeliveryInfo& sysInfo,
        const SessionInfo& sessionInfo, int downloadedContentCount,
        void* appDataBuffer, size_t appDataBufferSize, bool isUpdate) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(name);
        NN_SDK_ASSERT_NOT_NULL(appDataBuffer);
        NN_SDK_ASSERT_GREATER_EQUAL(appDataBufferSize, AppControlDataSize);

        ::std::lock_guard<nn::os::Mutex> guard(g_SendMutex);

        ApplicationDetailInfo detailInfo = {};
        ApplicationDeliveryInfo deliveryInfo[ApplicationDeliveryInfoCountMax] = {};
        int appDeliveryInfoCount = 0;
        size_t appInfoSize = 0;
        int appCount = 0;

        // 受信したいコンテンツがパッチのみだったら本体の記録があるかを確認してなければエラー
        bool isReceivable = false;
        nn::util::BitFlagSet<8, ContentsType> contentsFlag;
        contentsFlag.Reset();
        for (int i = 0; i < sessionInfo.contentsCount; ++i)
        {
            if (sessionInfo.contents[i].contentsFlag.Test(ContentsType::Application::Index))
            {
                isReceivable = true;
            }
            else if (sessionInfo.contents[i].contentsFlag.Test(ContentsType::Patch::Index))
            {
                if(HasApplication(sessionInfo.contents[i].applicationId.value))
                {
                    isReceivable = true;
                }
            }
        }
        if (!isReceivable)
        {
            return ResultNoSharableContentsSession();
        }

        // JoinPacket -> (AppInfoPacket -> AppDeliveryInfo(256Byteをそのまま) * 個数分) * コンテンツの数だけ送る
        // Jion パケットを送信
        for (int i = 0; i < sessionInfo.contentsCount; ++i)
        {
            if (GetApplicationDetailInfo(&detailInfo, appDataBuffer, appDataBufferSize,
                sessionInfo.contents[i].applicationId.value).IsSuccess() &&
                GetApplicationDeliveryInfo(&appDeliveryInfoCount, deliveryInfo,
                    ApplicationDeliveryInfoCountMax, sessionInfo.contents[i].applicationId.value).IsSuccess())
            {
                appInfoSize +=
                    sizeof(AppInfoPacket) + appDeliveryInfoCount * sizeof(ApplicationDeliveryInfo);
                ++appCount;
            }
        }
        NN_LCS_LOG_DEBUG("AppCount : %d / AppInfoSize : %d\n", appCount, appInfoSize);

        if (isUpdate)
        {
            NN_RESULT_THROW_UNLESS(
                SendUpdate(socket, name, lcsVersion, sysInfo, appCount, downloadedContentCount, appInfoSize) > 0, ResultSendError());
        }
        else
        {
            NN_RESULT_THROW_UNLESS(
                SendJoin(socket, name, lcsVersion, appCount, appInfoSize) > 0, ResultSendError());
        }

        // アプリケーションの情報送信
        AppInfoPacket packet;
        for (int i = 0; i < sessionInfo.contentsCount; ++i)
        {
            if (GetApplicationDetailInfo(&detailInfo, appDataBuffer, appDataBufferSize,
                sessionInfo.contents[i].applicationId.value).IsFailure())
            {
                continue;
            }
            NN_RESULT_DO(GetApplicationDeliveryInfo(&appDeliveryInfoCount, deliveryInfo,
                ApplicationDeliveryInfoCountMax, sessionInfo.contents[i].applicationId.value));
            packet.appDeliveryInfoCount = static_cast<uint8_t>(appDeliveryInfoCount);
            ::std::memcpy(packet.detailInfo, &detailInfo, ApplicationDetailInfoSize);

            NN_RESULT_THROW_UNLESS(
                Send(socket, &packet, sizeof(AppInfoPacket), nn::socket::MsgFlag::Msg_None) > 0, ResultSendError());

            for (int j = 0; j < appDeliveryInfoCount; ++j)
            {
                // ApplicationDeliveryInfo を生で個数分送信
                NN_RESULT_THROW_UNLESS(
                    Send(socket, &deliveryInfo[j], sizeof(ApplicationDeliveryInfo), nn::socket::MsgFlag::Msg_None) > 0, ResultSendError());
            }
        }
        NN_RESULT_SUCCESS;
    }

    Result SendSessionInfo(int socket,
        int appCount, ApplicationInfo* appInfo, int nodeCount, NodeInfo* nodeInfo) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(appInfo);
        NN_SDK_ASSERT_NOT_NULL(nodeInfo);

        ::std::lock_guard<nn::os::Mutex> guard(g_SendMutex);

        size_t packetSize = 0;

        for (int i = 0; i < appCount; ++i)
        {
            packetSize +=
                sizeof(AppInfoPacket) + (appInfo[i].deliveryInfoCount) * sizeof(ApplicationDeliveryInfo);
        }
        packetSize += nodeCount * sizeof(NodeInfoPacket);

        NN_RESULT_THROW_UNLESS(
            SendSessionInfoHeader(socket, appCount, nodeCount, packetSize) > 0, ResultSendError());

        for (int i = 0; i < appCount; ++i)
        {
            SendAppInfo(socket, appInfo[i]);
            for (int j = 0; j < appInfo[i].deliveryInfoCount; ++j)
            {
                NN_RESULT_THROW_UNLESS(
                    SendAppDeliveryInfo(socket, appInfo[i].deliveryInfo[j]) > 0, ResultSendError());
            }
        }

        for (int i = 0; i < nodeCount; ++i)
        {
            NN_RESULT_THROW_UNLESS(
                SendNodeInfo(socket, nodeInfo[i]) > 0, ResultSendError());
        }
        NN_RESULT_SUCCESS;
    }

    Result SendShareSystemClear(int socket, SystemUpdateReason reason) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(g_SendMutex);

        NN_RESULT_THROW_UNLESS(
            SendSystemShareClearPacket(socket, reason) > 0,
            ResultSendError());
        NN_RESULT_SUCCESS;
    }

    Result SendShareSystemClearRes(int socket, Response response) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(g_SendMutex);

        SystemDeliveryInfo sysInfo = {};

        NN_RESULT_THROW_UNLESS(
            SendSystemShareResPacket(socket, Type_ShareSystemClearRes, response, sysInfo) > 0,
            ResultSendError());
        NN_RESULT_SUCCESS;
    }

    Result SendShareSystemSendReq(
        int socket, TransferRole role, Ipv4Address address, uint32_t index, int channel) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(g_SendMutex);

        SystemDeliveryInfo sysInfo = {};

        NN_RESULT_THROW_UNLESS(
            SendSystemShareInfoPacket(socket, Type_ShareSystemSendReq, role, index, address, sysInfo, channel) > 0,
            ResultSendError());
        NN_RESULT_SUCCESS;
    }

    Result SendShareSystemSendRes(int socket, const SystemDeliveryInfo& sysInfo) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(g_SendMutex);

        NN_RESULT_THROW_UNLESS(
            SendSystemShareResPacket(socket, Type_ShareSystemSendRes, Response_Accept, sysInfo) > 0,
            ResultSendError());
        NN_RESULT_SUCCESS;
    }

    Result SendShareSystemRecvReq(int socket, TransferRole role, Ipv4Address address, uint32_t index,
        const SystemDeliveryInfo& sysInfo, int channel) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(g_SendMutex);

        NN_RESULT_THROW_UNLESS(
            SendSystemShareInfoPacket(socket, Type_ShareSystemRecvReq, role, index, address, sysInfo, channel) > 0,
            ResultSendError());
        NN_RESULT_SUCCESS;
    }

    Result SendShareSystemRecvRes(int socket, Response response) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(g_SendMutex);

        SystemDeliveryInfo sysInfo = {};

        NN_RESULT_THROW_UNLESS(
            SendSystemShareResPacket(socket, Type_ShareSystemRecvRes, response, sysInfo) > 0,
            ResultSendError());
        NN_RESULT_SUCCESS;
    }

    Result SendShareContentClear(int socket,
        nn::util::BitFlagSet<8, ContentsType> contentsFlag, Bit64 id) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(g_SendMutex);

        NN_RESULT_THROW_UNLESS(
            SendContentShareClearPacket(socket, contentsFlag, id) > 0,
            ResultSendError());
        NN_RESULT_SUCCESS;
    }

    Result SendShareContentClearRes(int socket, Response response) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(g_SendMutex);

        NN_RESULT_THROW_UNLESS(
            SendContentShareResPacket(socket, Type_ShareContentClearRes, response, 0, 0) > 0,
            ResultSendError());
        NN_RESULT_SUCCESS;
    }

    Result SendShareContentSendReq(
        int socket, TransferRole role, Ipv4Address address, uint32_t index, Bit64 id, int channel) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(g_SendMutex);

        NN_RESULT_THROW_UNLESS(
            SendContentShareInfoPacket(socket, Type_ShareContentSendReq, role, index, address, 0, id, channel, 0) > 0,
            ResultSendError());
        NN_RESULT_SUCCESS;
    }

    Result SendShareContentSendRes(int socket, ContentMetaKey* metaKey, int metaKeyCount) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(g_SendMutex);

        size_t packetSize = metaKeyCount * ContentMetaKeySize;

        NN_RESULT_THROW_UNLESS(
            SendContentShareResPacket(socket, Type_ShareContentSendRes, Response_Accept, metaKeyCount, packetSize) > 0,
            ResultSendError());

        for (int i = 0; i < metaKeyCount; ++i)
        {
            NN_RESULT_THROW_UNLESS(
                SendContentMetaKey(socket, metaKey[i]) > 0,
                ResultSendError());
        }
        NN_RESULT_SUCCESS;
    }

    Result SendShareContentRecvReq(int socket, TransferRole role, Ipv4Address address, uint32_t index,
        Bit64 id, ContentMetaKey* metaKey, int metaKeyCount, int channel) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(g_SendMutex);

        size_t packetSize = metaKeyCount * ContentMetaKeySize;

        NN_RESULT_THROW_UNLESS(
            SendContentShareInfoPacket(socket, Type_ShareContentRecvReq, role, index, address, metaKeyCount, id, channel, packetSize) > 0,
            ResultSendError());

        for (int i = 0; i < metaKeyCount; ++i)
        {
            NN_RESULT_THROW_UNLESS(
                SendContentMetaKey(socket, metaKey[i]) > 0,
                ResultSendError());
        }
        NN_RESULT_SUCCESS;
    }

    Result SendShareContentRecvRes(int socket, Response response) NN_NOEXCEPT
    {
        ::std::lock_guard<nn::os::Mutex> guard(g_SendMutex);

        NN_RESULT_THROW_UNLESS(
            SendContentShareResPacket(socket, Type_ShareContentRecvRes, response, 0, 0) > 0,
            ResultSendError());
        NN_RESULT_SUCCESS;
    }

    Result WaitJoinSessionRes(uint32_t* pOutIndex, int socket, void* packetBuffer, size_t packetBufferSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutIndex);
        NN_SDK_ASSERT_NOT_NULL(packetBuffer);
        NN_SDK_ASSERT_GREATER_EQUAL(packetBufferSize, PacketSize);

        // タイムアウトタイマーセット
        nn::os::TimerEventType timeoutEvent;
        nn::os::InitializeTimerEvent(&timeoutEvent, nn::os::EventClearMode_AutoClear);
        nn::os::ClearTimerEvent(&timeoutEvent);
        nn::TimeSpan interval = nn::TimeSpan::FromMilliSeconds(TimeoutMilliSeconds);
        nn::os::StartOneShotTimerEvent(&timeoutEvent, interval);

        Result result = ResultCommunicationError();
        while (!nn::os::TryWaitTimerEvent(&timeoutEvent))
        {
            CommonHeader header = {};
            ssize_t recvSize = RecvCommonHeaderWithPeek(socket, &header);
            if(recvSize > 0){
                if (header.type == Type_Accept)
                {
                    NN_LCS_LOG_DEBUG("Recv Accept\n");
                    AcceptPacket packet = {};
                    RecvData(socket, &packet, sizeof(AcceptPacket), sizeof(AcceptPacket));
                    *pOutIndex = nn::socket::InetNtohl(packet.index);
                    result = ResultSuccess();
                    break;
                }
                else if (header.type == Type_Reject)
                {
                    NN_LCS_LOG_DEBUG("Recv Reject\n");
                    RejectPacket packet = {};
                    RecvData(socket, &packet, sizeof(RejectPacket), sizeof(RejectPacket));
                    if (packet.rejectReason == RejectReason_ConnectionFailed)
                    {
                        NN_LCS_LOG_DEBUG("RejectReason_ConnectionFailed\n");
                        result = ResultConnectionFailed();
                    }
                    else if (packet.rejectReason == RejectReason_CommunicationError)
                    {
                        NN_LCS_LOG_DEBUG("RejectReason_CommunicationError\n");
                        result = ResultCommunicationError();
                    }
                    else if (packet.rejectReason == RejectReason_LowerSystemVersion)
                    {
                        NN_LCS_LOG_DEBUG("RejectReason_LowerSystemVersion\n");
                        result = ResultLowerSystemVersion();
                    }
                    else if (packet.rejectReason == RejectReason_HigherSystemVersion)
                    {
                        NN_LCS_LOG_DEBUG("RejectReason_HigherSystemVersion\n");
                        result = ResultHigherSystemVersion();
                    }
                    else if (packet.rejectReason == RejectReason_IncompatibleSystemVersion)
                    {
                        NN_LCS_LOG_DEBUG("RejectReason_IncompatibleSystemVersion\n");
                        result = ResultIncompatibleSystemVersion();
                    }
                    break;
                }
            }
            else if (recvSize == 0)
            {
                NN_LCS_LOG_DEBUG("Recv 0 Error\n");
                break;
            }
        }
        nn::os::FinalizeTimerEvent(&timeoutEvent);
        return result;
    }

    Result WaitRecvData(int socket, void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(buffer);

        // タイムアウトタイマーセット
        nn::os::TimerEventType timeoutEvent;
        nn::os::InitializeTimerEvent(&timeoutEvent, nn::os::EventClearMode_AutoClear);
        nn::os::ClearTimerEvent(&timeoutEvent);
        nn::TimeSpan interval = nn::TimeSpan::FromMilliSeconds(TimeoutMilliSeconds);
        nn::os::StartOneShotTimerEvent(&timeoutEvent, interval);

        Result result = ResultCommunicationError();
        while (!nn::os::TryWaitTimerEvent(&timeoutEvent))
        {
            ssize_t recvSize = RecvData(socket, buffer, bufferSize, bufferSize);
            if (recvSize == static_cast<ssize_t>(bufferSize))
            {
                //NN_LCS_LOG_DEBUG("recvSize : %d\n", recvSize);
                result = ResultSuccess();
                break;
            }
            else if (recvSize == 0)
            {
                NN_LCS_LOG_DEBUG("Recv 0 Error\n");
                break;
            }
        }
        nn::os::FinalizeTimerEvent(&timeoutEvent);
        return result;
    }

}}} // end of namespace nn::lcs
