﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/lcs/detail/Advertise/lcs_AdvertiseUtility.h>
#include <nn/lcs/detail/Debug/lcs_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Endian.h>

namespace nn { namespace lcs { namespace detail
{

    bool ConvertAdvertiseDataToSessionInfo(
        SessionInfo* pOutInfo, const AdvertiseData& data) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutInfo);

        // ホスト名の末尾が必ず 0 終端されるようにします
        ::std::strncpy(pOutInfo->hostUserName, data.hostUserName, UserNameBytesMax);
        pOutInfo->hostUserName[UserNameBytesMax] = 0;

        // コンテンツ情報に不正な値が含まれていないことを検証しつつコピーします。
        if (SharableContentsCountMax < data.contentCount)
        {
            NN_LCS_LOG_WARN("Failed to convert advertise data: contents count = %u\n",
                data.contentCount);
            return false;
        }
        pOutInfo->contentsCount = data.contentCount;
        for (int j = 0; j < data.contentCount; ++j)
        {
            pOutInfo->contents[j].applicationId.value =
                nn::util::LoadBigEndian(&data.contents[j].id);
            pOutInfo->contents[j].attributeFlag =
                nn::util::LoadBigEndian(&data.contents[j].attributeFlag);
            ::memcpy(pOutInfo->contents[j].displayVersion,
                data.contents[j].displayVersion, DisplayVersionSizeMax);

            pOutInfo->contents[j].contentsFlag.Reset();
            if ((data.contents[j].contentsFlag & (0x01 << ContentsType::System::Index)) > 0)
            {
                pOutInfo->contents[j].contentsFlag.Set<ContentsType::System>(true);
            }
            if ((data.contents[j].contentsFlag & (0x01 << ContentsType::Application::Index)) > 0)
            {
                pOutInfo->contents[j].contentsFlag.Set<ContentsType::Application>(true);
            }
            if ((data.contents[j].contentsFlag & (0x01 << ContentsType::Patch::Index)) > 0)
            {
                pOutInfo->contents[j].contentsFlag.Set<ContentsType::Patch>(true);
            }
        }

        // 現在接続中の端末数に不正がないことを確認しつつコピーします。
        if (data.nodeCountMax == 0U || NodeCountMax < data.nodeCountMax)
        {
            NN_LCS_LOG_WARN("Failed to convert advertise data: maximum node count = %u\n",
                data.nodeCountMax);
            return false;
        }
        else if (data.nodeCount == 0 || data.nodeCountMax < data.nodeCount)
        {
            NN_LCS_LOG_WARN("Failed to convert advertise data: node count = %u/%u\n",
                data.nodeCount, data.nodeCountMax);
            return false;
        }
        pOutInfo->nodeCount = data.nodeCount;
        pOutInfo->nodeCountMax = data.nodeCountMax;
        return true;
    }

}}} // end of namespace nn::lcs
