﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_Keyboard.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/kpr/kpr_KeyCode.h>
#include <nn/kpr/detail/kpr_KeyCodeMap.h>

#include "kpr_KeyCodeImpl.h"

namespace nn { namespace kpr { namespace detail {

namespace {

//!< 印字可能な文字コードか否かを表す値を返します。
bool IsPrintable(uint16_t code) NN_NOEXCEPT;

//!< 対応する ASCII コードが存在する文字コードか否かを表す値を返します。
bool IsAsciiConvertible(uint16_t code) NN_NOEXCEPT;

//!< Alt Gr キーが押下中か否かを表す値を返します。
bool IsAltGrKeyOn(
    ::nn::hid::KeyboardModifierSet modifiers,
    KeyCodeMapAttributeSet attributes) NN_NOEXCEPT;

//!< 文字コード変換表から文字コードの候補を取得します。
const uint16_t* GetKeyCodes(
    int usageId,
    uint32_t mapCount, uint32_t keyCount, const uint16_t data[]) NN_NOEXCEPT;

//!< 文字コードの候補から文字コードの候補を取得します。
uint16_t GetKeyCode(
    ::nn::hid::KeyboardModifierSet modifiers,
    const uint16_t codes[],
    bool isShiftable, KeyCodeMapLockKey lockKey) NN_NOEXCEPT;

//!< 文字コードの候補から文字コードの候補を取得します。
uint16_t GetKeyCode(
    ::nn::hid::KeyboardModifierSet modifiers,
    KeyCodeMapAttributeSet attributes, uint32_t altGrMapOffset,
    const uint16_t codes[], KeyCodeMapInternalMode mode) NN_NOEXCEPT;

} // namespace

bool IsKeyCodeMapModeSupported(
    const KeyCodeMapStorage& map, KeyCodeMapInternalMode mode) NN_NOEXCEPT
{
    switch (mode)
    {
    case KeyCodeMapInternalMode::Default:
        return map.attributes.Test<KeyCodeMapAttribute::Default>();
    case KeyCodeMapInternalMode::Hiragana:
        return map.attributes.Test<KeyCodeMapAttribute::Hiragana>();
    case KeyCodeMapInternalMode::Katakana:
        return map.attributes.Test<KeyCodeMapAttribute::Katakana>();
    case KeyCodeMapInternalMode::Cyrillic:
        return map.attributes.Test<KeyCodeMapAttribute::Cyrillic>();
    case KeyCodeMapInternalMode::Hangul:
        return map.attributes.Test<KeyCodeMapAttribute::Hangul>();
    case KeyCodeMapInternalMode::Zhuyin:
        return map.attributes.Test<KeyCodeMapAttribute::Zhuyin>();
    case KeyCodeMapInternalMode::Cangjie:
        return map.attributes.Test<KeyCodeMapAttribute::Cangjie>();
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void GetKeyCode(
    KeyCode* pOutValue,
    int usageId, ::nn::hid::KeyboardModifierSet modifiers,
    const KeyCodeMapStorage& map, KeyCodeMapInternalMode mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    KeyCode& outValue = *pOutValue;

    outValue = KeyCode();

    const uint16_t* const codes =
        GetKeyCodes(usageId, map.mapCount, map.keyCount, map.data);

    if (codes == nullptr)
    {
        return;
    }

    outValue.code = GetKeyCode(
        modifiers, map.attributes, map.altGrMapOffset, codes, mode);;
    outValue.attributes.Set<KeyCodeAttribute::IsPrintable>(
        IsPrintable(outValue.code));
    outValue.attributes.Set<KeyCodeAttribute::IsAsciiConvertible>(
        IsAsciiConvertible(outValue.code));
}

namespace {

bool IsPrintable(uint16_t code) NN_NOEXCEPT
{
    return !((code == 0x0000) || (0x0300 <= code && code <= 0x036F));
}

bool IsAsciiConvertible(uint16_t code) NN_NOEXCEPT
{
    return (0x20 <= code && code <= 0x7E);
}

bool IsAltGrKeyOn(
    ::nn::hid::KeyboardModifierSet modifiers,
    KeyCodeMapAttributeSet attributes) NN_NOEXCEPT
{
    if (!attributes.Test<KeyCodeMapAttribute::AltGr>())
    {
        return false;
    }
    else
    {
        if (modifiers.Test<::nn::hid::KeyboardModifier::Control>() &&
            modifiers.Test<::nn::hid::KeyboardModifier::LeftAlt>())
        {
            return true;
        }

        if (modifiers.Test<::nn::hid::KeyboardModifier::RightAlt>())
        {
            return true;
        }

        return false;
    }
}

const uint16_t* GetKeyCodes(
    int usageId,
    uint32_t mapCount, uint32_t keyCount, const uint16_t data[]) NN_NOEXCEPT
{
    const int index = usageId - ::nn::hid::KeyboardKey::A::Index;

    if (index < 0 || static_cast<int>(keyCount) <= index)
    {
        return nullptr;
    }
    else
    {
        return &data[mapCount * index];
    }
}

uint16_t GetKeyCode(
    ::nn::hid::KeyboardModifierSet modifiers,
    const uint16_t codes[],
    bool isShiftable, KeyCodeMapLockKey lockKey) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(codes);

    const uint16_t code0 = codes[0];
    const uint16_t code1 = !isShiftable ? 0u : codes[1];

    switch (lockKey)
    {
    case KeyCodeMapLockKey::CapsLock:
        return !modifiers.Test<::nn::hid::KeyboardModifier::CapsLock>()
            ? (!modifiers.Test<::nn::hid::KeyboardModifier::Shift>()
                   ? code0 : code1)
            : (!modifiers.Test<::nn::hid::KeyboardModifier::Shift>()
                   ? code1 : code0);

    case KeyCodeMapLockKey::NumLock:
        return !modifiers.Test<::nn::hid::KeyboardModifier::NumLock>()
            ? code0 : (!modifiers.Test<::nn::hid::KeyboardModifier::Shift>()
                           ? code1 : code0);

    default:
        return !modifiers.Test<::nn::hid::KeyboardModifier::Shift>()
            ? code0 : code1;
    }
}

uint16_t GetKeyCode(
    ::nn::hid::KeyboardModifierSet modifiers,
    KeyCodeMapAttributeSet attributes, uint32_t altGrMapOffset,
    const uint16_t codes[], KeyCodeMapInternalMode mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(codes);

    const KeyCodeMapLockInfo lockInfo = { codes[0] };

    const auto lockKey = static_cast<KeyCodeMapLockKey>(
        lockInfo.Get<KeyCodeMapLockInfoField::LockKey>());

    if (lockKey == KeyCodeMapLockKey::CapsLock &&
        IsAltGrKeyOn(modifiers, attributes))
    {
        return GetKeyCode(modifiers, &codes[altGrMapOffset],
                          attributes.Test<KeyCodeMapAttribute::AltGrShift>(),
                          lockInfo.Get<KeyCodeMapLockInfoField::AltGr>() ?
                              lockKey : KeyCodeMapLockKey::None);
    }

    if (mode == KeyCodeMapInternalMode::Hiragana &&
        attributes.Test<KeyCodeMapAttribute::Hiragana>())
    {
        return GetKeyCode(modifiers, &codes[3], true,
                          lockInfo.Get<KeyCodeMapLockInfoField::Hiragana>() ?
                              lockKey : KeyCodeMapLockKey::None);
    }

    if (mode == KeyCodeMapInternalMode::Katakana &&
        attributes.Test<KeyCodeMapAttribute::Katakana>())
    {
        return GetKeyCode(modifiers, &codes[5], true,
                          lockInfo.Get<KeyCodeMapLockInfoField::Katakana>() ?
                              lockKey : KeyCodeMapLockKey::None);
    }

    if (mode == KeyCodeMapInternalMode::Cyrillic &&
        attributes.Test<KeyCodeMapAttribute::Cyrillic>())
    {
        return GetKeyCode(modifiers, &codes[3], true,
                          lockInfo.Get<KeyCodeMapLockInfoField::Cyrillic>() ?
                              lockKey : KeyCodeMapLockKey::None);
    }

    if (mode == KeyCodeMapInternalMode::Hangul &&
        attributes.Test<KeyCodeMapAttribute::Hangul>())
    {
        return GetKeyCode(modifiers, &codes[3], true,
                          lockInfo.Get<KeyCodeMapLockInfoField::Hangul>() ?
                              lockKey : KeyCodeMapLockKey::None);
    }

    if (mode == KeyCodeMapInternalMode::Zhuyin &&
        attributes.Test<KeyCodeMapAttribute::Zhuyin>() &&
        !modifiers.Test<::nn::hid::KeyboardModifier::Shift>())
    {
        return GetKeyCode(modifiers, &codes[3], true,
                          lockInfo.Get<KeyCodeMapLockInfoField::Zhuyin>() ?
                              lockKey : KeyCodeMapLockKey::None);
    }

    if (mode == KeyCodeMapInternalMode::Cangjie &&
        attributes.Test<KeyCodeMapAttribute::Cangjie>() &&
        !modifiers.Test<::nn::hid::KeyboardModifier::Shift>())
    {
        return GetKeyCode(modifiers, &codes[5], true,
                          lockInfo.Get<KeyCodeMapLockInfoField::Cangjie>() ?
                              lockKey : KeyCodeMapLockKey::None);
    }

    return GetKeyCode(modifiers, &codes[1], true,
                      lockInfo.Get<KeyCodeMapLockInfoField::Basic>() ?
                          lockKey : KeyCodeMapLockKey::None);
}

} // namespace

}}} // namespace nn::kpr::detail
