﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/kpr/kpr_KeyCodeComposer.h>

namespace nn { namespace kpr { namespace detail {

//!< キーコードコンポーザの内部状態を表す構造体です。
struct KeyCodeComposerStorage
{
    uint16_t queue[KeyCodeCompositionCountMax]; //!< 組み立て処理キュー
    int16_t _padding;
    int32_t mode;           //!< 組み立てモード
    int32_t nativeCount;    //!< 組み立て中キーコード列の未確定部分の長さ
    int32_t outputCount;    //!< 組み立て中キーコード列の確定部分の長さ
    int64_t _reserved;
};

//!< キーコードコンポーザを初期化します。
void InitializeKeyCodeComposer(KeyCodeComposerStorage* pComposer) NN_NOEXCEPT;

//!< キーコードコンポーザの文字コードの組み立てモードを返します。
KeyCodeComposerMode GetKeyCodeComposerMode(
    const KeyCodeComposerStorage& composer) NN_NOEXCEPT;

//!< キーコードコンポーザを文字コードの組み立てモードを設定します。
void SetKeyCodeComposerMode(
    KeyCodeComposerStorage* pComposer, KeyCodeComposerMode mode) NN_NOEXCEPT;

//!< キーコードコンポーザの組み立て中文字コード列の長さを返します。
int GetKeyCodeComposerCompositionCount(
    const KeyCodeComposerStorage& composer) NN_NOEXCEPT;

//!< キーコードコンポーザの組み立て中文字コード列を取得します。
int GetKeyCodeComposerComposition(
    const KeyCodeComposerStorage& composer, uint16_t outBuffer[], int count
    ) NN_NOEXCEPT;

//!< キーコードコンポーザの組み立て中文字コード列の末尾に文字コードを追加します。
int EnqueueKeyCodeComposer(
    KeyCodeComposerStorage* pComposer, const uint16_t buffer[], int count
    ) NN_NOEXCEPT;

//!< キーコードコンポーザの組み立て中文字コード列の確定部分の長さを返します。
int GetKeyCodeComposerDequeueableCount(
    const KeyCodeComposerStorage& composer) NN_NOEXCEPT;

//!< キーコードコンポーザの組み立て中文字コード列から確定部分を取り出します。
int DequeueKeyCodeComposer(
    KeyCodeComposerStorage* pComposer, uint16_t outBuffer[], int count
    ) NN_NOEXCEPT;

//!< キーコードコンポーザの組み立て中文字コード列から未確定部分を取り出します。
int RemoveKeyCodeComposer(
    KeyCodeComposerStorage* pComposer, uint16_t outBuffer[], int count
    ) NN_NOEXCEPT;

//!< キーコードコンポーザの組み立て中文字コード列の未確定部分全てを明示的に確定させます。
void FlushKeyCodeComposer(KeyCodeComposerStorage* pComposer) NN_NOEXCEPT;

//!< キーコードコンポーザの組み立て中文字コード列を破棄します。
void ClearKeyCodeComposer(KeyCodeComposerStorage* pComposer) NN_NOEXCEPT;

}}} // namespace nn::kpr::detail
