﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/irsensor/irsensor_MarkerPlaneEstimation.h>
#include <../source/OteteLib/src/OteteLib/Rodin.h>
#include <../external/core/source/picore/nerd_assert.h>
#include <../external/core/source/numerical/nerd_numerical.h>


namespace nn { namespace irsensor {
    int GetPlaneId(nn::irsensor::MarkerPlaneHandle handle) NN_NOEXCEPT;
}}

namespace nerd {
namespace otete {
namespace rodin {

int const  RODIN_DRIVER_MAGIC_NUMBER = 0xAAAAAAAA;

const size_t RodinDriverMemorySize = 32 * 1024;
const size_t RodinUpdateWorkMemorySize = 16 * 1024;

struct Point
{
    float v[2];
    float area;
    bool isIncomplete;
};

struct PointList
{
    unsigned char count;
    Point p[64];
};

struct Pose
{
    float m[4][3];
};

struct RodinDriver
{
    RodinDriver(u8* initMemory, size_t initMemorySize, const PointList& pointList);

    RodinAllocator m_initAllocator;
    RodinReference m_staticReference;
    RodinStaticReferenceNormalizer m_normalizer;
    RodinStaticReferenceSeeker m_referenceSeeker;
    RodinPoseProcessor m_poseProcessor;
};

struct RodinDriverHandler
{
    unsigned char* initMemory;
    size_t initMemorySize;

    unsigned char* GetDriverInitWorkMemory();
    unsigned char* GetDriverUpdateWorkMemory();

    RodinDriver* GetDriver();
};

void GenerateDefaultStaticReference(RodinReference& ref){
    ref.points.EmplaceBackAssumeEnoughRoom(nerd::float2(-81.5, -47.f));
    ref.points.EmplaceBackAssumeEnoughRoom(nerd::float2(81.5, -47.f));
    ref.points.EmplaceBackAssumeEnoughRoom(nerd::float2(-81.5, 47.f));
    ref.points.EmplaceBackAssumeEnoughRoom(nerd::float2(81.5, 47.f));

    ref.points.EmplaceBackAssumeEnoughRoom(nerd::float2(0, -47.f));
};

void GenerateStaticReference(RodinReference& ref, const PointList& pointList)
{
    forrange0(i, pointList.count)
    {
        ref.points.EmplaceBackAssumeEnoughRoom(nerd::float2(pointList.p[i].v[0], pointList.p[i].v[1]));
    }
};

RodinDriver::RodinDriver(u8* initMemory, size_t initMemorySize, const PointList& pointList)
    : m_initAllocator(initMemory, initMemorySize)
    , m_staticReference([pointList](RodinReference& ref) {GenerateStaticReference(ref, pointList); } , &m_initAllocator)
    , m_normalizer(m_staticReference, 5.f, &m_initAllocator) // 20mm tolerance
    , m_referenceSeeker(m_staticReference, &m_initAllocator)
    , m_poseProcessor(&m_referenceSeeker, &m_normalizer, &m_initAllocator, 0.0008f)
{
    m_poseProcessor.ConfigureBorderWeightDecay(true, float2(0.9f,0.75f));
}

static RodinDriverHandler s_RodinDriverHandler[nn::irsensor::MarkerPlaneCountMax];

RodinDriverHandler* GetRodinDriverHandler(nn::irsensor::MarkerPlaneHandle handle)
{
    auto planeId = nn::irsensor::GetPlaneId(handle);
    NN_SDK_ASSERT(planeId >= 0);
    return &s_RodinDriverHandler[planeId];
}

bool RodinDriverIsInit(nn::irsensor::MarkerPlaneHandle handle)
{
    if(GetRodinDriverHandler(handle)->initMemory == 0x0)
    {
        return false;
    }

    if(GetRodinDriverHandler(handle)->initMemorySize == 0)
    {
        return false;
    }

    int* magicNumber = (int*) GetRodinDriverHandler(handle)->initMemory;

    if(*magicNumber != RODIN_DRIVER_MAGIC_NUMBER)
    {
        return false;
    }

    return true;
}

RodinDriver* RodinDriverHandler::GetDriver()
{
    RodinDriver* driver = reinterpret_cast<RodinDriver*>(initMemory + sizeof(int));
    return driver;
}

unsigned char* RodinDriverHandler::GetDriverInitWorkMemory()
{
    return initMemory + sizeof(int) + sizeof(nerd::otete::rodin::RodinDriver);
}

unsigned char* RodinDriverHandler::GetDriverUpdateWorkMemory()
{
    return initMemory + nerd::otete::rodin::RodinDriverMemorySize;
}

}}}

namespace nn { namespace irsensor {

::std::atomic<int> s_Count(0);
static nn::irsensor::MarkerPlaneHandle s_StaticRodinHandleTable[nn::irsensor::MarkerPlaneCountMax];

void CreateMarkerPlaneHandle(
    MarkerPlaneHandle* outValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(s_Count, 0, nn::irsensor::MarkerPlaneCountMax);
    outValue->_handle = s_Count;
    s_StaticRodinHandleTable[s_Count] = *outValue;
    s_Count++;
}

void DstroyMarkerPlaneHandle(
    MarkerPlaneHandle handle) NN_NOEXCEPT
{
    forrange0(i, nn::irsensor::MarkerPlaneCountMax)
    {
        if (s_StaticRodinHandleTable[i]._handle == handle._handle)
        {
            s_StaticRodinHandleTable[i]._handle = 0;
            s_Count--;
            break;
        }
    }
}

int GetPlaneId(nn::irsensor::MarkerPlaneHandle handle) NN_NOEXCEPT
{
    forrange0(i, nn::irsensor::MarkerPlaneCountMax)
    {
        if (s_StaticRodinHandleTable[i]._handle == handle._handle)
        {
            return static_cast<int>(i);
        }
    }
    return -1;
}

MarkerPlaneHandle MarkerPlaneEstimationInitialize(unsigned char* initMemory, size_t initMemorySize, const MarkerPlaneConfig& config) NN_NOEXCEPT
{
    MarkerPlaneHandle handle = {};
    CreateMarkerPlaneHandle(&handle);

    NN_SDK_ASSERT(!nerd::otete::rodin::RodinDriverIsInit(handle));
    NN_SDK_ASSERT(initMemorySize > 0);

    // TODO: impl  dynamic mode
    NN_SDK_ASSERT(config.type == nn::irsensor::MarkerPlaneReferenceType_Static);

    int* magicNumber = reinterpret_cast<int*>(initMemory);
    *magicNumber = nerd::otete::rodin::RODIN_DRIVER_MAGIC_NUMBER;

    NN_SDK_ASSERT(config.pointList.count < 1 || config.pointList.count >= 5);

    nerd::otete::rodin::GetRodinDriverHandler(handle)->initMemory = initMemory;
    nerd::otete::rodin::GetRodinDriverHandler(handle)->initMemorySize = initMemorySize;

    uint8_t* driverAddress = initMemory + sizeof(int);
    size_t memorySize = nerd::otete::rodin::RodinDriverMemorySize - (sizeof(nerd::otete::rodin::RodinDriver));

    nerd::otete::rodin::PointList pointList = {};
    pointList.count = config.pointList.count;
    forrange0(i, pointList.count)
    {
        pointList.p[i].area = config.pointList.p[i].area;
        pointList.p[i].isIncomplete = config.pointList.p[i].isIncomplete;
        pointList.p[i].v[0] = config.pointList.p[i].v[0];
        pointList.p[i].v[1] = config.pointList.p[i].v[1];
    }
    auto driver = new (driverAddress) nerd::otete::rodin::RodinDriver(
        (nerd::u8*)nerd::otete::rodin::GetRodinDriverHandler(handle)->GetDriverInitWorkMemory(), memorySize, pointList);
    NN_SDK_ASSERT(driver == nerd::otete::rodin::GetRodinDriverHandler(handle)->GetDriver());
    NN_SDK_ASSERT(driverAddress == reinterpret_cast<uint8_t*>(driver));
    NN_UNUSED(driver);

    NN_SDK_ASSERT(nerd::otete::rodin::RodinDriverIsInit(handle));
    return handle;
}

void MarkerPlaneEstimationFinalize(MarkerPlaneHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(nerd::otete::rodin::RodinDriverIsInit(handle));

    nerd::otete::rodin::GetRodinDriverHandler(handle)->initMemorySize = 0;
    nerd::otete::rodin::GetRodinDriverHandler(handle)->initMemory = 0x0;

    NN_SDK_ASSERT(!nerd::otete::rodin::RodinDriverIsInit(handle));
    DstroyMarkerPlaneHandle(handle);
}

::nn::Result UpdateMarkerPlaneEstimation(const PointList& points, MarkerPlaneHandle handle) NN_NOEXCEPT
{
    nerd::otete::rodin::RodinPoseProcessor& poseProcessor = nerd::otete::rodin::GetRodinDriverHandler(handle)->GetDriver()->m_poseProcessor;
    nerd::otete::rodin::RodinAllocator allocator(nerd::otete::rodin::GetRodinDriverHandler(handle)->GetDriverUpdateWorkMemory(), nerd::otete::rodin::RodinUpdateWorkMemorySize);
    nerd::otete::rodin::RodinPointContainer<nerd::otete::rodin::RodinPoint> pointsList(&allocator);

    forrange0(i, points.count)
    {
        Point const& p = points.p[i];
        pointsList.EmplaceBack(nerd::float2(p.v[0], p.v[1]), p.area, p.isIncomplete);
    }
    poseProcessor.Feed(pointsList, &allocator);
    return nn::ResultSuccess();
}

::nn::Result GetMarkerPlaneState(MarkerPlaneState* pState, MarkerPlaneHandle handle) NN_NOEXCEPT
{
    nerd::otete::rodin::RodinPoseProcessor& poseProcessor = nerd::otete::rodin::GetRodinDriverHandler(handle)->GetDriver()->m_poseProcessor;

    // Get confidence
    if (poseProcessor.HasValidPose())
    {
        pState->confidence = 1.0f;
    }
    else
    {
        pState->confidence = 0.0f;
    }

    // Get transform
    const nerd::otete::rodin::RodinPose& pose = poseProcessor.GetPose();

    for (size_t x = 0; x < 4; ++x)
            for (size_t y = 0; y < 3; ++y)
                pState->m[x][y] = pose.transform._[y][x];

    return nn::ResultSuccess();
}

::nn::Result GetBaseReference(PointList* pBaseReference, MarkerPlaneHandle handle) NN_NOEXCEPT
{
    nerd::otete::rodin::RodinReference const& reference = nerd::otete::rodin::GetRodinDriverHandler(handle)->GetDriver()->m_staticReference;

    pBaseReference->count = (uint8_t) reference.points.Size();
    forrange0(i, pBaseReference->count)
    {
        Point& p = pBaseReference->p[i];
        nerd::otete::rodin::RodinPoint const& point = reference.points[i];
        p.v[0] = point.pos.x();
        p.v[1] = point.pos.y();
    }
    return nn::ResultSuccess();
}

bool GetCurrentReference(PointList* pCurrentReference, MarkerPlaneHandle handle) NN_NOEXCEPT
{
    auto driver = nerd::otete::rodin::GetRodinDriverHandler(handle)->GetDriver();
    if (driver->m_referenceSeeker.HasReference())
    {
        nerd::otete::rodin::RodinReference const& reference = driver->m_poseProcessor.GetReference();
        pCurrentReference->count = (uint8_t)reference.points.Size();
        forrange0(i, pCurrentReference->count)
        {
            Point& p = pCurrentReference->p[i];
            nerd::otete::rodin::RodinPoint const& point = reference.points[i];
            p.v[0] = point.pos.x();
            p.v[1] = point.pos.y();
        }
        return true;
    }
    else
    {
        return false;
    }
}

} }
