﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <type_traits>

#include <gillette_Api.h>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/irsensor/irsensor_HandAnalysis.h>
#include <nn/irsensor/irsensor_HandAnalysisTypesPrivate.h>
#include <nn/irsensor/irsensor_HandAnalysisApiPrivate.h>
#include <nn/irsensor/irsensor_TeraPluginProcessorApiPrivate.h>
#include <nn/irsensor/irsensor_TeraPluginProcessorTypesPrivate.h>
#include <nn/irsensor/irsensor_ResultPrivate.h>
#include <nn/irsensor/irsensor_Result.public.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/result/result_ErrorResult.h>

#include <nn/util/util_Arithmetic.h>

namespace nerd {

// Note(nerd_marc.delorme): OteteHostLib need an assert callback to be implemented.
// return if we want the lib to debug break. I have chosen to always return false
// and let the NintendoSDK decide if we need to debug or not.
bool OnAssertionFailed(bool* o_bIgnored, const char* a_pFile, int a_iLine, const char* a_pCondition, const char* a_sMessage, ...)
{
    NN_UNUSED(o_bIgnored);

    if (a_sMessage != nullptr)
    {
        va_list args;
        va_start(args, a_sMessage);

        ::nn::diag::detail::OnAssertionFailure(
            ::nn::diag::AssertionType_SdkAssert,
            a_pCondition,
            "<Function Unknown>",
            a_pFile,
            a_iLine,
            a_sMessage,
            args);

        va_end(args);
    }
    else
    {
        ::nn::diag::detail::OnAssertionFailure(
            ::nn::diag::AssertionType_SdkAssert,
            a_pCondition,
            "<Function Unknown>",
            a_pFile,
            a_iLine);
    }

    return false;
}

}

namespace {

NN_STATIC_ASSERT(sizeof(nn::util::Float2) == sizeof(::nerd::otete::OteteFloat2));
NN_STATIC_ASSERT((std::is_same<decltype(nn::util::Float2::x), decltype(::nerd::otete::OteteFloat2::x)>::value));
NN_STATIC_ASSERT((std::is_same<decltype(nn::util::Float2::y), decltype(::nerd::otete::OteteFloat2::y)>::value));

}

namespace nn {
namespace irsensor {

namespace {

    template<typename T>
struct HandAnalysisStateTypeTraits;

template<>
struct HandAnalysisStateTypeTraits<HandAnalysisSilhouetteState>
{
    static const uint32_t prefix = HandAnalysisPrefixSilouette;
};

template<>
struct HandAnalysisStateTypeTraits<HandAnalysisSilhouetteStateWithFullFrameShape>
{
    static const uint32_t prefix = HandAnalysisPrefixSilouette;
};

template<>
struct HandAnalysisStateTypeTraits<HandAnalysisImageState>
{
    static const uint32_t prefix = HandAnalysisPrefixImage;
};

template<typename T>
struct OteteModeTraits;

template<>
struct OteteModeTraits<HandAnalysisSilhouetteState>
{
    static const ::nerd::otete::Mode mode = ::nerd::otete::Mode_Silhouette;
};

template<>
struct OteteModeTraits<HandAnalysisImageState>
{
    static const ::nerd::otete::Mode mode = ::nerd::otete::Mode_Image;
};

inline void AngloidToVector(nn::util::Float2* xy, float angloid)
{
    NN_SDK_ASSERT(-1.f <= angloid || angloid <= 1.f);
    float const signAngloid = angloid < 0.f ? -1.f : 1.f;
    float const absAngloid = std::abs(angloid);
    if (absAngloid < 0.25f)
    {
        xy->x = 1.f;
        xy->y = 4.f * angloid;
        return;
    }
    if (absAngloid < 0.75f)
    {
        xy->x = 2.f - 4.f * absAngloid;
        xy->y = signAngloid;
        return;
    }

    xy->x = -1.f;
    xy->y = 4.f * (signAngloid - angloid);
    return;
}

::nn::Result DecodeHandAnalysisState(HandAnalysisSilhouetteState* pOutSilhouette, HandAnalysisFullFrameShape* pOutFullFrame, const TeraPluginProcessorState& handAnalysisState, size_t* pReadCursor, const unsigned char* pPacketData)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSilhouette);

    bool ok = true;
    ::nerd::otete::Mode mode;
    ::nerd::otete::jaimev2::IntensityInPacket intensity;

    const size_t packetSize = sizeof(((TeraPluginProcessorState*)(0))->data);
    size_t& readCursor = *pReadCursor;

    pOutSilhouette->samplingNumber = handAnalysisState.samplingNumber;
    pOutSilhouette->ambientNoiseLevel = handAnalysisState.ambientNoiseLevel;

    {
        ok = ::nerd::otete::jaimev2::DecodePacketModeROK(mode, pPacketData, packetSize, &readCursor);
        if (!ok)
        {
            return ResultHandAnalysisPacketDecodingFailed();
        }
    }
    {
        ok = ::nerd::otete::jaimev2::DecodeIntensityROK(intensity, pPacketData, packetSize, &readCursor);
        if (!ok)
        {
            return ResultHandAnalysisPacketDecodingFailed();
        }
    }

    if (mode != OteteModeTraits<HandAnalysisSilhouetteState>::mode)
    {
        return ResultHandAnalysisModeIncorrect();
    }

    // Use temporary memory in user provided buffer
    NN_STATIC_ASSERT(sizeof(::nerd::otete::jaimev2::HandInPacket) <= sizeof(pOutSilhouette->points));
    ::nerd::otete::jaimev2::HandInPacket& handInPacket = *reinterpret_cast<::nerd::otete::jaimev2::HandInPacket*>(&pOutSilhouette->points[0]);

    pOutSilhouette->handCount = 0;

    for (int handIndex = 0;; ++handIndex)
    {
        bool hasNextHand = false;
        ok = nerd::otete::jaimev2::DecodeHasNextHandROK(hasNextHand, pPacketData, packetSize, &readCursor);
        if (!ok) {
            return ResultHandAnalysisPacketDecodingFailed();
        }
        if (!hasNextHand)
            break;

        ok = nerd::otete::jaimev2::DecodeHandROK(handInPacket, pPacketData, packetSize, &readCursor, intensity);
        if (!ok) {
            return ResultHandAnalysisPacketDecodingFailed();
        }

        if (handIndex >= nn::irsensor::IrHandAnalysisHandCountMax)
        {
            return ResultHandAnalysisPacketDecodingFailed();
        }

        ++pOutSilhouette->handCount;

        Hand& hand = pOutSilhouette->hands[handIndex];
        hand.shapeId = handInPacket.shapeId;
        hand.chirality = handInPacket.isRightHand ? HandChirality_Right : HandChirality_Left;
        hand.protrusionCount = handInPacket.protuberanceCount;
        for (int j = 0; j < handInPacket.protuberanceCount; ++j)
        {
            Protrusion& protrusion = hand.protrusions[j];
            protrusion.firstPointIndex = handInPacket.protuberances[j].firstPointIndex;
            protrusion.pointCount = handInPacket.protuberances[j].pointCount;
        }

        for (int j = 0; j < nerd::otete::jaimev2::HandInPacket::fingerCount; ++j)
        {
            Finger& finger = hand.fingers[j];
            finger.isValid = handInPacket.fingers[j].isValid;
            finger.tip.x = handInPacket.fingers[j].tipPosition.x;
            finger.tip.y = handInPacket.fingers[j].tipPosition.y;
            finger.root.x = handInPacket.fingers[j].rootPosition.x;
            finger.root.y = handInPacket.fingers[j].rootPosition.y;
            finger.protrusionIndex = handInPacket.fingers[j].protuberanceIndex;
            finger.tipDepthFactor = handInPacket.fingers[j].tipDepthFactor;
        }

        for (int j = 0; j < HandTouchingFingers_Count; ++j)
        {
            Finger& finger0 = hand.fingers[j + 1];
            Finger& finger1 = hand.fingers[j + 2];
            hand.areFingersTouching[j] = (finger0.isValid && finger1.isValid && finger0.protrusionIndex == finger1.protrusionIndex);
        }


        Palm& palm = hand.palm;
        palm.center.x = handInPacket.palmCentroid.x;
        palm.center.y = handInPacket.palmCentroid.y;
        palm.area = 3.1415f * sqrtf(handInPacket.palmRadius);
        palm.depthFactor = handInPacket.palmDepthFactor;

        Arm& arm = hand.arm;
        arm.isValid = handInPacket.arm.isValid;
        arm.wristPosition.x = handInPacket.arm.wristPosition.x;
        arm.wristPosition.y = handInPacket.arm.wristPosition.y;
        AngloidToVector(&arm.armDirection, handInPacket.arm.directionAngloid);
        arm.protrusionIndex = handInPacket.arm.protuberanceIndex;
    }

    pOutSilhouette->shapeCount = 0;
    int totalpointcount = 0;
    for (int shapeIndex = 0;; ++shapeIndex)
    {
        bool hasNextShape = false;
        ok = nerd::otete::jaimev2::DecodeHasNextShapeROK(hasNextShape, pPacketData, packetSize, &readCursor);
        if (!ok) {
            return ResultHandAnalysisPacketDecodingFailed();
        }
        if (!hasNextShape)
        {
            break;
        }

        nerd::otete::jaimev2::ShapeInPacket shape;
        size_t pointCount = 0;
        ok = nerd::otete::jaimev2::DecodeShapeAndSilhouetteROK(
            shape,
            reinterpret_cast<nerd::otete::OteteFloat2*>(pOutSilhouette->points) + totalpointcount,
            pointCount,
            IrHandAnalysisShapePointCountMax - totalpointcount, // Would like to use ARRAY_SIZE(points) instead of IrHandAnalyzerShapePointCountMax
            pPacketData,
            packetSize,
            &readCursor,
            intensity);
        if (!ok) {
            return ResultHandAnalysisPacketDecodingFailed();
        }

        NN_SDK_ASSERT(0 < pointCount);
        pOutSilhouette->shapes[shapeIndex].firstPointIndex = totalpointcount;
        pOutSilhouette->shapes[shapeIndex].pointCount = static_cast<int>(pointCount);
        pOutSilhouette->shapes[shapeIndex].intensityAverage = shape.intensityAverage;
        pOutSilhouette->shapes[shapeIndex].intensityCentroid.x = shape.intensityCentroid.x;
        pOutSilhouette->shapes[shapeIndex].intensityCentroid.y = shape.intensityCentroid.y;
        ++pOutSilhouette->shapeCount;
        totalpointcount += static_cast<int>(pointCount);
    }

    bool hasFullFrame = false;
    ok = nerd::otete::jaimev2::DecodeHasFullFrameShapeROK(hasFullFrame, pPacketData, packetSize, &readCursor);
    if (!ok) {
        return ResultHandAnalysisPacketDecodingFailed();
    }

    if (hasFullFrame)
    {
        nerd::otete::jaimev2::ShapeInPacket shape;
        size_t pointCount = 0;
        nerd::otete::jaimev2::FullFrameShapeInPacket fullFrameShape;

        ok = nerd::otete::jaimev2::DecodeFullFrameShapeROK(
            shape,
            reinterpret_cast<nerd::otete::OteteFloat2*>(pOutSilhouette->points + totalpointcount),
            pointCount,
            IrHandAnalysisShapePointCountMax - totalpointcount, // Would like to use ARRAY_SIZE(points) instead of IrHandAnalyzerShapePointCountMax
            fullFrameShape,
            pPacketData,
            packetSize,
            &readCursor,
            intensity);
        if (!ok) {
            return ResultHandAnalysisPacketDecodingFailed();
        }

        NN_SDK_ASSERT(0 < pointCount);
        int shapeIndex = 0;
        NN_SDK_ASSERT(pOutSilhouette->shapeCount == shapeIndex);
        pOutSilhouette->shapes[shapeIndex].firstPointIndex = totalpointcount;
        pOutSilhouette->shapes[shapeIndex].pointCount = static_cast<int>(pointCount);
        pOutSilhouette->shapes[shapeIndex].intensityAverage = shape.intensityAverage;
        pOutSilhouette->shapes[shapeIndex].intensityCentroid.x = shape.intensityCentroid.x;
        pOutSilhouette->shapes[shapeIndex].intensityCentroid.y = shape.intensityCentroid.y;
        ++pOutSilhouette->shapeCount;
        totalpointcount += static_cast<int>(pointCount);

        if (nullptr != pOutFullFrame)
        {
            pOutFullFrame->isFullFrame = true;
            pOutFullFrame->captureId = fullFrameShape.captureId;
            pOutFullFrame->intensitySamplingPeriod = fullFrameShape.intensitySamplingPeriod * 100;
            pOutFullFrame->intensitySamplingNumber = fullFrameShape.intensitySamplingNumber;
            pOutFullFrame->intensityHistoryCount = fullFrameShape.intensityHistorySize;
            for(int j = 0; j < IrHandAnalysisHistoryCapacity; j++)
            {
                pOutFullFrame->intensityHistory[j].relativeSamplingNumber = fullFrameShape.intensityHistory[j].relativeSamplingNumber;
                pOutFullFrame->intensityHistory[j].intensity = fullFrameShape.intensityHistory[j].intensity;
            }
        }
    }
    else
    {
        if (nullptr != pOutFullFrame)
        {
            pOutFullFrame->isFullFrame = false;
            pOutFullFrame->captureId = 0;
            pOutFullFrame->intensitySamplingPeriod = 0;
            pOutFullFrame->intensitySamplingNumber = 0;
            pOutFullFrame->intensityHistoryCount = 0;
            for (int j = 0; j < IrHandAnalysisHistoryCapacity; j++)
                pOutFullFrame->intensityHistory[j] = {};
        }
    }

    // fixup invalid ids (could come from invalid packet or bug in MCU)
    bool hasInvalidShapeBeenAdded = false;
    for(int handIndex = 0; handIndex < pOutSilhouette->handCount; handIndex++)
    {
        Hand& hand = pOutSilhouette->hands[handIndex];
        if (hand.shapeId > pOutSilhouette->shapeCount)
        {
            if (!hasInvalidShapeBeenAdded && pOutSilhouette->shapeCount < IrHandAnalysisShapeCountMax)
            {
                pOutSilhouette->shapes[pOutSilhouette->shapeCount].firstPointIndex = totalpointcount;
                pOutSilhouette->shapes[pOutSilhouette->shapeCount].pointCount = 0;
                pOutSilhouette->shapes[pOutSilhouette->shapeCount].intensityAverage = 0;
                pOutSilhouette->shapes[pOutSilhouette->shapeCount].intensityCentroid.x = 0.f;
                pOutSilhouette->shapes[pOutSilhouette->shapeCount].intensityCentroid.y = 0.f;
                hasInvalidShapeBeenAdded = true;
            }
            if (pOutSilhouette->shapeCount < IrHandAnalysisShapeCountMax)
                hand.shapeId = pOutSilhouette->shapeCount;
            else
                hand.shapeId = pOutSilhouette->shapeCount - 1;
        }
        bool hasInvalidProtrusionBeenAdded = false;
        for (int fingerIndex = 0; fingerIndex < HandFinger_Count; fingerIndex++)
        {
            if (!hand.fingers[fingerIndex].isValid)
                continue;
            if (hand.fingers[fingerIndex].protrusionIndex >= hand.protrusionCount)
            {
                if (!hasInvalidProtrusionBeenAdded && hand.protrusionCount < IrHandAnalysisProtrusionCountMax)
                {
                    hand.protrusions[hand.protrusionCount].firstPointIndex = 0;
                    hand.protrusions[hand.protrusionCount].pointCount = 0;
                    hasInvalidShapeBeenAdded = true;
                }
                if (hand.protrusionCount < IrHandAnalysisProtrusionCountMax)
                    hand.fingers[fingerIndex].protrusionIndex = hand.protrusionCount;
                else
                    hand.fingers[fingerIndex].protrusionIndex = hand.protrusionCount - 1;
            }
        }

        if (hand.arm.isValid)
        {
            if (hand.arm.protrusionIndex >= hand.protrusionCount)
            {
                if (!hasInvalidProtrusionBeenAdded && hand.protrusionCount < IrHandAnalysisProtrusionCountMax)
                {
                    hand.protrusions[hand.protrusionCount].firstPointIndex = 0;
                    hand.protrusions[hand.protrusionCount].pointCount = 0;
                    hasInvalidShapeBeenAdded = true;
                }
                if (hand.protrusionCount < IrHandAnalysisProtrusionCountMax)
                    hand.arm.protrusionIndex = hand.protrusionCount;
                else
                    hand.arm.protrusionIndex = hand.protrusionCount - 1;
            }
        }
    }

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

::nn::Result DecodeHandAnalysisState(HandAnalysisSilhouetteState* pOutValue, const TeraPluginProcessorState& handAnalysisState, size_t* pReadCursor, const unsigned char* pPacketData)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    DecodeHandAnalysisState(pOutValue, nullptr, handAnalysisState, pReadCursor, pPacketData);

    NN_RESULT_SUCCESS;
}

::nn::Result DecodeHandAnalysisState(HandAnalysisSilhouetteStateWithFullFrameShape* pOutValue, const TeraPluginProcessorState& handAnalysisState, size_t* pReadCursor, const unsigned char* pPacketData)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    DecodeHandAnalysisState(&pOutValue->silhouetteState, &pOutValue->fullframeState, handAnalysisState, pReadCursor, pPacketData);

    NN_RESULT_SUCCESS;
}


::nn::Result DecodeHandAnalysisState(HandAnalysisImageState* pOutValue, const TeraPluginProcessorState& handAnalysisState, size_t* pReadCursor, const unsigned char* pPacketData)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::nerd::otete::Mode mode;
    ::nerd::otete::jaimev2::IntensityInPacket intensity;

    const size_t packetSize = sizeof(((TeraPluginProcessorState*)(0))->data);
    size_t& readCursor = *pReadCursor;

    pOutValue->samplingNumber = handAnalysisState.samplingNumber;
    pOutValue->ambientNoiseLevel = handAnalysisState.ambientNoiseLevel;

    {
        bool ok = ::nerd::otete::jaimev2::DecodePacketModeROK(mode, pPacketData, packetSize, &readCursor);
        if (!ok)
        {
            return ResultHandAnalysisPacketDecodingFailed();
        }
    }
    {
        bool ok = ::nerd::otete::jaimev2::DecodeIntensityROK(intensity, pPacketData, packetSize, &readCursor);
        if (!ok)
        {
            return ResultHandAnalysisPacketDecodingFailed();
        }
    }

    if (mode != OteteModeTraits<HandAnalysisImageState>::mode)
    {
        return ResultHandAnalysisModeIncorrect();
    }

    {
        bool ok = ::nerd::otete::jaimev2::DecodeImageROK(pOutValue->image, pPacketData, packetSize, &readCursor, intensity);
        if (!ok)
        {
            return ResultHandAnalysisPacketDecodingFailed();
        }
    }

    NN_RESULT_SUCCESS;
}

template<typename StateType>
::nn::Result DecodeOneHandAnalysisState(StateType* pOutValue, const TeraPluginProcessorState& handAnalysisState)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    const unsigned char* pPacketData = handAnalysisState.data;
    size_t readCursor = 0;

    NN_RESULT_DO(DecodeHandAnalysisState(pOutValue, handAnalysisState, &readCursor, pPacketData));

    NN_RESULT_SUCCESS;
}

template<typename StateType>
::nn::Result DecodeAllHandAnalysisState(StateType* pOutValueArray, int *pReturnCount, int maxCount, int64_t infSamplingNumber, const IrCameraHandle& handle)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValueArray);
    NN_SDK_REQUIRES_NOT_NULL(pReturnCount);

    // NOTE(nerd_marc.delorme): I cannot allocate a dynamic array of HandAnalysisState and I do not want the
    // application developer to provide me some work memory. So I will use the StateType array memory as
    // working memory. If the StateType size is more than twice bigger than HandAnalysisState I am sure for
    // any index except the last one StateType[index] will not overlap with HandAnalysisState[index] if I put
    // the HandAnalysisState array at the end of the StateType buffer. Example:
    //
    // <------------ StateType ------------><------------ StateType ------------><------------ StateType ------------>
    //                                                             <- HndAnalysis -><- HndAnalysis -><- HndAnalysis ->
    //
    // I will just allocate one HandAnalysisState on the stack for the last element.
    NN_STATIC_ASSERT(sizeof(StateType) >= 2 * sizeof(TeraPluginProcessorState));

    const int outValueArraySize          = maxCount * sizeof(StateType);
    const int handAnalysisStateArraySize = maxCount * sizeof(TeraPluginProcessorState);

    TeraPluginProcessorState* pHandAnalysisState = reinterpret_cast<TeraPluginProcessorState*>(reinterpret_cast<char*>(pOutValueArray) + outValueArraySize - handAnalysisStateArraySize);

    NN_RESULT_DO(GetTeraPluginProcessorStates(pHandAnalysisState, pReturnCount, maxCount, infSamplingNumber, HandAnalysisStateTypeTraits<StateType>::prefix, HandAnalysisPrefixSize, handle));

    if (*pReturnCount > 0)
    {
        const int lastIndex   = *pReturnCount - 1;

        for (int i = 0; i < lastIndex; i++)
        {
            NN_RESULT_DO(DecodeOneHandAnalysisState(&pOutValueArray[i], pHandAnalysisState[i]));
        }

        {
            TeraPluginProcessorState handAnalysisState = pHandAnalysisState[lastIndex];
            NN_RESULT_DO(DecodeOneHandAnalysisState(&pOutValueArray[lastIndex], handAnalysisState));
        }
    }

    NN_RESULT_SUCCESS;
}

} // namespace

::nn::Result GetHandAnalysisSilhouetteStateWithFullFrameShape(HandAnalysisSilhouetteStateWithFullFrameShape* pOutValueArray, int *pReturnCount, int maxCount, int64_t infSamplingNumber, const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValueArray);
    NN_SDK_REQUIRES_NOT_NULL(pReturnCount);

    NN_RESULT_DO(DecodeAllHandAnalysisState(pOutValueArray, pReturnCount, maxCount, infSamplingNumber, handle));

    NN_RESULT_SUCCESS;
}

::nn::Result GetHandAnalysisSilhouetteState(HandAnalysisSilhouetteState* pOutValueArray, int *pReturnCount, int maxCount, int64_t infSamplingNumber, const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValueArray);
    NN_SDK_REQUIRES_NOT_NULL(pReturnCount);

    NN_RESULT_DO(DecodeAllHandAnalysisState(pOutValueArray, pReturnCount, maxCount, infSamplingNumber, handle));

    NN_RESULT_SUCCESS;
}

::nn::Result GetHandAnalysisImageState(HandAnalysisImageState* pOutValueArray, int *pReturnCount, int maxCount, int64_t infSamplingNumber, const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValueArray);
    NN_SDK_REQUIRES_NOT_NULL(pReturnCount);

    NN_RESULT_DO(DecodeAllHandAnalysisState(pOutValueArray, pReturnCount, maxCount, infSamplingNumber, handle));

    NN_RESULT_SUCCESS;
}

}} // namespace nn::irsensor
