﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/irsensor.h>
#include <nn/irsensor/irsensor_HandAnalysisApi.h>
#include <nn/irsensor/irsensor_HandAnalysisConstants.h>
#include <nn/irsensor/irsensor_HandAnalysisTypesPrivate.h>
#include <nn/irsensor/irsensor_PointingProcessorApiPrivate.h>
#include <nn/irsensor/irsensor_TeraPluginProcessorApiPrivate.h>
#include <nn/irsensor/irsensor_IrLedProcessorApi.h>
#include <nn/xcd/xcd_IrsensorTypes.h>

#include "detail/irsensor_ApiImpl.h"

namespace nn { namespace irsensor {

IrCameraHandle GetIrCameraHandle(
    const ::nn::hid::NpadIdType& npadIdType) NN_NOEXCEPT
{
    IrCameraHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        detail::GetIrCameraHandle(&handle, npadIdType));
    return handle;
}

void Initialize(const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::InitializeIrCamera(nn::irsensor::IrSensorFunctionLevel_Latest, handle));
}

void Finalize(const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::FinalizeIrCamera(handle));
}

IrCameraStatus GetIrCameraStatus(const IrCameraHandle& handle) NN_NOEXCEPT
{
    IrCameraStatus status;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        detail::GetIrCameraStatus(&status, handle));
    return status;
}

ImageProcessorStatus GetImageProcessorStatus(const IrCameraHandle& handle) NN_NOEXCEPT
{
    ImageProcessorStatus status;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        detail::GetImageProcessorStatus(&status, false, handle));
    return status;
}

::nn::Result CheckFirmwareUpdateNecessity(bool* pOutIsUpdateNeeded, const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutIsUpdateNeeded);
    return detail::CheckFirmwareUpdateNecessity(pOutIsUpdateNeeded, handle);
}

void StopImageProcessor(const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        detail::StopImageProcessor(handle));
}

void StopImageProcessorAsync(const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        detail::StopImageProcessorAsync(handle));
}

void GetMomentProcessorDefaultConfig(
    MomentProcessorConfig* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    pOutValue->irCameraConfig.exposureTime =
        ::nn::TimeSpan::FromMicroSeconds(300);
    pOutValue->irCameraConfig.gain = 8;
    pOutValue->irCameraConfig.isNegativeImageUsed = false;
    pOutValue->irCameraConfig.lightTarget = IrCameraLightTarget_AllObjects;

    pOutValue->preprocess = MomentProcessorPreprocess_Cutoff;
    pOutValue->preprocessIntensityThreshold = 80;

    pOutValue->windowOfInterest.x = 0;
    pOutValue->windowOfInterest.y = 0;
    pOutValue->windowOfInterest.width = IrCameraImageWidth;
    pOutValue->windowOfInterest.height = IrCameraImageHeight;
}

void RunMomentProcessor(
    const IrCameraHandle& handle,
    const MomentProcessorConfig& config) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(config.irCameraConfig.exposureTime,
        MomentProcessorExposureTimeMin,
        MomentProcessorExposureTimeMax);
    NN_SDK_REQUIRES_MINMAX(config.irCameraConfig.gain,
        IrCameraGainMin,
        IrCameraGainMax);
    NN_SDK_REQUIRES(
        config.irCameraConfig.lightTarget == IrCameraLightTarget_AllObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_NearObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_FarObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_None);
    NN_SDK_REQUIRES(
        config.preprocess == MomentProcessorPreprocess_Binarize
        || config.preprocess == MomentProcessorPreprocess_Cutoff);
    NN_SDK_REQUIRES_MINMAX(config.preprocessIntensityThreshold, 0, ::nn::irsensor::IrCameraIntensityMax);
    NN_SDK_REQUIRES_GREATER_EQUAL(config.windowOfInterest.x, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(config.windowOfInterest.width,
        MomentProcessorBlockColumnCount);
    NN_SDK_REQUIRES_LESS_EQUAL(
        config.windowOfInterest.x + config.windowOfInterest.width,
        IrCameraImageWidth);
    NN_SDK_REQUIRES_GREATER_EQUAL(config.windowOfInterest.y, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(config.windowOfInterest.height,
        MomentProcessorBlockRowCount);
    NN_SDK_REQUIRES_LESS_EQUAL(
        config.windowOfInterest.y + config.windowOfInterest.height,
        IrCameraImageHeight);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        detail::RunMomentProcessor(handle, config));
}

::nn::Result GetMomentProcessorState(
    MomentProcessorState* pOutValue, const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    return detail::GetMomentProcessorState(pOutValue, handle);
}

::nn::Result GetMomentProcessorStates(
    MomentProcessorState* pOutStates, int* pOutCount, int countMax,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_GREATER_EQUAL(countMax, 0);

    return detail::GetMomentProcessorStates(
        pOutStates, pOutCount, countMax, handle);
}

MomentStatistic CalculateMomentRegionStatistic(
    const MomentProcessorState* pState,
    const Rect& windowOfInterest,
    int startRow,
    int startColumn,
    int rowCount,
    int columnCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);
    NN_SDK_REQUIRES_LESS_EQUAL(0, startRow);
    NN_SDK_REQUIRES_LESS_EQUAL(1, rowCount);
    NN_SDK_REQUIRES_LESS_EQUAL(startRow + rowCount, MomentProcessorBlockRowCount);
    NN_SDK_REQUIRES_LESS_EQUAL(0, startColumn);
    NN_SDK_REQUIRES_LESS_EQUAL(1, columnCount);
    NN_SDK_REQUIRES_LESS_EQUAL(startColumn + columnCount, MomentProcessorBlockColumnCount);

    NN_SDK_REQUIRES_GREATER_EQUAL(windowOfInterest.x, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(windowOfInterest.width, MomentProcessorBlockColumnCount);
    NN_SDK_REQUIRES_LESS_EQUAL(
        windowOfInterest.x + windowOfInterest.width, IrCameraImageWidth);
    NN_SDK_REQUIRES_GREATER_EQUAL(windowOfInterest.y, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(windowOfInterest.height, MomentProcessorBlockRowCount);
    NN_SDK_REQUIRES_LESS_EQUAL(
        windowOfInterest.y + windowOfInterest.height, IrCameraImageHeight);

    // WOI 設定から対象のブロックサイズを計算する
    auto blockWidth = windowOfInterest.width / MomentProcessorBlockColumnCount;
    auto blockHeight = windowOfInterest.height / MomentProcessorBlockRowCount;

    // モーメントデータを再計算する
    double moment00 = 0;
    double moment01 = 0;
    double moment10 = 0;
    for (auto i = startRow; i < startRow + rowCount; ++i)
    {
        for (auto j = startColumn; j < startColumn + columnCount; ++j)
        {
            auto& block = pState->blocks[j + i * MomentProcessorBlockColumnCount];
            auto tempMoment00 = block.averageIntensity * blockWidth * blockHeight;
            moment00 += tempMoment00;
            moment10 += block.centroid.x * tempMoment00;
            moment01 += block.centroid.y * tempMoment00;
        }
    }

    auto blockCount = rowCount * columnCount;
    MomentStatistic statistic = {};
    statistic.averageIntensity = static_cast<float>(moment00 / (blockCount * blockWidth * blockHeight));
    if (moment00 != 0)
    {
        statistic.centroid.x = static_cast<float>(moment10 / moment00);
        statistic.centroid.y = static_cast<float>(moment01 / moment00);
    }
    else
    {
        statistic.centroid.x = 0.0f;
        statistic.centroid.y = 0.0f;
    }

    return statistic;
}

void GetClusteringProcessorDefaultConfig(
    ClusteringProcessorConfig* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    pOutValue->irCameraConfig.exposureTime =
        ::nn::TimeSpan::FromMicroSeconds(200);
    pOutValue->irCameraConfig.gain = 2;
    pOutValue->irCameraConfig.isNegativeImageUsed = false;
    pOutValue->irCameraConfig.lightTarget = IrCameraLightTarget_AllObjects;

    pOutValue->objectIntensityMin = 150;
    pOutValue->objectPixelCountMin = 3;
    pOutValue->objectPixelCountMax = ClusteringProcessorObjectPixelCountMax;

    pOutValue->windowOfInterest.x = 0;
    pOutValue->windowOfInterest.y = 0;
    pOutValue->windowOfInterest.width = IrCameraImageWidth;
    pOutValue->windowOfInterest.height = IrCameraImageHeight;

    pOutValue->isExternalLightFilterEnabled = true;
}

void RunClusteringProcessor(
    const IrCameraHandle& handle,
    const ClusteringProcessorConfig& config) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(config.irCameraConfig.exposureTime,
        ClusteringProcessorExposureTimeMin,
        ClusteringProcessorExposureTimeMax);
    NN_SDK_REQUIRES_MINMAX(config.irCameraConfig.gain,
        IrCameraGainMin,
        IrCameraGainMax);
    NN_SDK_REQUIRES(
        config.irCameraConfig.lightTarget == IrCameraLightTarget_AllObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_NearObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_FarObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_None);
    NN_SDK_REQUIRES_MINMAX(config.objectIntensityMin, 0, ::nn::irsensor::IrCameraIntensityMax);
    NN_SDK_REQUIRES_MINMAX(config.objectPixelCountMin,
        0,
        ClusteringProcessorObjectPixelCountMax);
    NN_SDK_REQUIRES_MINMAX(config.objectPixelCountMax,
        0,
        ClusteringProcessorObjectPixelCountMax);
    NN_SDK_REQUIRES_GREATER_EQUAL(config.windowOfInterest.x, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(config.windowOfInterest.width, 1);
    NN_SDK_REQUIRES_LESS_EQUAL(
        config.windowOfInterest.x + config.windowOfInterest.width,
        IrCameraImageWidth);
    NN_SDK_REQUIRES_GREATER_EQUAL(config.windowOfInterest.y, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(config.windowOfInterest.height, 1);
    NN_SDK_REQUIRES_LESS_EQUAL(
        config.windowOfInterest.y + config.windowOfInterest.height,
        IrCameraImageHeight);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        detail::RunClusteringProcessor(handle, config));
}

::nn::Result GetClusteringProcessorState(
    ClusteringProcessorState* pOutValue,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    return detail::GetClusteringProcessorState(pOutValue, handle);
}

::nn::Result GetClusteringProcessorStates(
    ClusteringProcessorState* pOutStates, int* pOutCount, int countMax,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_GREATER_EQUAL(countMax, 0);

    return detail::GetClusteringProcessorStates(
               pOutStates, pOutCount, countMax, handle);
}

void GetImageTransferProcessorDefaultConfig(
    ImageTransferProcessorConfig* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    pOutValue->irCameraConfig.exposureTime =
        ::nn::TimeSpan::FromMicroSeconds(300);
    pOutValue->irCameraConfig.gain = 8;
    pOutValue->irCameraConfig.isNegativeImageUsed = false;
    pOutValue->irCameraConfig.lightTarget = IrCameraLightTarget_AllObjects;

    pOutValue->format = ImageTransferProcessorFormat_320x240;
}

void GetImageTransferProcessorDefaultConfig(
    ImageTransferProcessorExConfig* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    pOutValue->irCameraConfig.exposureTime =
        ::nn::TimeSpan::FromMicroSeconds(300);
    pOutValue->irCameraConfig.gain = 8;
    pOutValue->irCameraConfig.isNegativeImageUsed = false;
    pOutValue->irCameraConfig.lightTarget = IrCameraLightTarget_AllObjects;

    pOutValue->origFormat = ImageTransferProcessorFormat_320x240;
    pOutValue->trimmingFormat = ImageTransferProcessorFormat_320x240;
    pOutValue->trimmingStartX = 0;
    pOutValue->trimmingStartY = 0;
}

void RunImageTransferProcessor(
    const IrCameraHandle& handle,
    const ImageTransferProcessorConfig& config,
    void* workBuffer,
    size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(config.irCameraConfig.exposureTime,
        ImageTransferProcessorExposureTimeMin,
        ImageTransferProcessorExposureTimeMax);
    NN_SDK_REQUIRES_MINMAX(config.irCameraConfig.gain,
        IrCameraGainMin,
        IrCameraGainMax);
    NN_SDK_REQUIRES(
        config.irCameraConfig.lightTarget == IrCameraLightTarget_AllObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_NearObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_FarObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_None);
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES_ALIGNED(workBuffer, ::nn::os::MemoryPageSize);
    NN_SDK_REQUIRES(
        (config.format == ImageTransferProcessorFormat_320x240
        && size == ImageTransferProcessorWorkBufferSize320x240)
        || (config.format == ImageTransferProcessorFormat_160x120
        && size == ImageTransferProcessorWorkBufferSize160x120)
        || (config.format == ImageTransferProcessorFormat_80x60
        && size == ImageTransferProcessorWorkBufferSize80x60)
        || (config.format == ImageTransferProcessorFormat_40x30
        && size == ImageTransferProcessorWorkBufferSize40x30)
        || (config.format == ImageTransferProcessorFormat_20x15
        && size == ImageTransferProcessorWorkBufferSize20x15));

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        detail::RunImageTransferProcessor(handle, config, workBuffer, size));
}

void RunImageTransferProcessor(
    const IrCameraHandle& handle,
    const ImageTransferProcessorExConfig& config,
    void* workBuffer,
    size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(config.irCameraConfig.exposureTime,
        ImageTransferProcessorExposureTimeMin,
        ImageTransferProcessorExposureTimeMax);
    NN_SDK_REQUIRES_MINMAX(config.irCameraConfig.gain,
        IrCameraGainMin,
        IrCameraGainMax);
    NN_SDK_REQUIRES(
        config.irCameraConfig.lightTarget == IrCameraLightTarget_AllObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_NearObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_FarObjects
        || config.irCameraConfig.lightTarget == IrCameraLightTarget_None);
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES_ALIGNED(workBuffer, ::nn::os::MemoryPageSize);
    NN_SDK_REQUIRES(
        (config.trimmingFormat == ImageTransferProcessorFormat_320x240
        && size == ImageTransferProcessorWorkBufferSize320x240)
        || (config.trimmingFormat == ImageTransferProcessorFormat_160x120
        && size == ImageTransferProcessorWorkBufferSize160x120)
        || (config.trimmingFormat == ImageTransferProcessorFormat_80x60
        && size == ImageTransferProcessorWorkBufferSize80x60)
        || (config.trimmingFormat == ImageTransferProcessorFormat_40x30
        && size == ImageTransferProcessorWorkBufferSize40x30)
        || (config.trimmingFormat == ImageTransferProcessorFormat_20x15
        && size == ImageTransferProcessorWorkBufferSize20x15));
    NN_SDK_REQUIRES_LESS_EQUAL(config.origFormat, config.trimmingFormat);

    NN_SDK_REQUIRES_MINMAX(
        config.trimmingStartX,
        0,
        (IrCameraImageWidth >> config.origFormat) - (IrCameraImageWidth >> config.trimmingFormat));
    NN_SDK_REQUIRES_MINMAX(
        config.trimmingStartY,
        0,
        (IrCameraImageHeight >> config.origFormat) - (IrCameraImageHeight >> config.trimmingFormat));

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        detail::RunImageTransferProcessor(handle, config, workBuffer, size));
}

::nn::Result GetImageTransferProcessorState(
    ImageTransferProcessorState* pOutState,
    void* pOutImage,
    size_t size,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);
    NN_SDK_REQUIRES_NOT_NULL(pOutImage);

    return detail::GetImageTransferProcessorState(
        pOutState, pOutImage, size, handle);
}

::nn::Result RunHandAnalysis(const IrCameraHandle& handle, const HandAnalysisConfig& config) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(
        config.mode == HandAnalysisMode_Image
        || config.mode == HandAnalysisMode_Silhouette
        || config.mode == HandAnalysisMode_SilhouetteAndImage
        || config.mode == HandAnalysisMode_SilhouetteOnly);

    int mode;
    switch (config.mode)
    {
    case HandAnalysisMode_Image:
    {
        mode = HandAnalysisModeImage_v2;
    }
    break;
    case HandAnalysisMode_Silhouette:
    {
        mode = HandAnalysisModeSilhouette_v2;
    }
    break;
    case HandAnalysisMode_SilhouetteAndImage:
    {
        mode = HandAnalysisModeSilhouetteAndImage_v2;
    }
    break;
    case HandAnalysisMode_SilhouetteOnly:
    {
        mode = HandAnalysisModeSilhouetteOnly_v2;
    }
    break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    TeraPluginProcessorConfig teraConfig = {};
    teraConfig.modeOffset = static_cast<uint8_t>(mode - static_cast<int>(nn::xcd::IrProcessorType::TeraPlugin));
    teraConfig.isParameterEnabled = false;

    return RunTeraPluginProcessor(handle, teraConfig);
}

void RunPointingProcessor(
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    // デフォルト設定で起動する。
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::RunPointingProcessor(handle));
}

::nn::Result GetPointingProcessorMarkerStates(
    PointingProcessorMarkerState* pOutStates, int* pOutCount, int countMax,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_GREATER_EQUAL(countMax, 0);

    return detail::GetPointingProcessorStates(pOutStates, pOutCount, countMax, handle);
}

::nn::Result GetPointingProcessorStates(
    PointingProcessorState* pOutStates, int* pOutCount, int countMax,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_GREATER_EQUAL(countMax, 0);

    PointingProcessorMarkerState states[::nn::irsensor::PointingProcessorStateCountMax];
    nn::Result result = GetPointingProcessorMarkerStates(&states[0], pOutCount, countMax, handle);

    // データ変換
    for (auto i = 0; i < *pOutCount; i++)
    {
        PointingProcessorState* pOutState = pOutStates + i;
        pOutState->samplingNumber = states[i].samplingNumber;
        pOutState->timeStamp = states[i].timeStamp;

        int validObjectCnt = 0;
        float centroidX = 0;
        float centroidY = 0;
        for (auto j = 0; j < ::nn::irsensor::PointingProcessorMarkerObjectCount; j++)
        {
            if (states[i].objects[j].isDataValid)
            {
                validObjectCnt++;
                // 有効なデータ点の平均値をポインティング座標とする
                centroidX += states[i].objects[j].centroid.x;
                centroidY += states[i].objects[j].centroid.y;
            }
        }

        if (validObjectCnt < ::nn::irsensor::PointingProcessorMarkerObjectCount)
        {
            pOutState->pointingStatus = ::nn::irsensor::PointingStatus_LackOfObjects;
        }
        else
        {
            pOutState->pointingStatus = ::nn::irsensor::PointingStatus_DataValid;
        }

        if (validObjectCnt == 0)
        {
            pOutState->position.x = 0.0f;
            pOutState->position.y = 0.0f;
        }
        else
        {
            pOutState->position.x = centroidX / static_cast<float>(validObjectCnt);
            pOutState->position.y = centroidY / static_cast<float>(validObjectCnt);
            // 結果を正規化。 中心が (0,0), 右上が (1.f, 1.f)
            pOutState->position.x = 1.0f - pOutState->position.x / (IrCameraImageWidth / 2.f);
            pOutState->position.y = pOutState->position.y / (IrCameraImageHeight / 2.f) - 1.0f;
        }
    }
    return result;
}

::nn::Result RunTeraPluginProcessor(const IrCameraHandle& handle, const TeraPluginProcessorConfig& config) NN_NOEXCEPT
{
    return detail::RunTeraPluginProcessor(handle, config);
}

::nn::Result GetTeraPluginProcessorStates(
    TeraPluginProcessorState* pOutStates, int* pOutCount, int countMax,
    int64_t infSamplingNumber,
    uint32_t prefix,
    int prefixSize,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_GREATER_EQUAL(countMax, 0);

    return detail::GetTeraPluginProcessorStates(
               pOutStates, pOutCount, countMax, infSamplingNumber, prefix, prefixSize, handle);
}

void GetIrLedProcessorDefaultConfig(
    IrLedProcessorConfig* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    pOutValue->lightTarget = IrCameraLightTarget_AllObjects;
}

void RunIrLedProcessor(const IrCameraHandle& handle, const IrLedProcessorConfig& config) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::RunIrLedProcessor(handle, config));
}

::nn::Result GetIrLedProcessorState(
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    return detail::GetIrLedProcessorState(handle);
}

Rect MakeRect(int x, int y, int width, int height) NN_NOEXCEPT
{
    Rect r;
    r.x = static_cast<int16_t>(x);
    r.y = static_cast<int16_t>(y);
    r.width = static_cast<int16_t>(width);
    r.height = static_cast<int16_t>(height);
    return r;
}

}} // namespace nn::irsensor
