﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/irsensor/irsensor_MomentProcessor.h>
#include <nn/irsensor/irsensor_ClusteringProcessor.h>
#include <nn/irsensor/irsensor_PointingProcessor.h>
#include <nn/irsensor/irsensor_PointingProcessorTypesPrivate.h>
#include <nn/irsensor/irsensor_TeraPluginProcessorTypesPrivate.h>
#include <nn/irsensor/irsensor_IrCameraTypesPrivate.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/hid/system/hid_Irsensor.h>

#include "irsensor_RingLifo.h"

namespace nn { namespace irsensor { namespace detail {

//!< IR センサーに対するアプレット状態管理テーブルのエントリの最大数
const size_t AppletStatusEntryCountMax = 4;

//!< アプレットの状態制御フラグ定義です。
struct AppletStatusFlag final
{
    typedef ::nn::util::BitFlagSet<32, AppletStatusFlag>::Flag<0>
        IsForeground;    //!< アプレットが FG か否か
};

//!< アプレットの状態制御フラグ集合を扱う型です。
typedef ::nn::util::BitFlagSet<32, AppletStatusFlag> AppletStatusFlagSet;

class StatusManager final
{
    NN_DISALLOW_COPY(StatusManager);
    NN_DISALLOW_MOVE(StatusManager);

public:
    typedef RingLifo<
        MomentProcessorState,
        MomentProcessorStateCountMax> MomentLifo;

    typedef RingLifo<
        ClusteringProcessorState,
        ClusteringProcessorStateCountMax> ClusteringLifo;

    typedef RingLifo<
        PointingProcessorMarkerState,
        PointingProcessorStateCountMax> PointingLifo;

    typedef RingLifo<
        TeraPluginProcessorState,
        TeraPluginProcessorStateCountMax> TeraPluginLifo;

private:
    enum class IrSensorMode : int32_t
    {
        None,
        Moment,
        Clustering,
        ImageTransfer,
        Pointing,
        TeraPlugin,
    };

    union RingLifoStorage
    {
        ::nn::util::TypedStorage<
            MomentLifo,
            sizeof(MomentLifo),
            NN_ALIGNOF(MomentLifo)> moment;
        ::nn::util::TypedStorage<
            ClusteringLifo,
            sizeof(ClusteringLifo),
            NN_ALIGNOF(ClusteringLifo)> clustering;
        ::nn::util::TypedStorage<
            PointingLifo,
            sizeof(PointingLifo),
            NN_ALIGNOF(PointingLifo)> pointing;
        ::nn::util::TypedStorage<
            TeraPluginLifo,
            sizeof(TeraPluginLifo),
            NN_ALIGNOF(TeraPluginLifo)> teraPlugin;
    };

    struct SharedMemoryFormatForDevice
    {
        ::std::atomic<int32_t> cameraStatus;
        ::std::atomic<int32_t> internalCameraStatus;
        ::std::atomic<IrSensorMode> mode;
        RingLifoStorage ringLifoStorage;
    };

    struct SharedMemoryFormatForApplet
    {
        ::std::atomic<::nn::applet::AppletResourceUserId> aruid;
        ::std::atomic<AppletStatusFlagSet> appletStatusFlag;
    };

    struct SharedMemoryFormat
    {
        SharedMemoryFormatForDevice deviceFormat[::nn::hid::system::IrSensorSupportedNpadIdsCount];
        SharedMemoryFormatForApplet appletFormat[AppletStatusEntryCountMax];
    };

    //!< SharedMemory に配置されるフォーマット
    SharedMemoryFormat m_SharedMemoryFormat;

public:
    StatusManager() NN_NOEXCEPT;

    ~StatusManager() NN_NOEXCEPT;

    //!< IrCameraStatus を取得します。
    IrCameraStatus GetIrCameraStatus(const IrCameraHandle& handle) const NN_NOEXCEPT;

    //!< IrCameraStatus を設定します。
    void SetIrCameraStatus(const IrCameraHandle& handle, IrCameraStatus status) NN_NOEXCEPT;

    //!< IrCameraInternalStatus を取得します。
    IrCameraInternalStatus GetIrCameraInternalStatus(const IrCameraHandle& handle) const NN_NOEXCEPT;

    //!< IrCameraInternalStatus を設定します。
    void SetIrCameraInternalStatus(const IrCameraHandle& handle, IrCameraInternalStatus status) NN_NOEXCEPT;

    //!< AppletResouceFlagSet を取得します。
    bool GetIrSensorAruidStatus(
        AppletStatusFlagSet* pOutValue,
        const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT;

    //!< AppletResourceFlagSet を設定します。
    void SetIrSensorAruidStatus(
        const ::nn::applet::AppletResourceUserId& aruid,
        AppletStatusFlagSet flag) NN_NOEXCEPT;

    //!< AppletResourceUserId を登録します。
    void RegisterIrSensorAruid(const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT;

    //!< AppletResourceUserId を破棄します。
    void UnregisterIrSensorAruid(const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT;

    //!< LIFO を Moment プロセッサのために確保します。
    void AllocateMomentLifo(const IrCameraHandle& handle) NN_NOEXCEPT;

    //!< LIFO を Clustering プロセッサのために確保します。
    void AllocateClusteringLifo(const IrCameraHandle& handle) NN_NOEXCEPT;

    //!< LIFO を Pointing プロセッサために確保します。
    void AllocatePointingLifo(const IrCameraHandle& handle) NN_NOEXCEPT;

    //!< LIFO を TeraPlugin プロセッサのために確保します。
    void AllocateTeraPluginLifo(const IrCameraHandle& handle) NN_NOEXCEPT;

    //!< LIFO を解放します。
    void FreeLifo(const IrCameraHandle& handle) NN_NOEXCEPT;

    //!< LIFO のデータをクリアします。
    void ClearLifo(const IrCameraHandle& handle) NN_NOEXCEPT;

    //!< Moment プロセッサのための LIFO を取得します。
    MomentLifo* GetMomentLifo(const IrCameraHandle& handle) const NN_NOEXCEPT;

    //!< Clustering プロセッサのための LIFO を取得します。
    ClusteringLifo* GetClusteringLifo(const IrCameraHandle& handle) const NN_NOEXCEPT;

    //!< Pointing プロセッサのための LIFO を取得します。
    PointingLifo* GetPointingLifo(const IrCameraHandle& handle) const NN_NOEXCEPT;

    //!< TeraPlugin プロセッサのための LIFO を取得します。
    TeraPluginLifo* GetTeraPluginLifo(const IrCameraHandle& handle) const NN_NOEXCEPT;
};

}}} // namespace nn::irsensor::detail
