﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_NativeHandle.h>

#include "irsensor_ActivationCount.h"
#include "irsensor_Session.h"
#include "irsensor_StatusManagerHolder.h"
#include "irsensor_InternalErrorHandler.h"

namespace nn { namespace irsensor { namespace detail {

//!< IR センサーのセッションを扱うためのクラスです。
class IrSensorSession final : public Session
{
    NN_DISALLOW_COPY(IrSensorSession);
    NN_DISALLOW_MOVE(IrSensorSession);

private:
    enum class IrSensorMode
    {
        None,
        Moment,
        Clustering,
        ImageTransfer,
        Pointing,
        TeraPlugin,
        IrLed,
    };

private:
    //!< irs サービスのセッション
    ::nn::sf::SharedPointer<IIrSensorServer> m_Session;

    //!< 呼び出し元アプレットの Aruid
    ::nn::applet::AppletResourceUserId m_Aruid;

    //!< IR センサーの状態
    StatusManagerHolder m_StatusManagerHolder;

    //!< IR センサーのモード
    IrSensorMode m_IrSensorMode[::nn::hid::system::IrSensorSupportedNpadIdsCount];

    //!< ImageTransfer プロセッサ利用時の画像フォーマット
    ImageTransferProcessorFormat m_ImageTransferProcessorFormat[::nn::hid::system::IrSensorSupportedNpadIdsCount];

    //!< 白黒反転設定
    bool m_UsesNegativeImage[::nn::hid::system::IrSensorSupportedNpadIdsCount];

    //!< WindowOfInterest
    Rect m_WindowOfInterest[::nn::hid::system::IrSensorSupportedNpadIdsCount];

    //!< バージョンチェックリクエストが呼び出し可能な状態かどうか。（連続呼び出しを避けるため）
    bool m_IsVersionCheckRequestEnabled[::nn::hid::system::IrSensorSupportedNpadIdsCount];

    //!< 内部エラー管理用
    InternalErrorHandler m_InternalErrorHandler;

    //!< FunctionLevel
    IrSensorFunctionLevel m_FunctionLevel;

public:
    explicit IrSensorSession(const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT;

    virtual ~IrSensorSession() NN_NOEXCEPT NN_OVERRIDE;

    const StatusManager::MomentLifo* GetMomentLifo(const IrCameraHandle& handle) const NN_NOEXCEPT;

    const StatusManager::ClusteringLifo* GetClusteringLifo(const IrCameraHandle& handle) const NN_NOEXCEPT;

    const StatusManager::PointingLifo* GetPointingLifo(const IrCameraHandle& handle) const NN_NOEXCEPT;

    const StatusManager::TeraPluginLifo* GetTeraPluginLifo(const IrCameraHandle& handle) const NN_NOEXCEPT;

    IrCameraStatus GetIrCameraStatus(const IrCameraHandle& handle) NN_NOEXCEPT;

    IrCameraInternalStatus GetIrCameraInternalStatus(const IrCameraHandle& handle) NN_NOEXCEPT;

    bool IsLibraryAppletCallEnabled(const IrCameraInternalStatus& status, const IrCameraHandle& handle) NN_NOEXCEPT;

    void ResetInternalStatusForLibraryAppletCall(const IrCameraHandle& handle) NN_NOEXCEPT;

    ::nn::Result CheckFirmwareVersion(const IrCameraHandle& handle, const PackedMcuVersion& requiredVersion) NN_NOEXCEPT;

    bool IsAppletForeground() NN_NOEXCEPT;

    void AttachSharedMemoryIfNotMapped() NN_NOEXCEPT;

    void SetFunctionLevel(const IrCameraHandle& handle, const IrSensorFunctionLevel& functionLevel) NN_NOEXCEPT;

    void GetFunctionLevel(IrSensorFunctionLevel* pOutValue, const IrCameraHandle& handle) NN_NOEXCEPT;

    ::nn::Result StopImageProcessorAsync(const IrCameraHandle& handle) NN_NOEXCEPT;

    ::nn::Result RunMomentProcessor(
        const IrCameraHandle& handle,
        const PackedMomentProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result RunClusteringProcessor(
        const IrCameraHandle& handle,
        const PackedClusteringProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result RunImageTransferProcessor(
        const IrCameraHandle& handle,
        const PackedImageTransferProcessorConfig& config,
        void* workBuffer,
        size_t size) NN_NOEXCEPT;

    ::nn::Result RunImageTransferProcessor(
        const IrCameraHandle& handle,
        const PackedImageTransferProcessorExConfig& config,
        void* workBuffer,
        size_t size) NN_NOEXCEPT;

    ::nn::Result RunPointingProcessor(
        const IrCameraHandle& handle,
        const PackedPointingProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result RunTeraPluginProcessor(
        const IrCameraHandle& handle,
        const PackedTeraPluginProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result RunIrLedProcessor(
        const IrCameraHandle& handle,
        const PackedIrLedProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result GetImageTransferState(
        const IrCameraHandle& handle,
        ImageTransferProcessorState* pOutState,
        void* pOutImage,
        size_t size) NN_NOEXCEPT;

    void GetMomentProcessorStateDefault(
        MomentProcessorState* pOutValue,
        const IrCameraHandle& handle) NN_NOEXCEPT;

    void GetClusteringProcessorStateDefault(
        ClusteringProcessorState* pOutValue,
        const IrCameraHandle& handle) NN_NOEXCEPT;

    void GetImageTransferProcessorStateDefault(
        ImageTransferProcessorState* pOutValue,
        void* pOutImage,
        size_t size,
        const IrCameraHandle& handle) NN_NOEXCEPT;

    size_t GetImageSize(ImageTransferProcessorFormat format) NN_NOEXCEPT;

    bool IsProcessorStopped(const IrCameraHandle& handle) NN_NOEXCEPT;

    bool IsVersionCheckRequestEnabled(const IrCameraHandle& handle) NN_NOEXCEPT;

    void SetVersionCheckRequestEnableFlag(const IrCameraHandle& handle, bool isEnable) NN_NOEXCEPT;

    Rect GetWindowOfInterestConfig(const IrCameraHandle& handle) NN_NOEXCEPT;

protected:
    virtual ::nn::Result ThrowResultActivationUpperLimitOver() NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result ThrowResultDeactivationLowerLimitOver() NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result ThrowResultAllocationFailed() NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result CreateProxy() NN_NOEXCEPT NN_OVERRIDE;

    virtual void DestroyProxy() NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result ActivateService() NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result ActivateServiceWithFunctionLevel(const IrSensorFunctionLevel& functionLevel) NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result DeactivateService() NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result GetSharedMemoryHandle(
        ::nn::sf::NativeHandle* outValue) NN_NOEXCEPT NN_OVERRIDE;

    virtual bool IsSharedMemoryMapped() NN_NOEXCEPT NN_OVERRIDE;

    virtual void AttachSharedMemory(::nn::os::NativeHandle handle,
                                    bool managed) NN_NOEXCEPT NN_OVERRIDE;

    virtual void FinalizeSharedMemory() NN_NOEXCEPT NN_OVERRIDE;
};

}}} // namespace nn::irsensor::detail
