﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/irsensor/irsensor_Result.h>
#include <nn/irsensor/irsensor_ResultPrivate.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/sf/sf_Buffers.h>
#include <nn/sf/sf_NativeHandle.h>

#include "irsensor_IrSensorServer.h"
#include "irsensor_IrSensorSystemServer.h"
#include "irsensor_IrCameraHandle.h"
#include "irsensor_StatusManagerHolder.h"
#include "irsensor_LockableMutexType.h"
#include "irsensor_IrSensorSession.h"
#include "irsensor_AppletResourceManager.h"

namespace nn { namespace irsensor { namespace detail {

IrSensorSession::IrSensorSession(
    const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
    : m_Session()
    , m_Aruid(aruid)
    , m_StatusManagerHolder()
    , m_FunctionLevel(IrSensorFunctionLevel_0)
{
    for (auto i = 0; i < ::nn::hid::system::IrSensorSupportedNpadIdsCount; i++)
    {
        m_IrSensorMode[i] = IrSensorMode::None;
        m_ImageTransferProcessorFormat[i] = ImageTransferProcessorFormat_320x240;
        m_UsesNegativeImage[i] = false;
        m_WindowOfInterest[i] = ::nn::irsensor::Rect();
        m_IsVersionCheckRequestEnabled[i] = true;
    }
}

IrSensorSession::~IrSensorSession() NN_NOEXCEPT
{
    // 何もしない
}

const StatusManager::MomentLifo* IrSensorSession::GetMomentLifo(
    const IrCameraHandle& handle
    ) const NN_NOEXCEPT
{
    if((m_IrSensorMode[GetIrCameraHandlePlayerNumber(handle)] != IrSensorMode::Moment)
        && (m_IrSensorMode[GetIrCameraHandlePlayerNumber(handle)] != IrSensorMode::IrLed))
    {
        return nullptr;
    }

    auto pStatusManager = m_StatusManagerHolder.GetStatusManager();
    NN_SDK_ASSERT_NOT_NULL(pStatusManager);

    return pStatusManager->GetMomentLifo(handle);
}

const StatusManager::ClusteringLifo* IrSensorSession::GetClusteringLifo(
    const IrCameraHandle& handle
    ) const NN_NOEXCEPT
{
    if(m_IrSensorMode[GetIrCameraHandlePlayerNumber(handle)] != IrSensorMode::Clustering)
    {
        return nullptr;
    }

    auto pStatusManager = m_StatusManagerHolder.GetStatusManager();
    NN_SDK_ASSERT_NOT_NULL(pStatusManager);

    return pStatusManager->GetClusteringLifo(handle);
}

const StatusManager::PointingLifo* IrSensorSession::GetPointingLifo(
    const IrCameraHandle& handle
    ) const NN_NOEXCEPT
{
    if(m_IrSensorMode[GetIrCameraHandlePlayerNumber(handle)] != IrSensorMode::Pointing)
    {
        return nullptr;
    }

    auto pStatusManager = m_StatusManagerHolder.GetStatusManager();
    NN_SDK_ASSERT_NOT_NULL(pStatusManager);

    return pStatusManager->GetPointingLifo(handle);
}

const StatusManager::TeraPluginLifo* IrSensorSession::GetTeraPluginLifo(
    const IrCameraHandle& handle
    ) const NN_NOEXCEPT
{
    if(m_IrSensorMode[GetIrCameraHandlePlayerNumber(handle)] != IrSensorMode::TeraPlugin)
    {
        return nullptr;
    }

    auto pStatusManager = m_StatusManagerHolder.GetStatusManager();
    NN_SDK_ASSERT_NOT_NULL(pStatusManager);

    return pStatusManager->GetTeraPluginLifo(handle);
}

IrCameraStatus IrSensorSession::GetIrCameraStatus(const IrCameraHandle& handle) NN_NOEXCEPT
{
    auto pStatusManager = m_StatusManagerHolder.GetStatusManager();
    NN_SDK_ASSERT_NOT_NULL(pStatusManager);

    return pStatusManager->GetIrCameraStatus(handle);
}

IrCameraInternalStatus IrSensorSession::GetIrCameraInternalStatus(const IrCameraHandle& handle) NN_NOEXCEPT
{
    auto pStatusManager = m_StatusManagerHolder.GetStatusManager();
    NN_SDK_ASSERT_NOT_NULL(pStatusManager);

    IrCameraInternalStatus internalStatus = pStatusManager->GetIrCameraInternalStatus(handle);
    return internalStatus;
}

::nn::Result IrSensorSession::CheckFirmwareVersion(
    const IrCameraHandle& handle, const ::nn::irsensor::PackedMcuVersion& requiredVersion) NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL(m_FunctionLevel, IrSensorFunctionLevel_1);
    NN_SDK_ASSERT(IsProcessorStopped(handle), "Cannot check firmware version while IR Camera running");

    NN_RESULT_DO(m_Session->CheckFirmwareVersion(m_Aruid, handle, requiredVersion));
    NN_RESULT_SUCCESS;
}

bool IrSensorSession::IsAppletForeground() NN_NOEXCEPT
{
    auto pStatusManager = m_StatusManagerHolder.GetStatusManager();
    NN_SDK_ASSERT_NOT_NULL(pStatusManager);

    bool isAppletForeground = true;
    AppletStatusFlagSet statusFlag;
    if (pStatusManager->GetIrSensorAruidStatus(&statusFlag, m_Aruid))
    {
        isAppletForeground = statusFlag.Test<AppletStatusFlag::IsForeground>();
    }
    return isAppletForeground;
}

void IrSensorSession::AttachSharedMemoryIfNotMapped() NN_NOEXCEPT
{
    if(!IsSharedMemoryMapped())
    {
        Activate(m_FunctionLevel);
    }
}

void IrSensorSession::SetFunctionLevel(const IrCameraHandle& handle, const IrSensorFunctionLevel& functionLevel) NN_NOEXCEPT
{
    NN_UNUSED(handle);
    // shim 側管理用に FunctionLevel を設定する
    m_FunctionLevel = functionLevel;
}

void IrSensorSession::GetFunctionLevel(IrSensorFunctionLevel* pOutValue, const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_UNUSED(handle);
    // shim 側管理用に FunctionLevelを取得する
    *pOutValue = m_FunctionLevel;
}

::nn::Result IrSensorSession::StopImageProcessorAsync(const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_RESULT_DO(m_Session->StopImageProcessorAsync(m_Aruid, handle));

    m_IrSensorMode[GetIrCameraHandlePlayerNumber(handle)] = IrSensorMode::None;
    NN_RESULT_SUCCESS;
}

bool IrSensorSession::IsLibraryAppletCallEnabled(
    const IrCameraInternalStatus& status,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    return m_InternalErrorHandler.IsLibraryAppletCallEnabled(status, handle);
}

void IrSensorSession::ResetInternalStatusForLibraryAppletCall(const IrCameraHandle& handle) NN_NOEXCEPT
{
    // InternalErrorHandler の内部状態を正常状態に戻す
    m_InternalErrorHandler.SetInternalStatus(IrCameraInternalStatus_Available, handle);
}

bool IrSensorSession::IsProcessorStopped(const IrCameraHandle& handle) NN_NOEXCEPT
{
    return m_IrSensorMode[GetIrCameraHandlePlayerNumber(handle)] == IrSensorMode::None;
}

bool IrSensorSession::IsVersionCheckRequestEnabled(const IrCameraHandle& handle) NN_NOEXCEPT
{
    return m_IsVersionCheckRequestEnabled[GetIrCameraHandlePlayerNumber(handle)];
}

void IrSensorSession::SetVersionCheckRequestEnableFlag(const IrCameraHandle& handle, bool isEnable) NN_NOEXCEPT
{
    m_IsVersionCheckRequestEnabled[GetIrCameraHandlePlayerNumber(handle)] = isEnable;
}

Rect IrSensorSession::GetWindowOfInterestConfig(const IrCameraHandle& handle) NN_NOEXCEPT
{
    return m_WindowOfInterest[GetIrCameraHandlePlayerNumber(handle)];
}

::nn::Result IrSensorSession::RunMomentProcessor(
    const IrCameraHandle& handle,
    const PackedMomentProcessorConfig& config
    ) NN_NOEXCEPT
{
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);

    if (m_FunctionLevel >= IrSensorFunctionLevel_1)
    {
        if ((m_IrSensorMode[playerNumber] != IrSensorMode::None)
            && (m_IrSensorMode[playerNumber] != IrSensorMode::Moment))
        {
            // 指定のプロセッサ以外のプロセッサ起動中に呼び出された場合は一旦中断する。
            NN_RESULT_DO(m_Session->SuspendImageProcessor(m_Aruid, handle));
        }
    }
    else
    {
        NN_SDK_REQUIRES_EQUAL(m_IrSensorMode[playerNumber], IrSensorMode::None);
    }

    // 同じプロセッサへの遷移の場合はプロセス内部で判別して、モード遷移を行わない。
    NN_RESULT_DO(m_Session->RunMomentProcessor(m_Aruid, handle, config));

    m_UsesNegativeImage[playerNumber] =
        config.irCameraConfig.isNegativeImageUsed;
    m_WindowOfInterest[playerNumber] = config.windowOfInterest;
    m_IrSensorMode[playerNumber] = IrSensorMode::Moment;
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorSession::RunClusteringProcessor(
    const IrCameraHandle& handle,
    const PackedClusteringProcessorConfig& config
    ) NN_NOEXCEPT
{
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);

    if (m_FunctionLevel >= IrSensorFunctionLevel_1)
    {
        if ((m_IrSensorMode[playerNumber] != IrSensorMode::None)
            && (m_IrSensorMode[playerNumber] != IrSensorMode::Clustering))
        {
            // 指定のプロセッサ以外のプロセッサ起動中に呼び出された場合は一旦中断する。
            NN_RESULT_DO(m_Session->SuspendImageProcessor(m_Aruid, handle));
        }
    }
    else
    {
        NN_SDK_REQUIRES_EQUAL(m_IrSensorMode[playerNumber], IrSensorMode::None);
    }

    // 同じプロセッサへの遷移の場合はプロセス内部で判別して、モード遷移を行わない。
    NN_RESULT_DO(m_Session->RunClusteringProcessor(m_Aruid, handle, config));

    m_UsesNegativeImage[playerNumber] =
        config.irCameraConfig.isNegativeImageUsed;
    m_WindowOfInterest[playerNumber] = config.windowOfInterest;
    m_IrSensorMode[playerNumber] = IrSensorMode::Clustering;
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorSession::RunImageTransferProcessor(
    const IrCameraHandle& handle,
    const PackedImageTransferProcessorConfig& config,
    void* workBuffer,
    size_t size) NN_NOEXCEPT
{
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);

    if (m_FunctionLevel >= IrSensorFunctionLevel_1)
    {
        if (m_IrSensorMode[playerNumber] != IrSensorMode::None)
        {
            // プロセッサ起動中に呼び出された場合は一旦中断する。
            NN_RESULT_DO(m_Session->SuspendImageProcessor(m_Aruid, handle));
        }
    }
    else
    {
        NN_SDK_REQUIRES_EQUAL(m_IrSensorMode[playerNumber], IrSensorMode::None);
    }

    ::nn::os::TransferMemoryType transferMemory;
    NN_RESULT_THROW_UNLESS(::nn::os::CreateTransferMemory(
        &transferMemory,
        workBuffer,
        size,
        ::nn::os::MemoryPermission_None).IsSuccess(),
        ResultIrsensorCreateTransferMemoryFailed());

    auto transferMemoryHandle = ::nn::os::DetachTransferMemory(&transferMemory);
    ::nn::os::DestroyTransferMemory(&transferMemory);

    NN_RESULT_DO(m_Session->RunImageTransferProcessor(
        m_Aruid, handle, config, ::nn::sf::NativeHandle(transferMemoryHandle, true), size));

    m_ImageTransferProcessorFormat[playerNumber] =
        static_cast<ImageTransferProcessorFormat>(config.format);
    m_UsesNegativeImage[playerNumber] =
        config.irCameraConfig.isNegativeImageUsed;
    m_IrSensorMode[playerNumber] = IrSensorMode::ImageTransfer;
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorSession::RunImageTransferProcessor(
    const IrCameraHandle& handle,
    const PackedImageTransferProcessorExConfig& config,
    void* workBuffer,
    size_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL(m_FunctionLevel, IrSensorFunctionLevel_1);
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);

    if (m_IrSensorMode[playerNumber] != IrSensorMode::None)
    {
        // プロセッサ起動中に呼び出された場合は一旦中断する。
        NN_RESULT_DO(m_Session->SuspendImageProcessor(m_Aruid, handle));
    }

    ::nn::os::TransferMemoryType transferMemory;
    NN_RESULT_THROW_UNLESS(::nn::os::CreateTransferMemory(
        &transferMemory,
        workBuffer,
        size,
        ::nn::os::MemoryPermission_None).IsSuccess(),
        ResultIrsensorCreateTransferMemoryFailed());

    auto transferMemoryHandle = ::nn::os::DetachTransferMemory(&transferMemory);
    ::nn::os::DestroyTransferMemory(&transferMemory);

    NN_RESULT_DO(m_Session->RunImageTransferExProcessor(
        m_Aruid, handle, config, ::nn::sf::NativeHandle(transferMemoryHandle, true), size));

    m_ImageTransferProcessorFormat[playerNumber] =
        static_cast<ImageTransferProcessorFormat>(config.trimmingFormat);
    m_UsesNegativeImage[playerNumber] =
        config.irCameraConfig.isNegativeImageUsed;
    m_IrSensorMode[playerNumber] = IrSensorMode::ImageTransfer;
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorSession::RunPointingProcessor(
    const IrCameraHandle& handle,
    const PackedPointingProcessorConfig& config
    ) NN_NOEXCEPT
{
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);

    if (m_FunctionLevel >= IrSensorFunctionLevel_1)
    {
        if ((m_IrSensorMode[playerNumber] != IrSensorMode::None)
            && (m_IrSensorMode[playerNumber] != IrSensorMode::Pointing))
        {
            // 指定のプロセッサ以外のプロセッサ起動中に呼び出された場合は一旦中断する。
            NN_RESULT_DO(m_Session->SuspendImageProcessor(m_Aruid, handle));
        }
    }
    else
    {
        NN_SDK_REQUIRES_EQUAL(m_IrSensorMode[playerNumber], IrSensorMode::None);
    }

    // 同じプロセッサへの遷移の場合はプロセス内部で判別して、モード遷移を行わない。
    NN_RESULT_DO(m_Session->RunPointingProcessor(m_Aruid, handle, config));

    m_IrSensorMode[playerNumber] = IrSensorMode::Pointing;
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorSession::RunTeraPluginProcessor(
    const IrCameraHandle& handle,
    const PackedTeraPluginProcessorConfig& config
    ) NN_NOEXCEPT
{
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);

    if (m_FunctionLevel >= IrSensorFunctionLevel_1)
    {
        if (m_IrSensorMode[playerNumber] != IrSensorMode::None)
        {
            // プロセッサ起動中に呼び出された場合は一旦中断する。
            NN_RESULT_DO(m_Session->SuspendImageProcessor(m_Aruid, handle));
        }
    }
    else
    {
        NN_SDK_REQUIRES_EQUAL(m_IrSensorMode[playerNumber], IrSensorMode::None);
    }

    // FunctionLevel と使用可能なモードの制限
    switch(m_FunctionLevel)
    {
    case IrSensorFunctionLevel_0:
        {
            // 7 : HandAnalysisModeSilhouetteAndImage_v2 - static_cast<int>(nn::xcd::IrProcessorType::TeraPlugin);
            NN_SDK_ASSERT_LESS_EQUAL(config.mode, 7);
        }
        break;
    case IrSensorFunctionLevel_1:
        {
            // 4.0.0 NUP 相当の場合は
            // 11: HandAnalysisModeSilhouetteOnly_v2 + 1 - static_cast<int>(nn::xcd::IrProcessorType::TeraPlugin);
            NN_SDK_ASSERT_LESS_EQUAL(config.mode, 11);
        }
        break;
    case IrSensorFunctionLevel_2:
        {
            // 最新版は無制限
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_DO(m_Session->RunTeraPluginProcessor(m_Aruid, handle, config));

    m_IrSensorMode[playerNumber] = IrSensorMode::TeraPlugin;
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorSession::RunIrLedProcessor(
    const IrCameraHandle& handle,
    const PackedIrLedProcessorConfig& config
    ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL(m_FunctionLevel, IrSensorFunctionLevel_1);

    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);

    if ((m_IrSensorMode[playerNumber] != IrSensorMode::None)
        && (m_IrSensorMode[playerNumber] != IrSensorMode::IrLed))
    {
        // プロセッサ起動中に呼び出された場合は一旦中断する。
        NN_RESULT_DO(m_Session->SuspendImageProcessor(m_Aruid, handle));
    }

    NN_RESULT_DO(m_Session->RunIrLedProcessor(m_Aruid, handle, config));

    m_IrSensorMode[playerNumber] = IrSensorMode::IrLed;
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorSession::GetImageTransferState(
    const IrCameraHandle& handle,
    ImageTransferProcessorState* pOutState,
    void* pOutImage,
    size_t size) NN_NOEXCEPT
{
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
    NN_SDK_REQUIRES_NOT_NULL(pOutState);
    NN_SDK_REQUIRES_NOT_NULL(pOutImage);

    NN_RESULT_THROW_UNLESS(m_IrSensorMode[playerNumber] == IrSensorMode::ImageTransfer,
        ::nn::irsensor::ResultIrsensorNotReady());

    NN_RESULT_DO(m_Session->GetImageTransferProcessorState(
        m_Aruid,
        pOutState,
        ::nn::sf::OutBuffer(reinterpret_cast<char*>(pOutImage), size),
        handle));

    // ImageTransfer は白黒反転に対応していないため CPU で白黒反転する
    if (m_UsesNegativeImage[playerNumber])
    {
        auto requiredSize = GetImageSize(m_ImageTransferProcessorFormat[playerNumber]);
        NN_SDK_REQUIRES_GREATER_EQUAL(size, requiredSize);
        auto buffer = reinterpret_cast<uint8_t*>(pOutImage);
        for (auto i = static_cast<size_t>(0); i < requiredSize; ++i)
        {
            buffer[i] = ::nn::irsensor::IrCameraIntensityMax - buffer[i];
        }
    }

    NN_RESULT_SUCCESS;
}

void IrSensorSession::GetMomentProcessorStateDefault(
    MomentProcessorState* pOutValue,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    auto& state = *pOutValue;

    state.samplingNumber = 0;
    state.timeStamp = ::nn::TimeSpan::FromMicroSeconds(0);
    state.ambientNoiseLevel = IrCameraAmbientNoiseLevel_Unknown;
    for (auto i = 0; i < MomentProcessorBlockRowCount; ++i)
    {
        for (auto j = 0; j < MomentProcessorBlockColumnCount; ++j)
        {
            auto& block = state.blocks[i * MomentProcessorBlockColumnCount + j];
            if (m_UsesNegativeImage[playerNumber])
            {
                auto blockWidth = m_WindowOfInterest[playerNumber].width / MomentProcessorBlockColumnCount;
                auto blockHeight = m_WindowOfInterest[playerNumber].height / MomentProcessorBlockRowCount;
                block.averageIntensity = ::nn::irsensor::IrCameraIntensityMax;
                block.centroid.x = ((blockWidth * j) + (blockWidth * (j + 1))) / 2.0f;
                block.centroid.y = ((blockHeight * i) + (blockHeight * (i + 1))) / 2.0f;
            }
            else
            {
                block.averageIntensity = 0;
                block.centroid.x = 0;
                block.centroid.y = 0;
            }
        }
    }
}

void IrSensorSession::GetClusteringProcessorStateDefault(
    ClusteringProcessorState* pOutValue,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_UNUSED(handle);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    auto& state = *pOutValue;

    state.samplingNumber = 0;
    state.timeStamp = ::nn::TimeSpan::FromMicroSeconds(0);
    state.objectCount = 0;
    state.ambientNoiseLevel = IrCameraAmbientNoiseLevel_Unknown;
}

void IrSensorSession::GetImageTransferProcessorStateDefault(
    ImageTransferProcessorState* pOutValue,
    void* pOutImage,
    size_t size,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
    NN_UNUSED(size);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    auto& state = *pOutValue;

    state.samplingNumber = 0;
    state.ambientNoiseLevel = IrCameraAmbientNoiseLevel_Unknown;

    auto requiredSize = GetImageSize(m_ImageTransferProcessorFormat[playerNumber]);
    NN_SDK_REQUIRES_GREATER_EQUAL(size, requiredSize);
    auto defaultValue = static_cast<uint8_t>(m_UsesNegativeImage[playerNumber]? ::nn::irsensor::IrCameraIntensityMax: 0);
    auto buffer = reinterpret_cast<uint8_t*>(pOutImage);
    for (auto i = static_cast<size_t>(0); i < requiredSize; ++i)
    {
        buffer[i] = defaultValue;
    }
}

size_t IrSensorSession::GetImageSize(ImageTransferProcessorFormat format) NN_NOEXCEPT
{
    switch (format)
    {
    case ImageTransferProcessorFormat_320x240:
        return ImageTransferProcessorImageSize320x240;
    case ImageTransferProcessorFormat_160x120:
        return ImageTransferProcessorImageSize160x120;
    case ImageTransferProcessorFormat_80x60:
        return ImageTransferProcessorImageSize80x60;
    case ImageTransferProcessorFormat_40x30:
        return ImageTransferProcessorImageSize40x30;
    case ImageTransferProcessorFormat_20x15:
        return ImageTransferProcessorImageSize20x15;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

::nn::Result IrSensorSession::ThrowResultActivationUpperLimitOver(
    ) NN_NOEXCEPT
{
    NN_RESULT_THROW(
        ResultIrsensorActivationUpperLimitOver());
}

::nn::Result IrSensorSession::ThrowResultDeactivationLowerLimitOver(
    ) NN_NOEXCEPT
{
    NN_RESULT_THROW(
        ResultIrsensorDeactivationLowerLimitOver());
}

::nn::Result IrSensorSession::ThrowResultAllocationFailed() NN_NOEXCEPT
{
    NN_RESULT_THROW(ResultIrsensorLifoAllocationFailed());
}

::nn::Result IrSensorSession::CreateProxy() NN_NOEXCEPT
{
    NN_RESULT_THROW(CreateIrSensorServerProxy(&m_Session));
}

void IrSensorSession::DestroyProxy() NN_NOEXCEPT
{
    m_Session.Reset();
}

::nn::Result IrSensorSession::ActivateService() NN_NOEXCEPT
{
    NN_RESULT_THROW(m_Session->ActivateIrsensor(m_Aruid));
}

::nn::Result IrSensorSession::ActivateServiceWithFunctionLevel(const IrSensorFunctionLevel& functionLevel) NN_NOEXCEPT
{
    PackedFunctionLevel packedFunctionLevel = {};
    packedFunctionLevel.level = static_cast<int8_t>(functionLevel);
    NN_RESULT_THROW(m_Session->ActivateIrsensorWithFunctionLevel(m_Aruid, packedFunctionLevel));
}

::nn::Result IrSensorSession::DeactivateService() NN_NOEXCEPT
{
    NN_RESULT_THROW(m_Session->DeactivateIrsensor(m_Aruid));
}

::nn::Result IrSensorSession::GetSharedMemoryHandle(
    ::nn::sf::NativeHandle* outValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_RESULT_THROW(
        m_Session->GetIrsensorSharedMemoryHandle(outValue, m_Aruid));
}

bool IrSensorSession::IsSharedMemoryMapped() NN_NOEXCEPT
{
    return m_StatusManagerHolder.IsSharedMemoryMapped();
}

void IrSensorSession::AttachSharedMemory(::nn::os::NativeHandle handle,
                                bool managed) NN_NOEXCEPT
{
    m_StatusManagerHolder.Attach(handle, managed);
}

void IrSensorSession::FinalizeSharedMemory() NN_NOEXCEPT
{
    m_StatusManagerHolder.Finalize();
}

}}} // namespace nn::irsensor::detail
