﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/irsensor/irsensor_IIrSensorServer.h>
#include <nn/sf/sf_Buffers.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_NativeHandle.h>

namespace nn { namespace irsensor { namespace detail {

//!< irs サービスを扱うクラスです。
class IrSensorServer final
{
    NN_DISALLOW_COPY(IrSensorServer);
    NN_DISALLOW_MOVE(IrSensorServer);

public:
    IrSensorServer() NN_NOEXCEPT;

    ~IrSensorServer() NN_NOEXCEPT;

    //!< IR センサーのハンドルを取得します。
    ::nn::Result GetNpadIrCameraHandle(
        ::nn::sf::Out<IrCameraHandle> outValue,
        ::nn::hid::NpadIdType npadIdType) NN_NOEXCEPT;

    //!< IR センサーをアクティブ化します。
    ::nn::Result ActivateIrsensor(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< IR センサーを FunctionLevel を設定してアクティブ化します。
    ::nn::Result ActivateIrsensorWithFunctionLevel(
        ::nn::applet::AppletResourceUserId aruid,
        const PackedFunctionLevel& functionLevel) NN_NOEXCEPT;

    //!< IR センサーを非アクティブ化します。
    ::nn::Result DeactivateIrsensor(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< IR センサーの LIFO のハンドルを取得します。
    ::nn::Result GetIrsensorSharedMemoryHandle(
        ::nn::sf::Out<::nn::sf::NativeHandle> outValue,
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< IR センサーのバージョンチェックをします。
    ::nn::Result CheckFirmwareVersion(
        ::nn::applet::AppletResourceUserId aruid,
        IrCameraHandle handle,
        const PackedMcuVersion& requiredVersion) NN_NOEXCEPT;

    //!< IR センサーの FunctionLevel を設定します。
    ::nn::Result SetFunctionLevel(
        ::nn::applet::AppletResourceUserId aruid,
        IrCameraHandle handle,
        const PackedFunctionLevel& functionLevel) NN_NOEXCEPT;

    //!< IR センサーの処理を同期的に停止します。
    ::nn::Result StopImageProcessor(
        ::nn::applet::AppletResourceUserId aruid,
        IrCameraHandle handle) NN_NOEXCEPT;

    //!< IR センサーの処理を非同期に停止します。
    ::nn::Result StopImageProcessorAsync(
        ::nn::applet::AppletResourceUserId aruid,
        IrCameraHandle handle) NN_NOEXCEPT;

    //!< IR センサーの処理を中断します。
    ::nn::Result SuspendImageProcessor(
        ::nn::applet::AppletResourceUserId aruid,
        IrCameraHandle handle) NN_NOEXCEPT;

    //!< IR センサーをモーメントプロセッサを利用して処理を開始します。
    ::nn::Result RunMomentProcessor(
        ::nn::applet::AppletResourceUserId aruid,
        IrCameraHandle handle,
        const PackedMomentProcessorConfig& config
        ) NN_NOEXCEPT;

    //!< IR センサーをクラスタリングプロセッサを利用して処理を開始します。
    ::nn::Result RunClusteringProcessor(
        ::nn::applet::AppletResourceUserId aruid,
        IrCameraHandle handle,
        const PackedClusteringProcessorConfig& config
        ) NN_NOEXCEPT;

    //!< IR センサーをイメージ転送プロセッサを利用して処理を開始します。
    ::nn::Result RunImageTransferProcessor(
        ::nn::applet::AppletResourceUserId aruid,
        IrCameraHandle handle,
        const nn::irsensor::PackedImageTransferProcessorConfig& config,
        ::nn::sf::NativeHandle&& transferMemoryHandle,
        uint64_t size) NN_NOEXCEPT;

    //!< IR センサーをイメージ転送プロセッサを利用して処理を開始します。(詳細設定用)
    ::nn::Result RunImageTransferExProcessor(
        ::nn::applet::AppletResourceUserId aruid,
        IrCameraHandle handle,
        const nn::irsensor::PackedImageTransferProcessorExConfig& config,
        ::nn::sf::NativeHandle&& transferMemoryHandle,
        uint64_t size) NN_NOEXCEPT;

    //!< イメージ転送プロセッサの処理結果を取得します。
    nn::Result GetImageTransferProcessorState(
        ::nn::applet::AppletResourceUserId aruid,
        ::nn::sf::Out<nn::irsensor::ImageTransferProcessorState> outState,
        const nn::sf::OutBuffer& outBuffer,
        IrCameraHandle handle) NN_NOEXCEPT;

    //!< IR センサーを Pointing プロセッサを利用して処理を開始します。
    ::nn::Result RunPointingProcessor(
        ::nn::applet::AppletResourceUserId aruid,
        IrCameraHandle handle,
        const PackedPointingProcessorConfig& config
        ) NN_NOEXCEPT;

    //!< IR センサーを TeraPlugin プロセッサを利用して処理を開始します。
    ::nn::Result RunTeraPluginProcessor(
        ::nn::applet::AppletResourceUserId aruid,
        IrCameraHandle handle,
        PackedTeraPluginProcessorConfig config) NN_NOEXCEPT;

    //!< IR センサーを IrLed プロセッサを利用して処理を開始します。
    ::nn::Result RunIrLedProcessor(
        ::nn::applet::AppletResourceUserId aruid,
        IrCameraHandle handle,
        PackedIrLedProcessorConfig config) NN_NOEXCEPT;
};

//!< irsensor サービスのプロクシを作成します。
::nn::Result CreateIrSensorServerProxy(
    ::nn::sf::SharedPointer<IIrSensorServer>* outValue) NN_NOEXCEPT;

}}} // namespace nn::irsensor::detail
