﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_BitPack.h>

#include "irsensor_IrCameraHandle.h"
#include "../../hid/detail/hid_NpadId.h"

namespace nn { namespace irsensor { namespace detail {

//<! IR センサーのハンドルのビットフィールド定義です。
struct IrCameraHandleField final
{
    //!< IR センサーが搭載されたデバイスのプレイヤー番号
    typedef ::nn::util::BitPack32::Field<0, 8, int8_t> PlayerNumber;
    //!< IR センサーが搭載されたデバイスの種類
    typedef ::nn::util::BitPack32::Field<8, 8, int8_t> DeviceType;
};

void MakeIrCameraHandle(
    IrCameraHandle* outValue,
    const ::nn::hid::NpadIdType& npadIdType) NN_NOEXCEPT
{
    auto playerNumber = ::nn::hid::detail::GetIndexFromNpadIdType(npadIdType);
    NN_SDK_REQUIRES_GREATER_EQUAL(playerNumber, 0);

    auto bitPack = ::nn::util::BitPack32();
    bitPack.Set<IrCameraHandleField::PlayerNumber>(
        static_cast<int8_t>(playerNumber));
    bitPack.Set<IrCameraHandleField::DeviceType>(
        static_cast<int8_t>(IrSensorDeviceType::Npad));

    outValue->_handle = bitPack.storage;
}

IrSensorDeviceType GetIrCameraHandleDeviceType(
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    ::nn::util::BitPack32 bitPack = { handle._handle };
    auto deviceType = bitPack.Get<IrCameraHandleField::DeviceType>();
    NN_SDK_REQUIRES_RANGE(
        deviceType, 0, static_cast<int>(IrSensorDeviceType::TypeCount));
    return static_cast<IrSensorDeviceType>(deviceType);
}

int GetIrCameraHandlePlayerNumber(
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    ::nn::util::BitPack32 bitPack = { handle._handle };
    auto playerNumber = bitPack.Get<IrCameraHandleField::PlayerNumber>();
    NN_SDK_REQUIRES_GREATER_EQUAL(playerNumber, 0);
    return playerNumber;
}

}}} // namespace nn::irsensor::detail
