﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/hidbus/hidbus.h>

#include "../detail/hidbus_RingLifo.h"

namespace nn { namespace hidbus { namespace detail {

const int JoyConEnableSixAxisSensorLifoCount  = 10;
const int JoyConDisableSixAxisSensorLifoCount = 10;

const int JoyConEnableSixAxisSensorMaxPollingDataSize  =  8;
const int JoyConDisableSixAxisSensorMaxPollingDataSize = 38;

struct JoyConEnableSixAxisSensorLifoFormat
{
    uint8_t data[JoyConEnableSixAxisSensorMaxPollingDataSize];
    uint8_t dataSize;
    int64_t samplingNumber;
};

struct JoyConDisableSixAxisSensorLifoFormat
{
    uint8_t data[JoyConDisableSixAxisSensorMaxPollingDataSize];
    uint8_t dataSize;
    int64_t samplingNumber;
};

template<typename PollingDataLifoFormatT, int PollingDataSize>
class PollingDataAccessor final
{
    NN_DISALLOW_COPY(PollingDataAccessor);
    NN_DISALLOW_MOVE(PollingDataAccessor);

private:

    struct PollingDataFormat
    {
        nn::Result                pollingModeResult;
        int64_t                   samplingNumber;
        RingLifo<PollingDataLifoFormatT, PollingDataSize>  pollingData;

        //char _padding[3792];
    };

    //NN_STATIC_ASSERT(sizeof(PollingDataFormat) == 4096U);

    PollingDataFormat m_DataFormat;

public:
    PollingDataAccessor() NN_NOEXCEPT
    {

        m_DataFormat.pollingModeResult   = nn::hidbus::ResultExternalDeviceTimeout();
        m_DataFormat.samplingNumber = 0;
    }

    ~PollingDataAccessor() NN_NOEXCEPT
    {
    }

    nn::Result GetPollingData(PollingDataLifoFormatT* pOutFormat, int count) NN_NOEXCEPT
    {
        m_DataFormat.pollingData.Read(pOutFormat, count);

        // 取り出した最後のデータのサンプリングナンバーが 0 の場合、まだデバイスからの受信データが来てないので、 Timeout 扱いにして上位に返す。
        if (pOutFormat[count - 1].samplingNumber == 0)
        {
            return nn::hidbus::ResultExternalDeviceTimeout();
        }

        return m_DataFormat.pollingModeResult;
    }

    void SetPollingData(uint8_t* pInData, uint16_t dataSize, nn::Result result) NN_NOEXCEPT
    {
        PollingDataLifoFormatT tempBuffer;

        uint16_t cpySize;
        if (sizeof(tempBuffer.data) >= dataSize)
        {
            cpySize = dataSize;
        }
        else
        {
            cpySize = sizeof(tempBuffer.data);
        }

        memcpy(tempBuffer.data, pInData, cpySize);
        tempBuffer.dataSize = static_cast<uint8_t>(cpySize);
        tempBuffer.samplingNumber = m_DataFormat.samplingNumber;
        m_DataFormat.samplingNumber++;
        m_DataFormat.pollingData.Append(tempBuffer);
        m_DataFormat.pollingModeResult = result;
    }
};

using JoyEnableSixAxisPollingDataAccessor = PollingDataAccessor<
                                            JoyConEnableSixAxisSensorLifoFormat,
                                            JoyConEnableSixAxisSensorLifoCount>;

using JoyDisableSixAxisPollingDataAccessor = PollingDataAccessor<
                                             JoyConDisableSixAxisSensorLifoFormat,
                                             JoyConDisableSixAxisSensorLifoCount>;

}}} // namespace nn::hidbus::detail
