﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <new>

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_MemoryFence.h>
#include <nn/os/os_MemoryHeap.h>
#include <nn/os/os_SharedMemory.h>
#include <nn/result/result_HandlingUtility.h>

#include "hidbus_ClientResourceHolder.h"

namespace nn { namespace hidbus { namespace detail {

namespace {

//!< 他プロセスに許可されるメモリアクセス権
const ::nn::os::MemoryPermission OtherPermission =
#if defined(NN_BUILD_CONFIG_OS_WIN)
    // Visual Studio の std::atomic 型が load 時にロックしてしまうため
    ::nn::os::MemoryPermission_ReadWrite;
#else
    ::nn::os::MemoryPermission_ReadOnly;
#endif

} // namespace

ClientResourceHolder::ClientResourceHolder() NN_NOEXCEPT
    : m_IsCreated(false)
    , m_IsSharedMemoryMapped(false)
    , m_IsFwUpdateRunning(false)
{
    m_IsSharedMemoryMappedMutex = NN_OS_SDK_MUTEX_INITIALIZER();
    m_FwUpdateCheckMutex = NN_OS_SDK_MUTEX_INITIALIZER();

    for (int i = 0; i < MaxHidbusNumber; i++)
    {
        m_HandleHolder[i] = InvalidHandle;
        m_ClientMutexForEachIndex[i] = NN_OS_SDK_RECURSIVE_MUTEX_INITIALIZER();
    }
}

ClientResourceHolder::~ClientResourceHolder() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result ClientResourceHolder::Clear() NN_NOEXCEPT
{
    if (m_IsCreated)
    {
        auto pAccessor = GetStatusManager();
        // コンストラクタ呼び出しで、データをクリア
        new (pAccessor) StatusManager();
        // 共有メモリに対するアクセスの完了を保証
        ::nn::os::FenceMemoryStoreLoad();
    }
    NN_RESULT_SUCCESS;
}

void ClientResourceHolder::Attach(
    ::nn::os::NativeHandle handle, bool managed) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_IsSharedMemoryMappedMutex)> locker(m_IsSharedMemoryMappedMutex);
    ::nn::os::AttachSharedMemory(&m_SharedMemory,
                                 ::nn::os::MemoryPageSize,
                                 handle,
                                 managed);

    ::nn::os::MapSharedMemory(&m_SharedMemory, OtherPermission);

    m_IsSharedMemoryMapped = true;
    m_IsCreated = true;
}

void ClientResourceHolder::Detach() NN_NOEXCEPT
{
    if (m_IsCreated)
    {
        auto pAccessor = GetStatusManager();
        pAccessor->~StatusManager();
        pAccessor = nullptr;

        // 共有メモリに対するアクセスの完了を保証
        ::nn::os::FenceMemoryAnyStore();
    }

    m_IsCreated = false;

    ::std::lock_guard<decltype(m_IsSharedMemoryMappedMutex)> locker(m_IsSharedMemoryMappedMutex);

    if (::nn::os::GetSharedMemoryAddress(&m_SharedMemory) != nullptr)
    {
        ::nn::os::UnmapSharedMemory(&m_SharedMemory);
    }

    ::nn::os::DestroySharedMemory(&m_SharedMemory);
    m_IsSharedMemoryMapped = false;
}

size_t ClientResourceHolder::GetSharedMemorySize() const NN_NOEXCEPT
{
    return ::nn::os::GetSharedMemorySize(&m_SharedMemory);
}

::nn::os::NativeHandle ClientResourceHolder::GetSharedMemoryHandle(
    ) const NN_NOEXCEPT
{
    return ::nn::os::GetSharedMemoryHandle(&m_SharedMemory);
}

StatusManager* ClientResourceHolder::GetStatusManager() const NN_NOEXCEPT
{
    auto pAccessor = reinterpret_cast<StatusManager*>(
        ::nn::os::GetSharedMemoryAddress(&m_SharedMemory));
    NN_SDK_REQUIRES_NOT_NULL(pAccessor);

    return pAccessor;
}

bool ClientResourceHolder::IsSharedMemoryMapped() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_IsSharedMemoryMappedMutex)> locker(m_IsSharedMemoryMappedMutex);

    return m_IsSharedMemoryMapped;
}

void ClientResourceHolder::SetTransferdBuffer(void* pBuffer, int index) NN_NOEXCEPT
{
    m_pTransferdBuffer[index] = pBuffer;
}

template<typename PollingDataLifoFormatT, int PollingDataSize>
nn::hidbus::detail::PollingDataAccessor<PollingDataLifoFormatT, PollingDataSize>* ClientResourceHolder::GetPollingDataAccessor(int index) NN_NOEXCEPT
{
    return reinterpret_cast<PollingDataAccessor<PollingDataLifoFormatT, PollingDataSize>*>(m_pTransferdBuffer[index]);
}

ClientResourceHolder& GetClientResourceHolder() NN_NOEXCEPT
{
    static ClientResourceHolder s_ClientResourceHolder;
    return s_ClientResourceHolder;
}

nn::hidbus::detail::JoyEnableSixAxisPollingDataAccessor* GetJoyEnableSixAxisPollingDataAccessor(int index) NN_NOEXCEPT
{
    return GetClientResourceHolder().GetPollingDataAccessor<
        detail::JoyConEnableSixAxisSensorLifoFormat,
        detail::JoyConEnableSixAxisSensorLifoCount>(index);
}

nn::hidbus::detail::JoyDisableSixAxisPollingDataAccessor* GetJoyDisableSixAxisPollingDataAccessor(int index) NN_NOEXCEPT
{
    return GetClientResourceHolder().GetPollingDataAccessor<
        detail::JoyConDisableSixAxisSensorLifoFormat,
        detail::JoyConDisableSixAxisSensorLifoCount>(index);
}

}}} // namespace nn::hidbus::detail
