﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/os/os_Mutex.h>
#include <nn/hid/hid_VibrationPlayer.h>
#include "detail/hid_LockableMutexType.h"

namespace {

//!< VibrationPlayer 内の排他処理に用いるミューテックス
nn::hid::detail::LockableMutexType s_VibrationPlayerMutex = { NN_OS_MUTEX_INITIALIZER(false) };

}

namespace nn { namespace hid {

VibrationPlayer::VibrationPlayer() NN_NOEXCEPT :
    m_IsLoaded(false),
    m_IsPlaying(false),
    m_IsLoop(false),
    m_CurrentPosition(0),
    m_NextPosition(0),
    m_LoopStartPosition(0),
    m_LoopEndPosition(0),
    m_LoopInterval(0) {}

VibrationPlayer::~VibrationPlayer() NN_NOEXCEPT {}

Result VibrationPlayer::Load(const void* address, size_t fileSize) NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationPlayerMutex)> lock(s_VibrationPlayerMutex);
    m_IsLoaded = false;
    Result result = nn::hid::ParseVibrationFile(&m_FileInfo, &m_ParserContext, address, fileSize);
    if(result.IsSuccess())
    {
        m_IsLoaded = true;
        m_IsPlaying = false;
        m_IsLoop = (m_FileInfo.isLoop != 0);
        m_CurrentPosition = 0;
        m_NextPosition = 0;
        SetCurrentVibration(VibrationValue::Make());
        m_LoopStartPosition = m_FileInfo.loopStartPosition;
        m_LoopEndPosition = m_FileInfo.loopEndPosition;
        m_LoopInterval = m_FileInfo.loopInterval;
    }
    return result;
}

void VibrationPlayer::Unload() NN_NOEXCEPT
{
    m_IsLoaded = false;
}

bool VibrationPlayer::IsLoaded() const NN_NOEXCEPT
{
    return m_IsLoaded;
}

VibrationFileInfo VibrationPlayer::GetFileInfo() const NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationPlayerMutex)> lock(s_VibrationPlayerMutex);
    return m_FileInfo;
}

void VibrationPlayer::Play() NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationPlayerMutex)> lock(s_VibrationPlayerMutex);
    if(IsLoaded())
    {
        m_IsPlaying = true;
    }
}

void VibrationPlayer::Stop() NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationPlayerMutex)> lock(s_VibrationPlayerMutex);
    if(IsLoaded())
    {
        m_IsPlaying = false;
        m_CurrentPosition = m_NextPosition;
        SetCurrentVibration(VibrationValue::Make());
    }
}

bool VibrationPlayer::IsPlaying() const NN_NOEXCEPT
{
    return m_IsPlaying;
}

void VibrationPlayer::SetCurrentPosition(int position) NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationPlayerMutex)> lock(s_VibrationPlayerMutex);
    if(IsLoaded())
    {
        m_CurrentPosition = position;
        m_NextPosition = position;
        if(m_IsPlaying)
        {
            VibrationValue v;
            nn::hid::RetrieveVibrationValue(&v, position, &m_ParserContext);
            SetCurrentVibration(v);
        }
    }
}

int VibrationPlayer::GetCurrentPosition() const NN_NOEXCEPT
{
    return m_CurrentPosition;
}

void VibrationPlayer::SetLoop(bool isLoop) NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationPlayerMutex)> lock(s_VibrationPlayerMutex);
    if(IsLoaded())
    {
        m_IsLoop = isLoop;
    }
}

bool VibrationPlayer::IsLoop() const NN_NOEXCEPT
{
    return m_IsLoop;
}

void VibrationPlayer::SetLoopRange(int loopStartPosition, int loopEndPosition) NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationPlayerMutex)> lock(s_VibrationPlayerMutex);
    if(IsLoaded())
    {
        m_LoopStartPosition = loopStartPosition;
        m_LoopEndPosition = loopEndPosition;
    }
}

void VibrationPlayer::SetLoopStartPosition(int position) NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationPlayerMutex)> lock(s_VibrationPlayerMutex);
    if(IsLoaded())
    {
        m_LoopStartPosition = position;
    }
}

int VibrationPlayer::GetLoopStartPosition() const NN_NOEXCEPT
{
    return m_LoopStartPosition;
}

void VibrationPlayer::SetLoopEndPosition(int position) NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationPlayerMutex)> lock(s_VibrationPlayerMutex);
    if(IsLoaded())
    {
        m_LoopEndPosition = position;
    }
}

int VibrationPlayer::GetLoopEndPosition() const NN_NOEXCEPT
{
    return m_LoopEndPosition;
}

void VibrationPlayer::SetLoopInterval(int interval) NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationPlayerMutex)> lock(s_VibrationPlayerMutex);
    if(IsLoaded())
    {
        m_LoopInterval = interval;
    }
}

int VibrationPlayer::GetLoopInterval() const NN_NOEXCEPT
{
    return m_LoopInterval;
}

void VibrationPlayer::OnNextSampleRequired(
    VibrationValue* pValue,
    VibrationNodeConnection::List* pInputConnections) NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationPlayerMutex)> lock(s_VibrationPlayerMutex);

    // 接続元の振動値は利用しません
    NN_UNUSED(pInputConnections);

    if(!m_IsLoaded)
    {
        *pValue = VibrationValue::Make();
        return;
    }

    // m_NextPosition の位置をループモードに応じて変更します
    if(m_IsPlaying)
    {
        if(m_IsLoop)
        {
            // ループ終了位置に達したらループインターバル開始位置またはループ開始位置に飛びます
            if(m_NextPosition < -m_LoopInterval || m_LoopEndPosition <= m_NextPosition)
            {
                if(m_LoopInterval > 0)
                {
                    m_NextPosition = -m_LoopInterval;
                }
                else
                {
                    m_NextPosition = m_LoopStartPosition;
                }
            }
        }
        else
        {
            // ファイル末尾に達したらファイル先頭に飛んで自動停止します
            if(m_FileInfo.sampleLength <= m_NextPosition)
            {
                m_CurrentPosition = 0;
                m_NextPosition = 0;
                m_IsPlaying = false;
                *pValue = VibrationValue::Make();
            }
        }

        // isLoop に関わらずループインターバルが終了したらループ開始位置に飛びます
        if(m_CurrentPosition < 0 && m_NextPosition == 0)
        {
            m_NextPosition = m_LoopStartPosition;
        }
    }

    if(m_IsPlaying)
    {
        if(0 <= m_NextPosition && m_NextPosition < m_FileInfo.sampleLength)
        {
            // 振動ファイルから振動値をとりだす処理
            nn::hid::RetrieveVibrationValue(pValue, m_NextPosition, &m_ParserContext);
        }
        else
        {
            *pValue = VibrationValue::Make();
        }

        m_CurrentPosition = m_NextPosition;
        m_NextPosition = m_NextPosition + 1;
    }
}

}} // namespace nn::hid
