﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/hid/system/hid_Nfc.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_NativeHandle.h>

#include "detail/hid_NfcImpl.h"
#include "detail/hid_NpadSystemImpl.h"

namespace nn { namespace hid { namespace system {


void BindNfcDeviceUpdateEvent(::nn::os::SystemEventType* pEvent,
                              ::nn::os::EventClearMode clearMode) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::BindNfcDeviceUpdateEvent(pEvent, clearMode));
}

int GetNpadsWithNfc(NpadIdType* pOutValue, int length) NN_NOEXCEPT
{
    int outCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::GetNpadsWithNfc(&outCount, pOutValue, length));

    return outCount;
}

Result GetXcdHandleForNpadWithNfc(nn::xcd::DeviceHandle* pOutHandle, NpadIdType npadId) NN_NOEXCEPT
{
    auto result = detail::GetXcdHandleForNpadWithNfc(pOutHandle, npadId);

    if(result.IsFailure() == true &&
       ResultNoNfcDeviceFoundOnNpad::Includes(result) == false)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    return result;
}

void BindNfcActivateEvent(NpadIdType npadId,
                          ::nn::os::SystemEventType* pEvent,
                          ::nn::os::EventClearMode clearMode) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::BindNfcActivateEvent(npadId, pEvent, clearMode));
}

bool IsNfcActivated(NpadIdType npadId) NN_NOEXCEPT
{
    bool isActivated;

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::IsNfcActivated(&isActivated, npadId));

    return isActivated;
}

Result ActivateNfc(NpadIdType npadId) NN_NOEXCEPT
{
    NN_RESULT_DO(ActivateNfc(npadId, ::nn::applet::AppletResourceUserId::GetInvalidId()));

    NN_RESULT_SUCCESS;
}

Result DeactivateNfc(NpadIdType npadId) NN_NOEXCEPT
{
    NN_RESULT_DO(DeactivateNfc(npadId, ::nn::applet::AppletResourceUserId::GetInvalidId()));

    NN_RESULT_SUCCESS;
}

Result ActivateNfc(NpadIdType npadId, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    auto result = detail::ActivateNfc(npadId, aruid);

    if(result.IsFailure() == true &&
       ResultNoNfcDeviceFoundOnNpad::Includes(result) == false &&
       ResultNfcActivatedOnOtherNpad::Includes(result) == false &&
       ResultNfcActivateFailureNpadBusy::Includes(result) == false)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    return result;
}

Result DeactivateNfc(NpadIdType npadId, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    auto result = detail::DeactivateNfc(npadId, aruid);

    if(result.IsFailure() == true &&
       ResultNoNfcDeviceFoundOnNpad::Includes(result) == false)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    return result;
}

bool GetXcdDeviceHandleForNfc(nn::xcd::DeviceHandle* pOutValue) NN_NOEXCEPT
{
    NN_UNUSED(pOutValue);
    return false;
}

}}} // namespace nn::hid::system

