﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_Xpad.h>
#include <nn/hid/hid_JoyXpad.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SystemEvent.h>

namespace nn { namespace hid { namespace detail {

//!< Xpad ID の解決を担うマネージャを扱うクラスです。
class XpadIdManager final
{
    NN_DISALLOW_COPY(XpadIdManager);
    NN_DISALLOW_MOVE(XpadIdManager);

private:
    //!< Xpad ID 再解決イベント
    ::nn::os::SystemEvent m_XpadIdSystemEvent;

    //!< Xpad ID 再解決イベントの通知先の ID
    uint64_t m_XpadIdSystemEventClientId;

    // 基本的な構成を持つ Xpad の解決された ID の数
    int m_BasicXpadIdCount;

    // 基本的な構成を持つ Xpad の解決された ID
    BasicXpadId m_BasicXpadIds[XpadIdCountMax];

    // JoyXpad の解決された ID の数
    int m_JoyXpadIdCount;

    // JoyXpad の解決された ID
    JoyXpadId m_JoyXpadIds[XpadIdCountMax];

public:
    XpadIdManager() NN_NOEXCEPT;

    ~XpadIdManager() NN_NOEXCEPT;

    //!< Xpad ID の再解決イベントをバインドします。
    ::nn::Result AcquireXpadIdEventHandle(::nn::os::NativeHandle* outHandle,
                                          uint64_t clientId) NN_NOEXCEPT;

    //!< Xpad ID の再解決イベントのバインドを解除します。
    ::nn::Result ReleaseXpadIdEventHandle(uint64_t clientId) NN_NOEXCEPT;

    //!< Xpad ID 再解決イベントをシグナル状態にします。
    void SignalXpadIdSystemEvent() NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad の解決された ID を取得します。
    ::nn::Result GetBasicXpadIds(
        int* outCount, BasicXpadId* outXpadIds, int count) NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad の解決された ID を設定します。
    void SetBasicXpadIds(const BasicXpadId* outXpadIds, int count) NN_NOEXCEPT;

    //!< JoyXpad の解決された ID を取得します。
    ::nn::Result GetJoyXpadIds(
        int* outCount, JoyXpadId* outXpadIds, int count) NN_NOEXCEPT;

    //!< JoyXpad の解決された ID を設定します。
    void SetJoyXpadIds(const JoyXpadId* outXpadIds, int count) NN_NOEXCEPT;

};

}}} // namespace nn::hid::detail
