﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_XpadIdManager.h"

namespace nn { namespace hid { namespace detail {

XpadIdManager::XpadIdManager() NN_NOEXCEPT
    : m_XpadIdSystemEvent(::nn::os::EventClearMode_ManualClear, true)
    , m_XpadIdSystemEventClientId(0)
    , m_BasicXpadIdCount(0)
    , m_JoyXpadIdCount(0)
{
    // 何もしない
}

XpadIdManager::~XpadIdManager() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result XpadIdManager::AcquireXpadIdEventHandle(
    ::nn::os::NativeHandle* outHandle,
    uint64_t clientId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outHandle);
    NN_RESULT_THROW_UNLESS(clientId != 0, ResultXpadIdEventInvalidArgument());

    // Xpad ID の再解決通知がバインド済み
    NN_RESULT_THROW_UNLESS(m_XpadIdSystemEventClientId == 0,
                           ResultXpadIdEventAlreadyBound());

    m_XpadIdSystemEventClientId = clientId;
    *outHandle = m_XpadIdSystemEvent.GetReadableHandle();

    NN_RESULT_SUCCESS;
}

::nn::Result XpadIdManager::ReleaseXpadIdEventHandle(uint64_t clientId
                                                     ) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(clientId != 0, ResultXpadIdEventInvalidArgument());

    // Xpad ID の再解決通知が未バインド
    NN_RESULT_THROW_UNLESS(m_XpadIdSystemEventClientId == clientId,
                           ResultXpadIdEventNotBound());

    m_XpadIdSystemEventClientId = 0;

    NN_RESULT_SUCCESS;
}

void XpadIdManager::SignalXpadIdSystemEvent() NN_NOEXCEPT
{
    m_XpadIdSystemEvent.Signal();
}

::nn::Result XpadIdManager::GetBasicXpadIds(int* outCount,
                                            BasicXpadId* outXpadIds,
                                            int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(count, 0);
    NN_RESULT_THROW_UNLESS(outCount != nullptr,
                           ResultBasicXpadNullIdCountBuffer());
    NN_RESULT_THROW_UNLESS(outXpadIds != nullptr,
                           ResultBasicXpadNullIdBuffer());

    const int minCount = ::std::min(::std::max(0, count), m_BasicXpadIdCount);

    for (int i = 0; i< minCount; ++i)
    {
        outXpadIds[i] = m_BasicXpadIds[i];
    }

    *outCount = minCount;

    NN_RESULT_SUCCESS;
}

void XpadIdManager::SetBasicXpadIds(const BasicXpadId* outXpadIds,
                                    int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outXpadIds);

    const int minCount = ::std::min(
        ::std::max(0, count),
        static_cast<int>(sizeof(m_BasicXpadIds) / sizeof(m_BasicXpadIds[0])));

    for (int i = 0; i< minCount; ++i)
    {
        m_BasicXpadIds[i] = outXpadIds[i];
    }

    m_BasicXpadIdCount = minCount;
}

::nn::Result XpadIdManager::GetJoyXpadIds(int* outCount,
                                          JoyXpadId* outXpadIds,
                                          int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(count, 0);
    NN_RESULT_THROW_UNLESS(outCount != nullptr,
                           ResultBasicXpadNullIdCountBuffer());
    NN_RESULT_THROW_UNLESS(outXpadIds != nullptr,
                           ResultBasicXpadNullIdBuffer());

    const int minCount = ::std::min(::std::max(0, count), m_JoyXpadIdCount);

    for (int i = 0; i< minCount; ++i)
    {
        outXpadIds[i] = m_JoyXpadIds[i];
    }

    *outCount = minCount;

    NN_RESULT_SUCCESS;
}

void XpadIdManager::SetJoyXpadIds(const JoyXpadId* outXpadIds,
                                  int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outXpadIds);

    const int minCount = ::std::min(
        ::std::max(0, count),
        static_cast<int>(sizeof(m_JoyXpadIds) / sizeof(m_JoyXpadIds[0])));

    for (int i = 0; i< minCount; ++i)
    {
        m_JoyXpadIds[i] = outXpadIds[i];
    }

    m_JoyXpadIdCount = minCount;
}
}}} // namespace nn::hid::detail
