﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>
#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Windows.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_WindowsHidSdi-os.win.h"
#include "hid_WindowsRawInput-os.win.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< 入力デバイスの数の最大値です。
const int RawInputDeviceCountMax = 256;

//!< Manufacture Id の定義です。
enum class ManufactureId
{
    Nintendo = 0x057e,
};

//!< XInput 対応コントローラか否かを表す値を返します。
bool IsXInput(char name[]) NN_NOEXCEPT;

//!< 入力デバイスを取得します。
::nn::Result GetRawInputDevices(
    int* pOutCount, RAWINPUTDEVICELIST outDevices[], int count) NN_NOEXCEPT;

//!< 入力デバイスの情報を取得します。
::nn::Result GetRawInputDeviceInformation(
    HANDLE handle, RID_DEVICE_INFO* pOutValue) NN_NOEXCEPT;

//!< 入力デバイスの名前を取得します。
::nn::Result GetRawInputDeviceName(
    HANDLE handle, char* outName, int count) NN_NOEXCEPT;

} // namespace

int GetRawInputPhysicalKeyboardCount() NN_NOEXCEPT
{
    static ::nn::os::SdkMutexType s_Mutex = NN_OS_SDK_MUTEX_INITIALIZER();

    ::std::lock_guard<decltype(s_Mutex)> locker(s_Mutex);

    static RAWINPUTDEVICELIST s_Devices[RawInputDeviceCountMax];

    auto deviceCount = int();

    if (GetRawInputDevices(
            &deviceCount, s_Devices, NN_ARRAY_SIZE(s_Devices)).IsFailure())
    {
        return 0;
    }

    RID_DEVICE_INFO rdi = {};

    for (int i = deviceCount - 1; 0 <= i; --i)
    {
        const RAWINPUTDEVICELIST& device = s_Devices[i];

        if (GetRawInputDeviceInformation(device.hDevice, &rdi).IsFailure() ||
            rdi.dwType != RIM_TYPEKEYBOARD)
        {
            --deviceCount;
        }
    }

    // 仮想キーボード分を減算
    return ::std::max(0, deviceCount - 1);
}

int GetRawInputPhysicalMouseCount() NN_NOEXCEPT
{
    static ::nn::os::SdkMutexType s_Mutex = NN_OS_SDK_MUTEX_INITIALIZER();

    ::std::lock_guard<decltype(s_Mutex)> locker(s_Mutex);

    static RAWINPUTDEVICELIST s_Devices[RawInputDeviceCountMax];

    auto deviceCount = int();

    if (GetRawInputDevices(
            &deviceCount, s_Devices, NN_ARRAY_SIZE(s_Devices)).IsFailure())
    {
        return 0;
    }

    RID_DEVICE_INFO rdi = {};

    for (int i = deviceCount - 1; 0 <= i; --i)
    {
        const RAWINPUTDEVICELIST& device = s_Devices[i];

        if (GetRawInputDeviceInformation(device.hDevice, &rdi).IsFailure() ||
            rdi.dwType != RIM_TYPEMOUSE)
        {
            --deviceCount;
        }
    }

    // 仮想マウス分を減算
    return ::std::max(0, deviceCount - 1);
}

int GetRawInputGamePadHandles(HANDLE outHandles[], int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outHandles);

    static ::nn::os::SdkMutexType s_Mutex = NN_OS_SDK_MUTEX_INITIALIZER();

    ::std::lock_guard<decltype(s_Mutex)> locker(s_Mutex);

    static RAWINPUTDEVICELIST s_Devices[RawInputDeviceCountMax];

    auto deviceCount = int();

    if (GetRawInputDevices(
            &deviceCount, s_Devices, NN_ARRAY_SIZE(s_Devices)).IsFailure())
    {
        return 0;
    }

    RID_DEVICE_INFO rdi = {};

    static char s_DeviceName[128];

    int handleCount = 0;

    for (int i = 0; i < deviceCount && handleCount < count; ++i)
    {
        const RAWINPUTDEVICELIST& device = s_Devices[i];

        if (device.dwType != RIM_TYPEHID)
        {
            continue;
        }

        if (GetRawInputDeviceInformation(device.hDevice, &rdi).IsFailure())
        {
            continue;
        }

        if (!IsHidSdiGamePad(rdi.hid.usUsagePage, rdi.hid.usUsage))
        {
            continue;
        }

        if (rdi.hid.dwVendorId == static_cast<DWORD>(ManufactureId::Nintendo))
        {
            continue;
        }

        if (GetRawInputDeviceName(
                device.hDevice, s_DeviceName, NN_ARRAY_SIZE(s_DeviceName)
                ).IsFailure())
        {
            continue;
        }

        if (IsXInput(s_DeviceName))
        {
            continue;
        }

        outHandles[handleCount] = device.hDevice;

        ++handleCount;
    }

    return handleCount;
}

::nn::Result GetRawInputGamePadDeviceName(
    HANDLE handle, char outName[], int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outName);

    NN_RESULT_DO(GetRawInputDeviceName(handle, outName, count));

    NN_RESULT_SUCCESS;
}

namespace {

bool IsXInput(char name[]) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(name);

    return ::std::strstr(name, "IG_") != nullptr;
}

::nn::Result GetRawInputDevices(
    int* pOutCount, RAWINPUTDEVICELIST outDevices[], int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(outDevices);
    NN_SDK_ASSERT_GREATER_EQUAL(count, 0);

    auto cbSize = static_cast<UINT>(count);

    cbSize = ::GetRawInputDeviceList(
        outDevices, &cbSize, sizeof(RAWINPUTDEVICELIST));

    if (cbSize == static_cast<UINT>(-1))
    {
        NN_RESULT_THROW(ResultWin32ApiFailedToGetRawInputDeviceList());
    }

    *pOutCount = static_cast<int>(cbSize);

    NN_RESULT_SUCCESS;
}

::nn::Result GetRawInputDeviceInformation(
    HANDLE handle, RID_DEVICE_INFO* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    RID_DEVICE_INFO& rdi = *pOutValue;
    rdi.cbSize = sizeof(RID_DEVICE_INFO);

    UINT cbSize = rdi.cbSize;

    if (static_cast<int>(::GetRawInputDeviceInfo(
            handle, RIDI_DEVICEINFO, &rdi, &cbSize)) < 0)
    {
        NN_RESULT_THROW(ResultWin32ApiFailedToGetRawInputDeviceInfo());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result GetRawInputDeviceName(
    HANDLE handle, char outName[], int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outName);
    NN_SDK_ASSERT_GREATER_EQUAL(count, 0);

    auto cbSize = static_cast<UINT>(count);

    if (static_cast<int>(::GetRawInputDeviceInfoA(
            handle, RIDI_DEVICENAME, outName, &cbSize)) < 0)
    {
        NN_RESULT_THROW(ResultWin32ApiFailedToGetRawInputDeviceInfo());
    }

    NN_RESULT_SUCCESS;
}

} // namespace

}}} // namespace nn::hid::detail
