﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Windows.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <hidsdi.h>

#include "hid_WindowsGenericPadAccessor-os.win.h"
#include "hid_WindowsHidSdi-os.win.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< Usage Page の定義です。
enum class UsagePage : USHORT
{
    GenericDesktopPage = 0x01,
    ButtonPage = 0x09,
};

//!< Generic Desktop Page の Usage Id の定義です。
enum class GenericDesktopPageUsageId : USHORT
{
    Joystick = 0x04,
    GamePad = 0x05,
    X = 0x30,
    Y = 0x31,
    Z = 0x32,
    RZ = 0x35,
    HatSwitch = 0x39,
};

//!< HID ゲームパッドの状態をパースします。
::nn::Result ParseHidSdiGamePadState(
    HidSdiHandle* pHandle, HidSdiGamePadState* pOutValue,
    HIDP_VALUE_CAPS valueCaps[], USHORT valueCapCount) NN_NOEXCEPT;

//!< HID ゲームパッドの能力を取得します。
void GetHidSdiGamePadCapability(
    WindowsGenericPadAbility* pOutValue,
    const char deviceName[],
    const HIDD_ATTRIBUTES& attributes,
    const HIDP_CAPS& caps,
    const HIDP_BUTTON_CAPS buttonCaps[],
    const HIDP_VALUE_CAPS valueCaps[]) NN_NOEXCEPT;

} // namespace

bool IsHidSdiGamePad(USHORT usagePage, USHORT usage) NN_NOEXCEPT
{
    if (usagePage != static_cast<USHORT>(UsagePage::GenericDesktopPage))
    {
        return false;
    }

    if (usage != static_cast<USHORT>(GenericDesktopPageUsageId::Joystick) &&
        usage != static_cast<USHORT>(GenericDesktopPageUsageId::GamePad))
    {
        return false;
    }

    return true;
}

HidSdiHandle CreateHidSdiHandle() NN_NOEXCEPT
{
    HidSdiHandle handle = { INVALID_HANDLE_VALUE, nullptr, FALSE, {}, 0, {} };

    return handle;
}

bool IsHidSdiHandleValid(const HidSdiHandle& handle) NN_NOEXCEPT
{
    return handle.handle != INVALID_HANDLE_VALUE;
}

::nn::Result OpenHidSdiDevice(
    HidSdiHandle* pOutHandle, const char deviceName[]) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutHandle);
    NN_SDK_REQUIRES_NOT_NULL(deviceName);

    HANDLE handle = ::CreateFileA(
        deviceName, GENERIC_READ | GENERIC_WRITE,
        FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_EXISTING,
        FILE_FLAG_OVERLAPPED, NULL);

    NN_RESULT_THROW_UNLESS(
        handle != INVALID_HANDLE_VALUE,
        ResultWin32ApiFailedToOpenHidDevice());

    PHIDP_PREPARSED_DATA pPreparsedData = nullptr;

    if (::HidD_GetPreparsedData(handle, &pPreparsedData) == FALSE)
    {
        ::CloseHandle(handle);

        NN_RESULT_THROW(ResultWin32ApiFailedToGetHidPreparsedData());
    }

    HIDP_CAPS caps = {};

    if (::HidP_GetCaps(pPreparsedData, &caps) != HIDP_STATUS_SUCCESS)
    {
        ::HidD_FreePreparsedData(pPreparsedData);

        ::CloseHandle(handle);

        NN_RESULT_THROW(ResultWin32ApiFailedToGetHidCapability());
    }

    pOutHandle->inputReportCount = caps.InputReportByteLength;

    pOutHandle->overlapped = OVERLAPPED();

    pOutHandle->overlapped.hEvent = ::CreateEvent(NULL, FALSE, FALSE, NULL);

    pOutHandle->isPending = FALSE;

    pOutHandle->pPreparsedData = pPreparsedData;

    pOutHandle->handle = handle;

    NN_RESULT_SUCCESS;
}

void CloseHidSdiDevice(HidSdiHandle* pHandle) NN_NOEXCEPT
{
    ::CloseHandle(pHandle->overlapped.hEvent);

    ::HidD_FreePreparsedData(pHandle->pPreparsedData);

    ::CloseHandle(pHandle->handle);

    pHandle->overlapped.hEvent = INVALID_HANDLE_VALUE;

    pHandle->handle = INVALID_HANDLE_VALUE;
}

::nn::Result GetHidSdiGamePadState(
    HidSdiHandle* pHandle, HidSdiGamePadState* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHandle);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    static ::nn::os::SdkMutexType s_Mutex = NN_OS_SDK_MUTEX_INITIALIZER();

    static HIDP_VALUE_CAPS s_ValueCaps[16];

    if (pHandle->isPending == FALSE)
    {
        ::ResetEvent(pHandle->overlapped.hEvent);

        auto count = DWORD();

        if (::ReadFile(
                pHandle->handle,
                pHandle->inputReport,
                ::std::min<ULONG>(
                    pHandle->inputReportCount,
                    NN_ARRAY_SIZE(pHandle->inputReport)),
                &count, &pHandle->overlapped) != 0)
        {
            NN_RESULT_THROW_UNLESS(
                pHandle->inputReportCount == count,
                ResultWin32ApiFailedToReadHidInputReport());

            ::std::lock_guard<decltype(s_Mutex)> locker(s_Mutex);

            NN_RESULT_DO(
                ParseHidSdiGamePadState(
                    pHandle, pOutValue,
                    s_ValueCaps, NN_ARRAY_SIZE(s_ValueCaps)));

            NN_RESULT_SUCCESS;
        }
        else if (::GetLastError() == ERROR_IO_PENDING)
        {
            pHandle->isPending = TRUE;

            NN_RESULT_THROW(ResultWin32ApiIoPending());
        }
        else
        {
            ::CancelIo(pHandle->handle);

            NN_RESULT_THROW(ResultWin32ApiFailedToReadHidInputReport());
        }
    }
    else
    {
        if (::WaitForSingleObject(
                pHandle->overlapped.hEvent, 0) != WAIT_OBJECT_0)
        {
            NN_RESULT_THROW(ResultWin32ApiIoPending());
        }
        else
        {
            pHandle->isPending = FALSE;

            auto count = DWORD();

            if (::GetOverlappedResult(
                    pHandle->handle, &pHandle->overlapped, &count, TRUE) == 0)
            {
                NN_RESULT_THROW(ResultWin32ApiFailedToReadHidInputReport());
            }
            else
            {
                NN_RESULT_THROW_UNLESS(
                    pHandle->inputReportCount == count,
                    ResultWin32ApiFailedToReadHidInputReport());

                ::std::lock_guard<decltype(s_Mutex)> locker(s_Mutex);

                NN_RESULT_DO(
                    ParseHidSdiGamePadState(
                        pHandle, pOutValue,
                        s_ValueCaps, NN_ARRAY_SIZE(s_ValueCaps)));

                NN_RESULT_SUCCESS;
            }
        }
    }
}

::nn::Result GetHidSdiGamePadCapability(
    WindowsGenericPadAbility* pOutValue, const char deviceName[]) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(deviceName);

    static ::nn::os::SdkMutexType s_Mutex = NN_OS_SDK_MUTEX_INITIALIZER();

    ::std::lock_guard<decltype(s_Mutex)> locker(s_Mutex);

    HidSdiHandle handle = {};

    NN_RESULT_DO(OpenHidSdiDevice(&handle, deviceName));

    NN_UTIL_SCOPE_EXIT
    {
        CloseHidSdiDevice(&handle);
    };

    HIDD_ATTRIBUTES attributes = {};
    attributes.Size = sizeof(attributes);

    NN_RESULT_THROW_UNLESS(
        ::HidD_GetAttributes(handle.handle, &attributes) == TRUE,
        ResultWin32ApiFailedToGetHidAttributes());

    HIDP_CAPS caps = {};

    NN_RESULT_THROW_UNLESS(
        ::HidP_GetCaps(handle.pPreparsedData, &caps) == HIDP_STATUS_SUCCESS,
        ResultWin32ApiFailedToGetHidCapability());

    static HIDP_BUTTON_CAPS s_ButtonCaps[16];

    caps.NumberInputButtonCaps = ::std::min<USHORT>(
        caps.NumberInputButtonCaps, NN_ARRAY_SIZE(s_ButtonCaps));

    NN_RESULT_THROW_UNLESS(
        ::HidP_GetButtonCaps(
            HidP_Input, s_ButtonCaps, &caps.NumberInputButtonCaps,
            handle.pPreparsedData) == HIDP_STATUS_SUCCESS,
        ResultWin32ApiFailedToGetHidButtonCapability());

    static HIDP_VALUE_CAPS s_ValueCaps[16];

    caps.NumberInputValueCaps = ::std::min<USHORT>(
        caps.NumberInputValueCaps, NN_ARRAY_SIZE(s_ValueCaps));

    NN_RESULT_THROW_UNLESS(
        ::HidP_GetValueCaps(
            HidP_Input, s_ValueCaps, &caps.NumberInputValueCaps,
            handle.pPreparsedData) == HIDP_STATUS_SUCCESS,
        ResultWin32ApiFailedToGetHidValueCapability());

    GetHidSdiGamePadCapability(
        pOutValue, deviceName, attributes, caps, s_ButtonCaps, s_ValueCaps);

    NN_RESULT_SUCCESS;
}

namespace {

::nn::Result ParseHidSdiGamePadButtons(
    HidSdiHandle* pHandle, uint64_t* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHandle);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    USAGE usages[32] = {};
    ULONG usageCount = NN_ARRAY_SIZE(usages);

    NN_RESULT_THROW_UNLESS(
        ::HidP_GetUsages(
            HidP_Input, static_cast<USAGE>(UsagePage::ButtonPage), 0,
            usages, &usageCount, pHandle->pPreparsedData,
            pHandle->inputReport, pHandle->inputReportCount
            ) == HIDP_STATUS_SUCCESS,
        ResultWin32ApiFailedToGetHidUsage());

    *pOutValue = uint64_t();

    for (ULONG i = 0; i < usageCount; ++i)
    {
        *pOutValue |= 1ull << (usages[i] - 1);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result ParseHidSdiGamePadHatSwitch(
    HidSdiHandle* pHandle, int* pOutValue,
    const HIDP_VALUE_CAPS& valueCaps) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHandle);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    auto value = ULONG();

    NN_RESULT_THROW_UNLESS(
        ::HidP_GetUsageValue(
            HidP_Input, valueCaps.UsagePage, 0, valueCaps.Range.UsageMin,
            &value,
            pHandle->pPreparsedData,
            pHandle->inputReport, pHandle->inputReportCount
            ) == HIDP_STATUS_SUCCESS,
        ResultWin32ApiFailedToGetHidUsageValue());

    auto pov = static_cast<int>(value);

    if (pov < valueCaps.LogicalMin || valueCaps.LogicalMax < pov)
    {
        *pOutValue = -1;
    }
    else
    {
        *pOutValue = pov - valueCaps.LogicalMin;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result ParseHidSdiGamePadPosition(
    HidSdiHandle* pHandle, int* pOutValue,
    const HIDP_VALUE_CAPS& valueCaps) NN_NOEXCEPT
{
    if (valueCaps.LogicalMin < 0)
    {
        auto value = LONG();

        NN_RESULT_THROW_UNLESS(
            ::HidP_GetScaledUsageValue(
                HidP_Input, valueCaps.UsagePage, 0, valueCaps.Range.UsageMin,
                &value,
                pHandle->pPreparsedData,
                pHandle->inputReport, pHandle->inputReportCount
                ) == HIDP_STATUS_SUCCESS,
            ResultWin32ApiFailedToGetHidUsageValue());

        *pOutValue = static_cast<int>(value);
    }
    else
    {
        auto value = ULONG();

        NN_RESULT_THROW_UNLESS(
            ::HidP_GetUsageValue(
                HidP_Input, valueCaps.UsagePage, 0, valueCaps.Range.UsageMin,
                &value,
                pHandle->pPreparsedData,
                pHandle->inputReport, pHandle->inputReportCount
                ) == HIDP_STATUS_SUCCESS,
            ResultWin32ApiFailedToGetHidUsageValue());

        *pOutValue = static_cast<int>(value);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result ParseHidSdiGamePadState(
    HidSdiHandle* pHandle, HidSdiGamePadState* pOutValue,
    HIDP_VALUE_CAPS valueCaps[], USHORT valueCapCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHandle);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(valueCaps);

    NN_RESULT_THROW_UNLESS(
        ::HidP_GetValueCaps(
            HidP_Input, valueCaps, &valueCapCount,
            pHandle->pPreparsedData) == HIDP_STATUS_SUCCESS,
        ResultWin32ApiFailedToGetHidValueCapability());

    auto buttons = uint64_t();

    NN_RESULT_DO(ParseHidSdiGamePadButtons(pHandle, &buttons));

    int pov = -1;

    int positions[6] = {};

    for (size_t i = 0; i < valueCapCount; ++i)
    {
        if (valueCaps[i].UsagePage !=
            static_cast<USHORT>(UsagePage::GenericDesktopPage))
        {
            continue;
        }

        switch (valueCaps[i].Range.UsageMin)
        {
        case static_cast<USHORT>(GenericDesktopPageUsageId::X):
            NN_RESULT_DO(
                ParseHidSdiGamePadPosition(
                    pHandle, &positions[0], valueCaps[i]));
            break;

        case static_cast<USHORT>(GenericDesktopPageUsageId::Y):
            NN_RESULT_DO(
                ParseHidSdiGamePadPosition(
                    pHandle, &positions[1], valueCaps[i]));
            break;

        case static_cast<USHORT>(GenericDesktopPageUsageId::Z):
            NN_RESULT_DO(
                ParseHidSdiGamePadPosition(
                    pHandle, &positions[2], valueCaps[i]));
            break;

        case static_cast<USHORT>(GenericDesktopPageUsageId::RZ):
            NN_RESULT_DO(
                ParseHidSdiGamePadPosition(
                    pHandle, &positions[3], valueCaps[i]));
            break;

        case static_cast<USHORT>(GenericDesktopPageUsageId::HatSwitch):
            NN_RESULT_DO(
                ParseHidSdiGamePadHatSwitch(pHandle, &pov, valueCaps[i]));
            break;

        default:
            break;
        }
    }

    pOutValue->buttons = buttons;

    pOutValue->pov = pov;

    ::std::copy(
        ::std::begin(positions), ::std::end(positions), pOutValue->positions);

    NN_RESULT_SUCCESS;
}

void GetHidSdiGamePadCapability(
    WindowsGenericPadAbility* pOutValue,
    const char deviceName[],
    const HIDD_ATTRIBUTES& attributes,
    const HIDP_CAPS& caps,
    const HIDP_BUTTON_CAPS buttonCaps[],
    const HIDP_VALUE_CAPS valueCaps[]) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(buttonCaps);
    NN_SDK_REQUIRES_NOT_NULL(valueCaps);

    WindowsGenericPadAbility& outValue = *pOutValue;

    outValue = WindowsGenericPadAbility();

    ::nn::util::Strlcpy(
        outValue.deviceName, deviceName, NN_ARRAY_SIZE(outValue.deviceName));

    outValue.vid = attributes.VendorID;

    outValue.pid = attributes.ProductID;

    for (size_t i = 0; i < caps.NumberInputButtonCaps; ++i)
    {
        outValue.buttonCount +=
            buttonCaps[i].Range.UsageMax - buttonCaps[i].Range.UsageMin + 1;
    }

    for (size_t i = 0; i < caps.NumberInputValueCaps; ++i)
    {
        if (valueCaps[i].UsagePage !=
            static_cast<USHORT>(UsagePage::GenericDesktopPage))
        {
            continue;
        }

        switch (valueCaps[i].Range.UsageMin)
        {
        case static_cast<USHORT>(GenericDesktopPageUsageId::X):
            outValue.xMax = valueCaps[i].LogicalMax;
            outValue.xMin = valueCaps[i].LogicalMin;
            break;

        case static_cast<USHORT>(GenericDesktopPageUsageId::Y):
            outValue.yMax = valueCaps[i].LogicalMax;
            outValue.yMin = valueCaps[i].LogicalMin;
            break;

        case static_cast<USHORT>(GenericDesktopPageUsageId::Z):
            outValue.flags.Set<WindowsGenericPadAbilityFlag::HasZ>();
            outValue.zMax = valueCaps[i].LogicalMax;
            outValue.zMin = valueCaps[i].LogicalMin;
            break;

        case static_cast<USHORT>(GenericDesktopPageUsageId::RZ):
            outValue.flags.Set<WindowsGenericPadAbilityFlag::HasR>();
            outValue.rMax = valueCaps[i].LogicalMax;
            outValue.rMin = valueCaps[i].LogicalMin;
            break;

        case static_cast<USHORT>(GenericDesktopPageUsageId::HatSwitch):
            if (valueCaps[i].LogicalMax - valueCaps[i].LogicalMin == 7)
            {
                outValue.flags.Set<WindowsGenericPadAbilityFlag::HasPov>();
                outValue.buttonCount += 4;
            }

            break;

        default:
            break;
        }
    }

    outValue.flags.Set<WindowsGenericPadAbilityFlag::IsJoystick>();
}

} // namespace

}}} // namespace nn::hid::detail
