﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_Windows.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <XInput.h>

#include "hid_WindowsGenericPadAccessor-os.win.h"
#include "hid_WindowsGenericPadBroker-os.win.h"
#include "hid_WindowsHidSdi-os.win.h"
#include "hid_WindowsRawInput-os.win.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< 汎用コントローラの識別子を表す列挙体です。
enum WindowsGenericPadId : uint32_t
{
    WindowsGenericPadId_XInput1,
    WindowsGenericPadId_XInput2,
    WindowsGenericPadId_XInput3,
    WindowsGenericPadId_XInput4,
    WindowsGenericPadId_Joystick1,
    WindowsGenericPadId_Joystick2,
    WindowsGenericPadId_Joystick3,
    WindowsGenericPadId_Joystick4,
    WindowsGenericPadId_Joystick5,
    WindowsGenericPadId_Joystick6,
    WindowsGenericPadId_Joystick7,
    WindowsGenericPadId_Joystick8,
    WindowsGenericPadId_Joystick9,
    WindowsGenericPadId_Joystick10,
    WindowsGenericPadId_Joystick11,
    WindowsGenericPadId_Joystick12,
    WindowsGenericPadId_Joystick13,
    WindowsGenericPadId_Joystick14,
    WindowsGenericPadId_Joystick15,
    WindowsGenericPadId_Joystick16,
};

//!< XInput から汎用コントローラの能力を取得します。
bool GetWindowsGenericPadAbilityFromXInput(
    WindowsGenericPadAbility* pOutValue,
    uint32_t index) NN_NOEXCEPT;

//!< Joystick から汎用コントローラの能力を取得します。
bool GetWindowsGenericPadAbilityFromJoystick(
    WindowsGenericPadAbility* pOutValue,
    uint32_t index, HANDLE handle, char name[], int nameCount) NN_NOEXCEPT;

//!< Joystick のハンドルを更新します。
template<int N>
void UpdateJoystickHandles(HANDLE (&outHandles)[N]) NN_NOEXCEPT;

} // namespace

const ::nn::TimeSpan WindowsGenericPadBroker::SamplingInterval =
    ::nn::TimeSpan::FromMilliSeconds(25);

WindowsGenericPadBroker::WindowsGenericPadBroker() NN_NOEXCEPT
    : m_ActivationCount()
    , m_pTimerEvent()
    , m_FocusedGenericPadId()
    , m_GenericPadCount()
{
    for (WindowsGenericPadAbility& ability : m_GenericPadAbilities)
    {
        ability = WindowsGenericPadAbility();
    }

    for (HANDLE& handle : m_JoystickHandles)
    {
        handle = INVALID_HANDLE_VALUE;
    }
}

void WindowsGenericPadBroker::SetTimerEvent(
    ::nn::os::TimerEventType* pTimerEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pTimerEvent);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pTimerEvent = pTimerEvent;
}

::nn::Result WindowsGenericPadBroker::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTimerEvent);

    NN_RESULT_THROW_UNLESS(
        !m_ActivationCount.IsMax(),
        ResultWindowsGenericPadBrokerActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        // タイマーイベントをアクティブ化
        ::nn::os::StartPeriodicTimerEvent(
            m_pTimerEvent, SamplingInterval, SamplingInterval);
    }

    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result WindowsGenericPadBroker::Deactivate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTimerEvent);

    NN_RESULT_THROW_UNLESS(
        !m_ActivationCount.IsZero(),
        ResultWindowsGenericPadBrokerDeactivationLowerLimitOver());

    --m_ActivationCount;

    if (m_ActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施

        // タイマーイベントを非アクティブ化
        ::nn::os::StopTimerEvent(m_pTimerEvent);
    }

    NN_RESULT_SUCCESS;
}

void WindowsGenericPadBroker::Sample() NN_NOEXCEPT
{
    if (!m_ActivationCount.IsZero())
    {
        this->Update();
    }
}

void WindowsGenericPadBroker::Update() NN_NOEXCEPT
{
    if (m_FocusedGenericPadId == 0)
    {
        m_GenericPadCount = 0;
    }

    switch (m_FocusedGenericPadId)
    {
    case WindowsGenericPadId_XInput1:
    case WindowsGenericPadId_XInput2:
    case WindowsGenericPadId_XInput3:
    case WindowsGenericPadId_XInput4:
        {
            const uint32_t index =
                m_FocusedGenericPadId -
                static_cast<uint32_t>(WindowsGenericPadId_XInput1);

            if (m_GenericPadCount < NN_ARRAY_SIZE(m_GenericPadAbilities) &&
                GetWindowsGenericPadAbilityFromXInput(
                    &m_GenericPadAbilities[m_GenericPadCount], index))
            {
                m_GenericPadCount += 1;
            }

            m_FocusedGenericPadId += 1;

            break;
        }

    case WindowsGenericPadId_Joystick1:
    case WindowsGenericPadId_Joystick2:
    case WindowsGenericPadId_Joystick3:
    case WindowsGenericPadId_Joystick4:
    case WindowsGenericPadId_Joystick5:
    case WindowsGenericPadId_Joystick6:
    case WindowsGenericPadId_Joystick7:
    case WindowsGenericPadId_Joystick8:
    case WindowsGenericPadId_Joystick9:
    case WindowsGenericPadId_Joystick10:
    case WindowsGenericPadId_Joystick11:
    case WindowsGenericPadId_Joystick12:
    case WindowsGenericPadId_Joystick13:
    case WindowsGenericPadId_Joystick14:
    case WindowsGenericPadId_Joystick15:
    case WindowsGenericPadId_Joystick16:
        {
            const uint32_t index =
                m_FocusedGenericPadId -
                static_cast<uint32_t>(WindowsGenericPadId_Joystick1);

            if (m_GenericPadCount < NN_ARRAY_SIZE(m_GenericPadAbilities) &&
                GetWindowsGenericPadAbilityFromJoystick(
                    &m_GenericPadAbilities[m_GenericPadCount], index,
                    m_JoystickHandles[index],
                    m_DeviceName, NN_ARRAY_SIZE(m_DeviceName)))
            {
                m_GenericPadCount += 1;
            }

            m_FocusedGenericPadId += 1;

            break;
        }

    default:
        {
            UpdateJoystickHandles(m_JoystickHandles);

            NN_ABORT_UNLESS_RESULT_SUCCESS(
                UpdateWindowsGenericPadAbilities(
                    m_GenericPadAbilities, m_GenericPadCount));

            m_FocusedGenericPadId = 0;

            break;
        }
    }
}

namespace {

bool GetWindowsGenericPadAbilityFromXInput(
    WindowsGenericPadAbility* pOutValue,
    uint32_t index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    WindowsGenericPadAbility& outValue = *pOutValue;

    XINPUT_CAPABILITIES capabilities = {};

    if (::XInputGetCapabilities(
            index, XINPUT_FLAG_GAMEPAD, &capabilities) == ERROR_SUCCESS)
    {
        outValue = WindowsGenericPadAbility();
        outValue.flags.Set<WindowsGenericPadAbilityFlag::IsXInput>();
        outValue.index = index;

        return true;
    }

    return false;
}

bool GetWindowsGenericPadAbilityFromJoystick(
    WindowsGenericPadAbility* pOutValue,
    uint32_t index, HANDLE handle, char name[], int nameCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(name);
    NN_UNUSED(index);

    if (handle != INVALID_HANDLE_VALUE &&
        GetRawInputGamePadDeviceName(handle, name, nameCount).IsSuccess())
    {
        return GetHidSdiGamePadCapability(pOutValue, name).IsSuccess();
    }

    return false;
}

template<int N>
void UpdateJoystickHandles(HANDLE (&outHandles)[N]) NN_NOEXCEPT
{
    HANDLE handles[N] = {};

    const int count = GetRawInputGamePadHandles(handles, N);

    for (int i = 0; i < count; ++i)
    {
        outHandles[i] = handles[i];
    }

    for (int i = count; i < N; ++i)
    {
        outHandles[i] = INVALID_HANDLE_VALUE;
    }
}

} // namespace

}}} // namespace nn::hid::detail
