﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadGc.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/hid_VibrationDeviceApi.h>
#include <nn/hid/hid_VibrationGcErm.h>
#include <nn/hid/hid_VibrationValueTypes.h>

#include "hid_NpadId.h"

namespace nn { namespace hid { namespace detail {

//!< Npad のスタイル、プレイヤー番号、振動子番号から VibrationDeviceHandle を作成します。
VibrationDeviceHandle MakeVibrationDeviceHandle(uint8_t npadStyleId, int playerNumber, int vibrationDeviceIdx) NN_NOEXCEPT;

//!< VibrationDeviceHandle から Npad のスタイル (に対応する整数値) を取得します。
uint8_t GetVibrationDeviceHandleXpadTypeValue(const VibrationDeviceHandle& handle) NN_NOEXCEPT;

//!< VibrationDeviceHandle からプレイヤー番号を取得します。
int GetVibrationDeviceHandlePlayerNumber(const VibrationDeviceHandle& handle) NN_NOEXCEPT;

//!< VibrationDeviceHandle から振動子番号を取得します。
int GetVibrationDeviceHandleDeviceIdx(const VibrationDeviceHandle& handle) NN_NOEXCEPT;

//!< Npad の振動子番号
enum NpadVibrationDeviceIdx
{
    NpadVibrationDeviceIdx_Left = 0,
    NpadVibrationDeviceIdx_Right = 1,
    NpadVibrationDeviceIdx_None = 0
};

//!< VibrationDevicePosition から NpadVibrationDeviceIdx を取得します。
NpadVibrationDeviceIdx GetVibrationDeviceIdxFromPosition(VibrationDevicePosition position) NN_NOEXCEPT;

//!< NpadVibrationDeviceIdx から VibrationDevicePosition を取得します。
VibrationDevicePosition GetVibrationDevicePositionFromIdx(VibrationDeviceType type, NpadVibrationDeviceIdx vibrationDeviceIdx) NN_NOEXCEPT;

//!< Npad 用の TypeValue 生成オフセット
const int NpadVibrationTypeValueOffset = 3;

inline VibrationDeviceType GetVibrationDeviceHandleDeviceType(const VibrationDeviceHandle& handle) NN_NOEXCEPT
{
    // NPADかどうかを判定するだけの暫定対応
    int typeValue = GetVibrationDeviceHandleXpadTypeValue(handle) - NpadVibrationTypeValueOffset;
    switch (typeValue)
    {
    case NpadStyleFullKey::Index:
    case NpadStyleHandheld::Index:
    case NpadStyleJoyDual::Index:
    case NpadStyleJoyLeft::Index:
    case NpadStyleJoyRight::Index:
    case system::NpadStyleSystem::Index:
    case system::NpadStyleSystemExt::Index:
        return VibrationDeviceType_LinearResonantActuator;
    case NpadStyleGc::Index:
        return VibrationDeviceType_GcErm;
    default:
        return VibrationDeviceType_Unknown;
    }

}

//!< VibrationDeviceHandle から VibrationDevicePosition を取得します。
inline VibrationDevicePosition GetVibrationDeviceHandleDevicePosition(const VibrationDeviceHandle& handle) NN_NOEXCEPT
{
    int vibrationDeviceIdx = GetVibrationDeviceHandleDeviceIdx(handle);
    auto vibrationDeviceType = GetVibrationDeviceHandleDeviceType(handle);
    return GetVibrationDevicePositionFromIdx(vibrationDeviceType, static_cast<NpadVibrationDeviceIdx>(vibrationDeviceIdx));
}

//!< NpadIdType と NpadStyleSet から ハンドルを取得します。
::nn::Result GetVibrationDeviceHandles(VibrationDeviceHandle* pOutHandle,
                                      int* pOutCount,
                                      int size,
                                      NpadIdType npadId,
                                      NpadStyleSet style) NN_NOEXCEPT;

}}} // namespace nn::hid::detail
