﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <mutex>
#include <new>
#include <nn/nn_Macro.h>
#include <nn/os/os_Mutex.h>
#include <nn/util/util_TypedStorage.h>

#include "hid_LockableMutexType.h"
#include "hid_UniquePadId.h"

namespace nn { namespace hid { namespace detail {

//!< UniquePad のセッションを管理するためのクラスです。
template<typename T>
class UniquePadSessionManager final
{
    NN_DISALLOW_COPY(UniquePadSessionManager);
    NN_DISALLOW_MOVE(UniquePadSessionManager);

private:
    //!< UniquePad のセッションのエントリを表す構造体です。
    struct Entry final
    {
        // セッションが初期化済みか否かを表す値
        ::std::atomic_bool isInitialized;

        // セッションの初期化排他用ミューテックス
        LockableMutexType mutex;

        // セッションの格納先となるメモリ領域
        ::nn::util::TypedStorage<T, sizeof(T), NN_ALIGNOF(T)> storage;
    };

    //!< UniquePad のセッションのエントリ
    Entry m_Entries[system::UniquePadIdCountMax];

public:
    UniquePadSessionManager() NN_NOEXCEPT
    {
        for (Entry& entry : m_Entries)
        {
            entry.isInitialized = false;
            ::nn::os::InitializeMutex(&entry.mutex._mutex, false, 0);
        }
    }

    ~UniquePadSessionManager() NN_NOEXCEPT { /* 何もしない */ }

    //!< UniquePad のセッションを返します。
    T* GetSession(const system::UniquePadId& id) NN_NOEXCEPT
    {
        Entry& entry = m_Entries[GetIndexFromUniquePadId(id)];

        T* pSession = &::nn::util::Get(entry.storage);

        if (!entry.isInitialized)
        {
            ::std::lock_guard<decltype(entry.mutex)> locker(entry.mutex);

            if (!entry.isInitialized)
            {
                new(pSession) T(id);

                entry.isInitialized = true;
            }
        }

        return pSession;
    }
};

}}} // namespace nn::hid::detail
