﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/hid/system/hid_Result.h>

#include "hid_UniquePadResourceManager.h"

namespace nn { namespace hid { namespace detail {

UniquePadResourceManager::UniquePadResourceManager() NN_NOEXCEPT
    : m_UniquePadMutex(false)
{
    // 何もしない
}

UniquePadResourceManager::~UniquePadResourceManager() NN_NOEXCEPT
{
    // 何もしない
}

void UniquePadResourceManager::Initialize(HandheldManager* pHandheldManager,
                                          InterruptSceneNotifier* pInterruptSceneNotifier) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHandheldManager);
    NN_SDK_REQUIRES_NOT_NULL(pInterruptSceneNotifier);

    for (int i = 0; i < NN_ARRAY_SIZE(m_UniquePadManagers); ++i)
    {
        m_UniquePadManagers[i].SetHandheldManager(pHandheldManager);
        m_UniquePadManagers[i].Initialize();
        m_XcdFirmwareUpdater.SetUniquePadManagers(i, &m_UniquePadManagers[i]);
    }

    for (int64_t i = 0; i < system::UniquePadIdCountMax; ++i)
    {
        m_AssignmentManager.SetUniquePadManagers(
            m_UniquePadManagers, NN_ARRAY_SIZE(m_UniquePadManagers));
    }

    m_XcdFirmwareUpdater.SetInterruptSceneNotifier(pInterruptSceneNotifier);
}

void UniquePadResourceManager::SetSystemEvents(nn::os::SystemEventType* pBluetoothFirmwareUpdateEvent,
                                               nn::os::SystemEventType* pMcuFirmwareUpdateEvent,
                                               nn::os::SystemEventType* pSampleUpdateEvent,
                                               nn::os::TimerEventType* pResetTimeoutEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBluetoothFirmwareUpdateEvent);
    NN_SDK_REQUIRES_NOT_NULL(pMcuFirmwareUpdateEvent);
    NN_SDK_REQUIRES_NOT_NULL(pSampleUpdateEvent);
    NN_SDK_REQUIRES_NOT_NULL(pResetTimeoutEvent);

    m_XcdFirmwareUpdater.SetBluetoothUpdateEvent(pBluetoothFirmwareUpdateEvent);
    m_XcdFirmwareUpdater.SetMcuUpdateEvent(pMcuFirmwareUpdateEvent);
    m_XcdFirmwareUpdater.SetSampleUpdateEvent(pSampleUpdateEvent);
    m_XcdFirmwareUpdater.SetResetTimeoutEvent(pResetTimeoutEvent);
}

void UniquePadResourceManager::AddIAbstractedPad(IAbstractedPad* pPads) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPads);
    m_AssignmentManager.AddIAbstractedPad(pPads);
}

::nn::Result UniquePadResourceManager::Activate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultGamePadDriverActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        auto needsRollback = true;

        ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);

        NN_RESULT_DO(m_AssignmentManager.Activate());
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_AssignmentManager.Deactivate();
            }
        };

        needsRollback = false;
    }

    // アクティブ化した回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result UniquePadResourceManager::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultGamePadDriverDeactivationLowerLimitOver());

    // アクティブ化した回数をデクリメント
    --m_ActivationCount;

    if (m_ActivationCount.IsZero())
    {
        NN_RESULT_DO(this->m_AssignmentManager.Deactivate());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result UniquePadResourceManager::AcquireUniquePadConnectionEventHandle(::nn::os::NativeHandle* pOutHandle) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    NN_RESULT_DO(
        m_AssignmentManager.AcquireUniquePadConnectionEventHandle(pOutHandle));
    NN_RESULT_SUCCESS;
}

void UniquePadResourceManager::NotifyAppletResourceUserId(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    m_XcdFirmwareUpdater.NotifyApplicationResourceUserId(aruid);
}

void UniquePadResourceManager::HandlePeriodicalEvent() NN_NOEXCEPT
{
    if (m_ActivationCount.IsZero())
    {
        // 初期化前は何もせずに返る
        return;
    }

    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    m_AssignmentManager.UpdateDeviceMap();
    for (int i = 0; i < NN_ARRAY_SIZE(m_UniquePadManagers); ++i)
    {
        m_UniquePadManagers[i].Sample();
    }
}

void UniquePadResourceManager::HandleDeviceUpdate() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    m_XcdFirmwareUpdater.DeviceUpdated();
}

void UniquePadResourceManager::HandleBluetoothFirmwareUpdate() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    m_XcdFirmwareUpdater.BluetoothFirmwareUpdateCompleted();
}

void UniquePadResourceManager::HandleMcuFirmwareUpdate() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);

    m_XcdFirmwareUpdater.McuFirmwareUpdateCompleted();
}

void UniquePadResourceManager::HandleNewSample() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    m_XcdFirmwareUpdater.SampleRecieved();
}

void UniquePadResourceManager::HandleResetTimedOut() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    m_XcdFirmwareUpdater.ResetTimedOut();
}

void UniquePadResourceManager::HandleConnectionTriggerTimeout() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);

    m_XcdFirmwareUpdater.ConnectionTriggerTimedOut();
}

::nn::Result UniquePadResourceManager::GetIds(int* pOutCount,
                                              system::UniquePadId* outUniquePadIds,
                                              int count) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    NN_RESULT_DO(
        m_AssignmentManager.GetUniquePadIds(pOutCount, outUniquePadIds, count));
    NN_RESULT_SUCCESS;
}

bool UniquePadResourceManager::IsConnected(system::UniquePadId& id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    auto pManager = this->GetUniquePadManager(id);
    if (pManager != nullptr)
    {
        return pManager->IsConnected();
    }
    return false;
}

::nn::Result UniquePadResourceManager::GetControllerNumber(int* pOutValue, system::UniquePadId& id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    auto pManager = this->GetUniquePadManager(id);
    NN_RESULT_THROW_UNLESS(pManager != nullptr, system::ResultUniquePadDisconnected());
    NN_RESULT_THROW_UNLESS(pManager->IsConnected(), system::ResultUniquePadDisconnected());

    *pOutValue = pManager->GetControllerNumber();
    NN_RESULT_SUCCESS;
}

::nn::Result UniquePadResourceManager::GetType(system::UniquePadType* pOutValue, system::UniquePadId& id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    auto pManager = this->GetUniquePadManager(id);
    if (pManager != nullptr && pManager->IsConnected())
    {
        *pOutValue = pManager->GetType();
    }
    else
    {
        *pOutValue = system::UniquePadType_Unknown;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result UniquePadResourceManager::GetInterface(system::UniquePadInterface* pOutValue, system::UniquePadId& id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    auto pManager = this->GetUniquePadManager(id);
    NN_RESULT_THROW_UNLESS(pManager != nullptr, system::ResultUniquePadDisconnected());
    NN_RESULT_THROW_UNLESS(pManager->IsConnected(), system::ResultUniquePadDisconnected());

    NN_RESULT_DO(pManager->GetInterface(pOutValue));
    NN_RESULT_SUCCESS;
}

::nn::Result UniquePadResourceManager::GetSerialNumber(system::UniquePadSerialNumber* pOutValue, system::UniquePadId& id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    auto pManager = this->GetUniquePadManager(id);
    NN_RESULT_THROW_UNLESS(pManager != nullptr, system::ResultUniquePadDisconnected());
    NN_RESULT_THROW_UNLESS(pManager->IsConnected(), system::ResultUniquePadDisconnected());

    NN_RESULT_DO(pManager->GetSerialNumber(pOutValue));
    NN_RESULT_SUCCESS;
}

Result UniquePadResourceManager::GetBluetoothAddress(::nn::bluetooth::Address* pOutAddress,
                                                system::UniquePadId& id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    auto pManager = this->GetUniquePadManager(id);
    NN_RESULT_THROW_UNLESS(pManager != nullptr, system::ResultUniquePadDisconnected());
    NN_RESULT_THROW_UNLESS(pManager->IsConnected(), system::ResultUniquePadDisconnected());
    NN_RESULT_DO(
        pManager->GetBluetoothAddress(pOutAddress));
    NN_RESULT_SUCCESS;
}

Result UniquePadResourceManager::DisconnectUniquePad(system::UniquePadId& id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    auto pManager = this->GetUniquePadManager(id);
    NN_RESULT_THROW_UNLESS(pManager != nullptr, system::ResultUniquePadDisconnected());
    NN_RESULT_DO(
        pManager->Disconnect());
    NN_RESULT_SUCCESS;
}

Result UniquePadResourceManager::ActivateFirmwareUpdate() NN_NOEXCEPT
{
    auto needsRollback = true;

    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    NN_RESULT_DO(m_XcdFirmwareUpdater.ActivateFirmwareUpdate());
    needsRollback = false;

    NN_RESULT_SUCCESS;
}

Result UniquePadResourceManager::DeactivateFirmwareUpdate() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    NN_RESULT_DO(m_XcdFirmwareUpdater.DeactivateFirmwareUpdate());
    NN_RESULT_SUCCESS;
}

Result UniquePadResourceManager::StartFirmwareUpdate(
    system::FirmwareUpdateDeviceHandle* pOutDeviceHandle,
    system::UniquePadId id) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutDeviceHandle);

    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    NN_RESULT_THROW(m_XcdFirmwareUpdater.StartFirmwareUpdateForSystem(
        pOutDeviceHandle,
        id));
}

Result UniquePadResourceManager::StartFirmwareUpdateForDebug(
    system::UniquePadId id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    NN_RESULT_THROW(m_XcdFirmwareUpdater.StartFirmwareUpdateForDebug(id));
}

Result UniquePadResourceManager::StartFirmwareUpdateForRevert(
    system::UniquePadId id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    NN_RESULT_THROW(m_XcdFirmwareUpdater.StartFirmwareUpdateForRevert(id));
}

Result UniquePadResourceManager::StartFirmwareUpdateByTransferMemory(
    system::FirmwareUpdateDeviceHandle* pOutDeviceHandle,
    system::UniquePadId id,
    XcdFirmwareUpdater::FirmwareUpdateTarget target,
    const XcdFirmwareImageInfo& imageInfo) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    NN_RESULT_THROW(m_XcdFirmwareUpdater.StartFirmwareUpdateByTransferMemory(pOutDeviceHandle, id, target, imageInfo));
}

Result UniquePadResourceManager::GetFirmwareVersion(
    system::FirmwareVersion* pOutValue,
    system::UniquePadId id) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    NN_RESULT_THROW(m_XcdFirmwareUpdater.GetFirmwareVersion(pOutValue, id));
}

Result UniquePadResourceManager::GetDestinationFirmwareVersion(
    system::FirmwareVersion *pOutValue,
    system::UniquePadId id) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    NN_RESULT_THROW(m_XcdFirmwareUpdater.GetDestinationFirmwareVersion(pOutValue, id));
}

Result UniquePadResourceManager::GetDestinationFirmwareVersionForRevert(
    debug::FirmwareVersion *pOutValue,
    system::UniquePadId id) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    NN_RESULT_THROW(m_XcdFirmwareUpdater.GetDestinationFirmwareVersionForRevert(pOutValue, id));
}

Result UniquePadResourceManager::AbortFirmwareUpdate() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    NN_RESULT_THROW(m_XcdFirmwareUpdater.AbortFirmwareUpdate());
}

Result UniquePadResourceManager::IsFirmwareUpdateAvailable(
    bool* pOutIsAvailable,
    system::UniquePadId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutIsAvailable);

    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    NN_RESULT_THROW(m_XcdFirmwareUpdater.IsFirmwareUpdateAvailable(pOutIsAvailable, id));
}

Result UniquePadResourceManager::CheckFirmwareUpdateRequired(
    system::FirmwareUpdateRequiredReason* pOutReason,
    system::UniquePadId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutReason);

    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    NN_RESULT_THROW(m_XcdFirmwareUpdater.CheckFirmwareUpdateRequired(pOutReason, id));
}

Result UniquePadResourceManager::GetFirmwareUpdateStage(debug::FirmwareUpdateStage* pOutStage, uint8_t* pOutProgress) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    NN_RESULT_THROW(m_XcdFirmwareUpdater.GetFirmwareUpdateStage(pOutStage, pOutProgress));
}

Result UniquePadResourceManager::GetFirmwareUpdateState(
    system::FirmwareUpdateState* pOutState,
    system::FirmwareUpdateDeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);

    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    NN_RESULT_THROW(m_XcdFirmwareUpdater.GetFirmwareUpdateState(pOutState, handle));
}


Result UniquePadResourceManager::IsFirmwareUpdatingDevice(
    bool* pOutIsUpdating,
    system::UniquePadId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutIsUpdating);

    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    NN_RESULT_THROW(m_XcdFirmwareUpdater.IsFirmwareUpdatingDevice(pOutIsUpdating, id));
}

Result UniquePadResourceManager::DiscardFirmwareInfoCacheForRevert() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    NN_RESULT_THROW(m_XcdFirmwareUpdater.DiscardSystemDataCacheForRevert());
}

Result UniquePadResourceManager::SetFirmwareHotfixUpdateSkipEnabled(bool isEnabled) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    NN_RESULT_THROW(m_XcdFirmwareUpdater.SetFirmwareHotfixUpdateSkipEnabled(isEnabled));
}

Result UniquePadResourceManager::StartAnalogStickManualCalibration(system::UniquePadId id, system::AnalogStickPosition position) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    auto pManager = this->GetUniquePadManager(id);
    NN_RESULT_THROW_UNLESS(pManager != nullptr, system::ResultUniquePadDisconnected());
    NN_RESULT_DO(
        pManager->GetAnalogStickDriver(position).StartAnalogStickManualCalibration());
    NN_RESULT_SUCCESS;
}

Result UniquePadResourceManager::RetryCurrentAnalogStickManualCalibrationStage(system::UniquePadId id, system::AnalogStickPosition position) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    auto pManager = this->GetUniquePadManager(id);
    NN_RESULT_THROW_UNLESS(pManager != nullptr, system::ResultUniquePadDisconnected());
    NN_RESULT_DO(
        pManager->GetAnalogStickDriver(position).RetryCurrentAnalogStickManualCalibrationStage());
    NN_RESULT_SUCCESS;
}

Result UniquePadResourceManager::CancelAnalogStickManualCalibration(system::UniquePadId id, system::AnalogStickPosition position) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    auto pManager = this->GetUniquePadManager(id);
    if (pManager != nullptr)
    {
        pManager->GetAnalogStickDriver(position).CancelAnalogStickManualCalibration();
    }
    // Cancel はどんなときでも成功する
    NN_RESULT_SUCCESS;
}

Result UniquePadResourceManager::ResetAnalogStickManualCalibration(system::UniquePadId id, system::AnalogStickPosition position) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    auto pManager = this->GetUniquePadManager(id);
    NN_RESULT_THROW_UNLESS(pManager != nullptr, system::ResultUniquePadDisconnected());
    NN_RESULT_DO(
        pManager->GetAnalogStickDriver(position).ResetAnalogStickManualCalibration());
    NN_RESULT_SUCCESS;
}

void UniquePadResourceManager::GetAnalogStickState(AnalogStickState* pOutValue, system::UniquePadId id, system::AnalogStickPosition position) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    auto pManager = this->GetUniquePadManager(id);
    if (pManager != nullptr)
    {
        pManager->GetAnalogStickDriver(position).GetAnalogStickState(pOutValue);
    }
    else
    {
        *pOutValue = AnalogStickState();
    }
}

Result UniquePadResourceManager::GetAnalogStickManualCalibrationStage(system::AnalogStickManualCalibrationStage* pOutValue,
                                                                      system::UniquePadId& id,
                                                                      system::AnalogStickPosition position) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    auto pManager = this->GetUniquePadManager(id);
    NN_RESULT_THROW_UNLESS(pManager != nullptr, system::ResultUniquePadDisconnected());
    NN_RESULT_DO(
        pManager->GetAnalogStickDriver(position).GetAnalogStickManualCalibrationStage(pOutValue));
    NN_RESULT_SUCCESS;
}

bool UniquePadResourceManager::IsAnalogStickButtonPressed(system::UniquePadId id, system::AnalogStickPosition position) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);

    auto pManager = this->GetUniquePadManager(id);
    if (pManager != nullptr)
    {
        return pManager->GetAnalogStickDriver(position).IsAnalogStickButtonPressed();
    }

    return false;
}

bool UniquePadResourceManager::IsAnalogStickInReleasePosition(system::UniquePadId id, system::AnalogStickPosition position) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    auto pManager = this->GetUniquePadManager(id);
    if (pManager != nullptr)
    {
        return pManager->GetAnalogStickDriver(position).IsAnalogStickInReleasePosition();
    }

    return false;
}

bool UniquePadResourceManager::IsAnalogStickInCircumference(system::UniquePadId id, system::AnalogStickPosition position) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    auto pManager = this->GetUniquePadManager(id);
    if (pManager != nullptr)
    {
        return pManager->GetAnalogStickDriver(position).IsAnalogStickInCircumference();
    }

    return false;
}

::nn::Result UniquePadResourceManager::GetUniqueSixAxisSensorHandles(int* pOutCount,
                                                                     system::UniqueSixAxisSensorHandle* outUniqueSixAxisSensorHandles,
                                                                     int count,
                                                                     system::UniquePadId id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    auto pManager = this->GetUniquePadManager(id);
    if (pManager != nullptr)
    {
        NN_RESULT_DO(pManager->GetUniqueSixAxisSensorHandles(pOutCount, outUniqueSixAxisSensorHandles, count));
    }
    NN_RESULT_SUCCESS;
}

::nn::Result UniquePadResourceManager::IsSixAxisSensorUserCalibrationSupported(bool* pOutIsSupported,
                                                                               system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);

    ::nn::hid::system::UniquePadId id = GetUniquePadIdFromUniqueSixAxisSensorHandle(handle);
    auto pManager = this->GetUniquePadManager(id);
    NN_RESULT_THROW_UNLESS(pManager != nullptr, system::ResultUniquePadDisconnected());

    *pOutIsSupported = pManager->GetSixAxisSensorDriver(handle).IsSixAxisSensorUserCalibrationSupported();
    NN_RESULT_SUCCESS;

}

::nn::Result UniquePadResourceManager::ResetSixAxisSensorCalibrationValues(system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);

    ::nn::hid::system::UniquePadId id = GetUniquePadIdFromUniqueSixAxisSensorHandle(handle);
    auto pManager = this->GetUniquePadManager(id);
    NN_RESULT_THROW_UNLESS(pManager != nullptr, system::ResultUniquePadDisconnected());

    NN_RESULT_DO(
        pManager->GetSixAxisSensorDriver(handle).ResetSixAxisSensorCalibrationValues());
    NN_RESULT_SUCCESS;
}

Result UniquePadResourceManager::StartSixAxisSensorUserCalibration(system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);

    ::nn::hid::system::UniquePadId id = GetUniquePadIdFromUniqueSixAxisSensorHandle(handle);
    auto pManager = this->GetUniquePadManager(id);
    NN_RESULT_THROW_UNLESS(pManager != nullptr, system::ResultUniquePadDisconnected());

    NN_RESULT_DO(
        pManager->GetSixAxisSensorDriver(handle).StartSixAxisSensorUserCalibration());
    NN_RESULT_SUCCESS;
}

Result UniquePadResourceManager::CancelSixAxisSensorUserCalibration(system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);

    ::nn::hid::system::UniquePadId id = GetUniquePadIdFromUniqueSixAxisSensorHandle(handle);
    auto pManager = this->GetUniquePadManager(id);
    NN_RESULT_THROW_UNLESS(pManager != nullptr, system::ResultUniquePadDisconnected());

    NN_RESULT_DO(
        pManager->GetSixAxisSensorDriver(handle).CancelSixAxisSensorUserCalibration());
    NN_RESULT_SUCCESS;
}


::nn::Result UniquePadResourceManager::GetSixAxisSensorUserCalibrationStage(system::SixAxisSensorUserCalibrationStage* pOutValue, system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);

    ::nn::hid::system::UniquePadId id = GetUniquePadIdFromUniqueSixAxisSensorHandle(handle);
    auto pManager = this->GetUniquePadManager(id);
    NN_RESULT_THROW_UNLESS(pManager != nullptr, system::ResultUniquePadDisconnected());

    NN_RESULT_DO(
        pManager->GetSixAxisSensorDriver(handle).GetSixAxisSensorUserCalibrationStage(pOutValue));
    NN_RESULT_SUCCESS;
}

::nn::Result UniquePadResourceManager::IsUsbConnected(bool* pOutConnected, system::UniquePadId id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);

    auto pManager = this->GetUniquePadManager(id);
    if (pManager == nullptr)
    {
        *pOutConnected = false;
    }
    else
    {
        *pOutConnected =pManager->IsUsbConnected();
    }
    NN_RESULT_SUCCESS;
}

int UniquePadResourceManager::GetAllowedBluetoothLinksCount() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    return m_AssignmentManager.GetAllowedBluetoothLinksCount();
}

Result UniquePadResourceManager::UpdateControllerColor(const nn::util::Color4u8Type& mainColor,
                                                     const nn::util::Color4u8Type& subColor,
                                                     const system::UniquePadId& id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    auto pManager = this->GetUniquePadManager(id);
    NN_RESULT_THROW_UNLESS(pManager != nullptr, system::ResultUniquePadDisconnected());
    NN_RESULT_DO(pManager->UpdateControllerColor(mainColor, subColor));
    NN_RESULT_SUCCESS;
}

Result UniquePadResourceManager::UpdateDesignInfo(const nn::util::Color4u8Type& mainColor,
                                                const nn::util::Color4u8Type& subColor,
                                                const nn::util::Color4u8Type& thirdColor,
                                                const nn::util::Color4u8Type& forthColor,
                                                uint8_t variation,
                                                const system::UniquePadId& id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    auto pManager = this->GetUniquePadManager(id);
    NN_RESULT_THROW_UNLESS(pManager != nullptr, system::ResultUniquePadDisconnected());
    NN_RESULT_DO(pManager->UpdateDesignInfo(mainColor, subColor, thirdColor, forthColor, variation));
    NN_RESULT_SUCCESS;
}

Result UniquePadResourceManager::ConnectUsbPadsAsync() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    return m_AssignmentManager.ConnectUsbPadsAsync();
}

Result UniquePadResourceManager::DisconnectUsbPadsAsync() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    return m_AssignmentManager.DisconnectUsbPadsAsync();
}

Result UniquePadResourceManager::GetPadDriverState(PadDriverState* pOutState,
                                                   const system::UniquePadId& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    auto pManager = this->GetUniquePadManager(id);
    NN_RESULT_THROW_UNLESS(pManager != nullptr, system::ResultUniquePadDisconnected());
    NN_RESULT_DO(pManager->GetPadDriverState(pOutState));
    NN_RESULT_SUCCESS;
}

Result UniquePadResourceManager::GetSixAxisSensorDriverStates(int* pOutCount,
                                                              SixAxisSensorDriverState* pOutStates,
                                                              int count,
                                                              const system::UniquePadId& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    auto pManager = this->GetUniquePadManager(id);
    NN_RESULT_THROW_UNLESS(pManager != nullptr, system::ResultUniquePadDisconnected());
    NN_RESULT_DO(pManager->GetSixAxisSensorDriverStates(pOutCount, pOutStates, count));
    NN_RESULT_SUCCESS;
}

Result UniquePadResourceManager::GetRxPacketHistory(RxPacketHistory* pOutValue, const system::UniquePadId& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    ::std::lock_guard<decltype(m_UniquePadMutex)> locker(m_UniquePadMutex);
    auto pManager = this->GetUniquePadManager(id);
    NN_RESULT_THROW_UNLESS(pManager != nullptr, system::ResultUniquePadDisconnected());
    NN_RESULT_DO(pManager->GetRxPacketHistory(pOutValue));
    NN_RESULT_SUCCESS;
}

Result UniquePadResourceManager::AcquireSerialFlashEventHandle(
    ::nn::os::NativeHandle* pOutHandle,
    const system::UniquePadId& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutHandle);
    auto pManager = this->GetUniquePadManager(id);
    NN_RESULT_THROW_UNLESS(pManager != nullptr, system::ResultUniquePadDisconnected());
    NN_RESULT_DO(pManager->AcquireSerialFlashEventHandle(pOutHandle));
    NN_RESULT_SUCCESS;
}

Result UniquePadResourceManager::ReadSerialFlash(
    const uint32_t address,
    const nn::os::NativeHandle& osHandle,
    bool isManaged,
    int size,
    const system::UniquePadId& id) NN_NOEXCEPT
{
    auto pManager = this->GetUniquePadManager(id);
    NN_RESULT_THROW_UNLESS(pManager != nullptr, system::ResultUniquePadDisconnected());
    NN_RESULT_DO(pManager->ReadSerialFlash(address, osHandle, isManaged, size));
    NN_RESULT_SUCCESS;
}

Result UniquePadResourceManager::WriteSerialFlash(
    const uint32_t address,
    const nn::os::NativeHandle& osHandle,
    bool isManaged,
    int bufferSize,
    int writeSize,
    const system::UniquePadId& id) NN_NOEXCEPT
{
    auto pManager = this->GetUniquePadManager(id);
    NN_RESULT_THROW_UNLESS(pManager != nullptr, system::ResultUniquePadDisconnected());
    NN_RESULT_DO(pManager->WriteSerialFlash(address, osHandle, isManaged, bufferSize, writeSize));
    NN_RESULT_SUCCESS;
}

Result UniquePadResourceManager::GetSerialFlashResult(const system::UniquePadId& id) NN_NOEXCEPT
{
    auto pManager = this->GetUniquePadManager(id);
    NN_RESULT_THROW_UNLESS(pManager != nullptr, system::ResultUniquePadDisconnected());
    NN_RESULT_DO(pManager->GetSerialFlashResult());
    NN_RESULT_SUCCESS;
}

UniquePadManager* UniquePadResourceManager::GetUniquePadManager(const system::UniquePadId& id) NN_NOEXCEPT
{
    for (auto& manager : m_UniquePadManagers)
    {
        if (manager.GetId()._storage == id._storage)
        {
            return &manager;
        }
    }

    return nullptr;
}

}}} // namespace nn::hid::detail
