﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <functional>
#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/hid/system/hid_UniquePad.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd_SerialFlash.h>

#include "hid_Settings.h"
#include "hid_UniquePadId.h"
#include "hid_UniquePadManager.h"

namespace nn { namespace hid { namespace detail {

UniquePadManager::UniquePadManager() NN_NOEXCEPT
    : m_AbstractedPadHolder()
    , m_SixAxisSensorDriver()
    , m_IsSerialFlashEventInitialized(false)
{
    m_UniquePadId._storage = 0xff;
}

UniquePadManager::~UniquePadManager() NN_NOEXCEPT { /* 何もしない */ }

void UniquePadManager::SetHandheldManager(HandheldManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    for (auto& driver : m_AnalogStickDrivers)
    {
        driver.SetHandheldManager(pManager);
    }
}

void UniquePadManager::Initialize() NN_NOEXCEPT
{
    for (int index = 0; index < AnalogStickPositionCountMax; ++index)
    {
        m_AnalogStickDrivers[index].SetUniquePadAbstractedPadHolder(&m_AbstractedPadHolder);
        m_AnalogStickDrivers[index].SetPosition(GetAnalogStickPositionFromIndex(index));
    }
    m_SixAxisSensorDriver.SetUniquePadAbstractedPadHolder(&m_AbstractedPadHolder);
}

UniquePadAnalogStickDriver& UniquePadManager::GetAnalogStickDriver(system::AnalogStickPosition position) NN_NOEXCEPT
{
    return m_AnalogStickDrivers[GetIndexFromAnalogStickPosition(position)];
}

UniquePadSixAxisSensorDriver& UniquePadManager::GetSixAxisSensorDriver(system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT
{
    NN_UNUSED(handle);
    return m_SixAxisSensorDriver;
}

::nn::Result UniquePadManager::GetUniqueSixAxisSensorHandles(int* pOutCount,
                                                             system::UniqueSixAxisSensorHandle* outUniqueSixAxisSensorHandles,
                                                             int count) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(outUniqueSixAxisSensorHandles);
    NN_UNUSED(count);

    if (m_AbstractedPadHolder.IsConnected() == false)
    {
        *pOutCount = 0;
        NN_RESULT_SUCCESS;
    }

    *outUniqueSixAxisSensorHandles = GetUniqueSixAxisSensorHandleFromUniquePadId(m_UniquePadId);
    *pOutCount = 1;
    NN_RESULT_SUCCESS;
}

void UniquePadManager::AttachAbstractedPad(IAbstractedPad* pPad, system::UniquePadId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPad);

    m_AbstractedPadHolder.AttachAbstractedPad(pPad);
    m_UniquePadId = id;
    for (auto& driver : m_AnalogStickDrivers)
    {
        driver.Reset();
    }
    m_SixAxisSensorDriver.Reset();
}

bool UniquePadManager::CheckForDeviceDetach() NN_NOEXCEPT
{
    if (m_AbstractedPadHolder.CheckForDeviceDetach())
    {
        // デバイスが切断されている場合は、内部の初期化を行う
        for (auto& driver : m_AnalogStickDrivers)
        {
            driver.Reset();
        }
        m_SixAxisSensorDriver.Reset();
        m_UniquePadId._storage = 0xff;

        if (m_IsSerialFlashEventInitialized)
        {
            nn::os::DestroySystemEvent(&m_SerialFlashEvent);
            m_IsSerialFlashEventInitialized = false;
        }
        return true;
    }
    return false;
}

bool UniquePadManager::IsConnected() const NN_NOEXCEPT
{
    return m_AbstractedPadHolder.IsConnected();
}

bool UniquePadManager::IsConnectedAbstractedPad(const AbstractedPadId& id) const NN_NOEXCEPT
{
    return m_AbstractedPadHolder.IsConnectedAbstractedPad(id);
}

bool UniquePadManager::GetIAbstractedPad(IAbstractedPad** ppOutValue) const NN_NOEXCEPT
{
    return m_AbstractedPadHolder.GetIAbstractedPad(ppOutValue);
}

system::UniquePadId UniquePadManager::GetId() NN_NOEXCEPT
{
    return m_UniquePadId;
}

system::UniquePadType UniquePadManager::GetType() NN_NOEXCEPT
{
    IAbstractedPad* pPad;
    if (m_AbstractedPadHolder.GetIAbstractedPad(&pPad) == false)
    {
        return system::UniquePadType_Unknown;
    }

    // 未知のデバイスエミュレーション機能が有効の場合は Unknown で返す
    if (::nn::hid::detail::FirmwareDebugSettings::EmulateFutureDevice())
    {
        return system::UniquePadType_Unknown;
    }

    auto deviceType = pPad->GetDeviceType();

    NN_SDK_REQUIRES_EQUAL(deviceType.CountPopulation(), 1);
    if (deviceType == system::DeviceType::FullKeyController::Mask)
    {
        return system::UniquePadType_FullKeyController;
    }
    if (deviceType == system::DeviceType::JoyConLeft::Mask ||
        deviceType == system::DeviceType::HandheldJoyLeft::Mask)
    {
        return system::UniquePadType_LeftController;
    }
    if (deviceType == system::DeviceType::JoyConRight::Mask ||
        deviceType == system::DeviceType::HandheldJoyRight::Mask)
    {
        return system::UniquePadType_RightController;
    }

    return system::UniquePadType_Unknown;
}

::nn::Result UniquePadManager::GetBluetoothAddress(::nn::bluetooth::Address* pOutAddress) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        m_AbstractedPadHolder.IsConnected(),
        system::ResultUniquePadDisconnected());

    nn::xcd::DeviceHandle xcdHandle;
    NN_RESULT_THROW_UNLESS(
        m_AbstractedPadHolder.GetXcdDeviceHandle(&xcdHandle),
        system::ResultUniquePadNoBluetoothAddress());

    ::nn::xcd::DeviceInfo info;
    NN_RESULT_THROW_UNLESS(
        ::nn::xcd::GetDeviceInfo(&info, xcdHandle).IsSuccess(),
        system::ResultUniquePadDisconnected());

    if(info.deviceType == ::nn::xcd::DeviceType_MiyabiLeft ||
       info.deviceType == ::nn::xcd::DeviceType_MiyabiRight)
    {
        return system::ResultUniquePadNoBluetoothAddress();
    }

    *pOutAddress = info.address;
    NN_RESULT_SUCCESS;
}

::nn::Result UniquePadManager::GetInterface(system::UniquePadInterface* pOutValue) NN_NOEXCEPT
{
    IAbstractedPad* pPad;
    NN_RESULT_THROW_UNLESS(
        m_AbstractedPadHolder.GetIAbstractedPad(&pPad),
        system::ResultUniquePadDisconnected());

    switch(pPad->GetInterfaceType())
    {
    case system::InterfaceType_Bluetooth:
        *pOutValue = ::nn::hid::system::UniquePadInterface_Bluetooth;
        break;
    case system::InterfaceType_Rail:
        *pOutValue = ::nn::hid::system::UniquePadInterface_Rail;
        break;
    case system::InterfaceType_Usb:
        *pOutValue = ::nn::hid::system::UniquePadInterface_Usb;
        break;
    default:
        NN_ABORT("Unexpected InterfaceType (%x) returned", pPad->GetInterfaceType());
        NN_UNEXPECTED_DEFAULT;
    }
    NN_RESULT_SUCCESS;
}

::nn::Result UniquePadManager::GetSerialNumber(system::UniquePadSerialNumber* pOutValue) NN_NOEXCEPT
{
    IAbstractedPad* pPad;
    NN_RESULT_THROW_UNLESS(
        m_AbstractedPadHolder.GetIAbstractedPad(&pPad),
        system::ResultUniquePadDisconnected());
    auto deviceType = pPad->GetDeviceType();

    nn::xcd::DeviceHandle xcdHandle;
    if (m_AbstractedPadHolder.GetXcdDeviceHandle(&xcdHandle))
    {
        if (deviceType.Test<system::DeviceType::JoyConLeft>() ||
            deviceType.Test<system::DeviceType::HandheldJoyLeft>() ||
            deviceType.Test<system::DeviceType::JoyConRight>() ||
            deviceType.Test<system::DeviceType::HandheldJoyRight>())
        {
            // Joy-Con の場合はそのまま SerialNumber を渡す
            nn::xcd::IdentificationCode code;
            nn::xcd::GetIdentificationCode(&code, xcdHandle);
            NN_ABORT_UNLESS_GREATER_EQUAL(::nn::hid::system::UniquePadSerialNumberLength, ::nn::xcd::IdentificationCodeLength);

            std::memcpy(pOutValue->_storage, code._code, ::nn::xcd::IdentificationCodeLength);
        }
        else if(deviceType.Test<system::DeviceType::FullKeyController>())
        {
            // Switch Pro Controller の場合は BdAddress を変換して渡す
            nn::xcd::DeviceInfo info;
            nn::xcd::GetDeviceInfo(&info, xcdHandle);
            std::memcpy(pOutValue->_storage, info.address.address, sizeof(info.address));
        }
    }
    else
    {
        auto abstractedPadId = pPad->GetId();
        std::memset(pOutValue->_storage, 0xff, sizeof(system::UniquePadSerialNumber));
        std::memcpy(pOutValue->_storage, &abstractedPadId, sizeof(abstractedPadId));
    }
    NN_RESULT_SUCCESS;
}

int UniquePadManager::GetControllerNumber() NN_NOEXCEPT
{
    IAbstractedPad* pPad;
    if (m_AbstractedPadHolder.GetIAbstractedPad(&pPad) == false)
    {
        return 0;
    }

    return pPad->GetControllerNumber();
}

bool UniquePadManager::IsUsbConnected() NN_NOEXCEPT
{
    IAbstractedPad* pPad;
    if (m_AbstractedPadHolder.GetIAbstractedPad(&pPad) == false)
    {
        return false;
    }
    return pPad->IsUsbConnected();
}

::nn::Result UniquePadManager::Disconnect() NN_NOEXCEPT
{
    IAbstractedPad* pPad;
    NN_RESULT_THROW_UNLESS(
        m_AbstractedPadHolder.GetIAbstractedPad(&pPad),
        system::ResultUniquePadDisconnected());
    auto interfaceType = pPad->GetInterfaceType();
    NN_RESULT_THROW_UNLESS(
        interfaceType == system::InterfaceType_Bluetooth || interfaceType == system::InterfaceType_Usb,
        system::ResultUniquePadNoWirelessConnection());
    pPad->Detach();
    NN_RESULT_SUCCESS;
}

::nn::Result UniquePadManager::UpdateControllerColor(const nn::util::Color4u8Type& mainColor,
                                             const nn::util::Color4u8Type& subColor) NN_NOEXCEPT
{
    nn::xcd::DeviceHandle handle;
    NN_RESULT_THROW_UNLESS(m_AbstractedPadHolder.GetXcdDeviceHandle(&handle), system::ResultUniquePadDisconnected());
    NN_RESULT_THROW_UNLESS(::nn::xcd::UpdateDeviceColor(mainColor, subColor, handle).IsSuccess(), system::ResultUniquePadDisconnected());
    NN_RESULT_SUCCESS;
}

::nn::Result UniquePadManager::UpdateDesignInfo(const nn::util::Color4u8Type& mainColor,
                                        const nn::util::Color4u8Type& subColor,
                                        const nn::util::Color4u8Type& thirdColor,
                                        const nn::util::Color4u8Type& forthColor,
                                        uint8_t variation) NN_NOEXCEPT
{
    nn::xcd::DeviceHandle handle;
    NN_RESULT_THROW_UNLESS(m_AbstractedPadHolder.GetXcdDeviceHandle(&handle), system::ResultUniquePadDisconnected());
    ::nn::xcd::DeviceColor color = { { { mainColor, subColor, thirdColor, forthColor } } };
    NN_RESULT_THROW_UNLESS(::nn::xcd::UpdateDesignInfo(color, variation, handle).IsSuccess(), system::ResultUniquePadDisconnected());
    NN_RESULT_SUCCESS;
}

Result UniquePadManager::GetPadDriverState(PadDriverState* pOutState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);
    IAbstractedPad* pPad;
    NN_RESULT_THROW_UNLESS(
        m_AbstractedPadHolder.GetIAbstractedPad(&pPad),
        system::ResultUniquePadDisconnected());
    pPad->GetPadDriverState(pOutState);
    NN_RESULT_SUCCESS;

}

Result UniquePadManager::GetSixAxisSensorDriverStates(int* pOutCount,
                                                      SixAxisSensorDriverState* pOutStates,
                                                      int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);
    IAbstractedPad* pPad;
    NN_RESULT_THROW_UNLESS(
        m_AbstractedPadHolder.GetIAbstractedPad(&pPad),
        system::ResultUniquePadDisconnected());
    *pOutCount = pPad->GetSixAxisSensorDriverStates(pOutStates, count);
    NN_RESULT_SUCCESS;
}

Result UniquePadManager::GetRxPacketHistory(RxPacketHistory* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    IAbstractedPad* pPad;
    NN_RESULT_THROW_UNLESS(
        m_AbstractedPadHolder.GetIAbstractedPad(&pPad),
        system::ResultUniquePadDisconnected());
    *pOutValue = pPad->GetRxPacketHistory();
    NN_RESULT_SUCCESS;
}


void UniquePadManager::Sample() NN_NOEXCEPT
{
    for (auto& driver : m_AnalogStickDrivers)
    {
        driver.Update();
    }
    m_SixAxisSensorDriver.Update();
}

::nn::Result UniquePadManager::AcquireSerialFlashEventHandle(::nn::os::NativeHandle* pOutHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutHandle);

    if (m_IsSerialFlashEventInitialized == false)
    {
        nn::os::CreateSystemEvent(&m_SerialFlashEvent, ::nn::os::EventClearMode_ManualClear, true);
        m_IsSerialFlashEventInitialized = true;
    }

    *pOutHandle = nn::os::GetReadableHandleOfSystemEvent(&m_SerialFlashEvent);

    NN_RESULT_SUCCESS;
}

Result UniquePadManager::ReadSerialFlash(
    const uint32_t address,
    const nn::os::NativeHandle& osHandle,
    bool isManaged,
    int size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsSerialFlashEventInitialized);

    nn::xcd::DeviceHandle xcdHandle;
    NN_RESULT_THROW_UNLESS(
        m_AbstractedPadHolder.GetXcdDeviceHandle(&xcdHandle),
        system::ResultUniquePadDisconnected());

    ::nn::os::AttachTransferMemory(&m_TransferMemory, static_cast<size_t>(size), osHandle, isManaged);
    uint8_t* buffer = nullptr;
    NN_RESULT_DO(::nn::os::MapTransferMemory(reinterpret_cast<void**>(&buffer), &m_TransferMemory, ::nn::os::MemoryPermission_ReadWrite));

    auto result = ::nn::xcd::ReadSerialFlash(address, buffer, size, &m_SerialFlashEvent, xcdHandle);
    if (result.IsFailure())
    {
        ::nn::os::UnmapTransferMemory(&m_TransferMemory);
        ::nn::os::DestroyTransferMemory(&m_TransferMemory);
    }
    NN_RESULT_TRY(result);
    NN_RESULT_CATCH_CONVERT(::nn::xcd::ResultNotConnected, system::ResultUniquePadDisconnected());
    NN_RESULT_CATCH_CONVERT(::nn::xcd::ResultSerialFlashOnWrite, ResultUniquePadSerialFlashAccessBusy());
    NN_RESULT_CATCH_CONVERT(::nn::xcd::ResultSerialFlashOnRead, ResultUniquePadSerialFlashAccessBusy());
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

Result UniquePadManager::WriteSerialFlash(
    const uint32_t address,
    const nn::os::NativeHandle& osHandle,
    bool isManaged,
    int bufferSize,
    int writeSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsSerialFlashEventInitialized);

    nn::xcd::DeviceHandle xcdHandle;
    NN_RESULT_THROW_UNLESS(
        m_AbstractedPadHolder.GetXcdDeviceHandle(&xcdHandle),
        system::ResultUniquePadDisconnected());

    ::nn::os::AttachTransferMemory(&m_TransferMemory, static_cast<size_t>(bufferSize), osHandle, isManaged);
    uint8_t* buffer = nullptr;
    NN_RESULT_DO(::nn::os::MapTransferMemory(reinterpret_cast<void**>(&buffer), &m_TransferMemory, ::nn::os::MemoryPermission_ReadOnly));

    NN_RESULT_TRY(::nn::xcd::WriteSerialFlash(address, buffer, writeSize, &m_SerialFlashEvent, xcdHandle));
    NN_RESULT_CATCH_CONVERT(::nn::xcd::ResultNotConnected, system::ResultUniquePadDisconnected());
    NN_RESULT_CATCH_CONVERT(::nn::xcd::ResultSerialFlashOnWrite, ResultUniquePadSerialFlashAccessBusy());
    NN_RESULT_CATCH_CONVERT(::nn::xcd::ResultSerialFlashOnRead, ResultUniquePadSerialFlashAccessBusy());
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

Result UniquePadManager::GetSerialFlashResult() NN_NOEXCEPT
{
    nn::xcd::DeviceHandle xcdHandle;
    NN_RESULT_THROW_UNLESS(
        m_AbstractedPadHolder.GetXcdDeviceHandle(&xcdHandle),
        system::ResultUniquePadDisconnected());

    NN_RESULT_TRY(::nn::xcd::GetSerialFlashResult(xcdHandle));
    NN_RESULT_CATCH_CONVERT(::nn::xcd::ResultNotConnected, system::ResultUniquePadDisconnected());
    NN_RESULT_CATCH_CONVERT(::nn::xcd::ResultSerialFlashNoResult, ResultUniquePadSerialFlashNoOperation());
    NN_RESULT_CATCH_CONVERT(::nn::xcd::ResultSerialFlashVerifyError, ResultUniquePadSerialFlashVerifyError());
    NN_RESULT_END_TRY

    ::nn::os::UnmapTransferMemory(&m_TransferMemory);
    ::nn::os::DestroyTransferMemory(&m_TransferMemory);

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
