﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/applet/applet.h>
#include <nn/hid/hid_IHidSystemServer.sfdl.h>
#include <nn/hid/hid_NpadJoyCommon.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/system/hid_Result.system.h>
#include <nn/hid/system/hid_UniquePad.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_NativeHandle.h>

#include "hid_AppletResourceUserId.h"
#include "hid_HidDebugServer.h"
#include "hid_HidSystemServer.h"
#include "hid_UniquePadId.h"
#include "hid_UniquePadImpl.h"
#include "hid_Session.h"
#include "hid_StaticObject.h"

namespace nn { namespace hid { namespace detail {

::nn::Result BindUniquePadConnectionEvent(::nn::os::SystemEventType* pEvent,
                                          ::nn::os::EventClearMode clearMode) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    auto handle = ::nn::sf::NativeHandle();
    NN_RESULT_DO(pProxy->AcquireUniquePadConnectionEventHandle(&handle));

    ::nn::os::AttachReadableHandleToSystemEvent(pEvent,
                                                handle.GetOsHandle(),
                                                handle.IsManaged(),
                                                clearMode);

    // ハンドルの管理権を放棄
    handle.Detach();

    NN_RESULT_SUCCESS;
}

::nn::Result ListUniquePads(int* outGotCount,
                            ::nn::hid::system::UniquePadId* outValues,
                            int count) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(outGotCount);
    NN_ABORT_UNLESS_NOT_NULL(outValues);

    NN_RESULT_THROW_UNLESS(outGotCount != nullptr,
                           ResultUniquePadNullIdCountBuffer());
    NN_RESULT_THROW_UNLESS(count >= 0, ResultUniquePadInvalidIdCount());
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);
    auto gotCount = int64_t();
    NN_RESULT_DO(
        pProxy->GetUniquePadIds(::nn::sf::Out<int64_t>(&gotCount),
                                ::nn::sf::OutArray<system::UniquePadId>(
                                outValues, static_cast<size_t>(count))));
    *outGotCount = static_cast<int>(gotCount);
    NN_RESULT_SUCCESS;
}

::nn::Result GetUniquePadType(::nn::hid::system::UniquePadType * pOutValue,
                              ::nn::hid::system::UniquePadId id) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutValue);

    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);
    auto gotValue = int64_t();
    NN_RESULT_DO(
        pProxy->GetUniquePadType(::nn::sf::Out<int64_t>(&gotValue),
                                 id));
    *pOutValue = static_cast<nn::hid::system::UniquePadType>(gotValue);
    NN_RESULT_SUCCESS;
}

::nn::Result GetUniquePadInterface(::nn::hid::system::UniquePadInterface* pOutValue,
                                   ::nn::hid::system::UniquePadId id) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutValue);

    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);
    auto gotValue = int64_t();
    NN_RESULT_DO(
        pProxy->GetUniquePadInterface(::nn::sf::Out<int64_t>(&gotValue),
                                      id));
    *pOutValue = static_cast<nn::hid::system::UniquePadInterface>(gotValue);
    NN_RESULT_SUCCESS;
}

::nn::Result GetUniquePadControllerNumber(int* pOutValue,
                                          ::nn::hid::system::UniquePadId id) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutValue);

    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);
    auto gotValue = int64_t();
    NN_RESULT_DO(
        pProxy->GetUniquePadControllerNumber(::nn::sf::Out<int64_t>(&gotValue),
                                             id));
    *pOutValue = static_cast<int>(gotValue);
    NN_RESULT_SUCCESS;
}

::nn::Result GetUniquePadSerialNumber(::nn::hid::system::UniquePadSerialNumber *pOutValue,
                                      ::nn::hid::system::UniquePadId id) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutValue);

    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);
    NN_RESULT_DO(
        pProxy->GetUniquePadSerialNumber(::nn::sf::Out<system::UniquePadSerialNumber>(pOutValue),
                                         id));
    NN_RESULT_SUCCESS;
}

::nn::Result GetUniquePadBluetoothAddress(nn::bluetooth::Address *pOutValue, system::UniquePadId id) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutValue);

    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    NN_RESULT_DO(
        pProxy->GetUniquePadBluetoothAddress(::nn::sf::Out<::nn::bluetooth::Address>(pOutValue),
                                             id));
    NN_RESULT_SUCCESS;
}

::nn::Result DisconnectUniquePad(system::UniquePadId id) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    NN_RESULT_DO(
        pProxy->DisconnectUniquePad(id));
    NN_RESULT_SUCCESS;
}

::nn::Result ListSixAxisSensorHandles(int* pOutCount,
                                      system::UniqueSixAxisSensorHandle* pOutValues,
                                      system::UniquePadId id,
                                      int count) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutCount);
    NN_ABORT_UNLESS_NOT_NULL(pOutValues);
    NN_RESULT_THROW_UNLESS(count >= 0, ResultUniquePadInvalidIdCount());

    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    auto gotCount = int64_t();
    NN_RESULT_DO(
        pProxy->ListSixAxisSensorHandles(::nn::sf::Out<int64_t>(&gotCount),
                                         ::nn::sf::OutArray<system::UniqueSixAxisSensorHandle>(
                                             pOutValues, static_cast<size_t>(count)),
                                         id));
    *pOutCount = static_cast<int>(gotCount);
    NN_RESULT_SUCCESS;
}

::nn::Result IsSixAxisSensorUserCalibrationSupported(bool* pOutIsSupported,
                                                     system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutIsSupported);

    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    auto outIsSupported = bool();
    NN_RESULT_DO(pProxy->IsSixAxisSensorUserCalibrationSupported(::nn::sf::Out<bool>(&outIsSupported),
                                                                 handle));
    *pOutIsSupported = static_cast<bool>(outIsSupported);
    NN_RESULT_SUCCESS;
}

::nn::Result ResetSixAxisSensorCalibrationValues(system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->ResetSixAxisSensorCalibrationValues(handle));
    NN_RESULT_SUCCESS;
}

::nn::Result StartSixAxisSensorUserCalibration(system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->StartSixAxisSensorUserCalibration(handle));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSixAxisSensorUserCalibrationStage(system::SixAxisSensorUserCalibrationStage* pOutValue,
                                                  system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);
    auto gotValue = int64_t();
    auto result = pProxy->GetSixAxisSensorUserCalibrationStage(::nn::sf::Out<int64_t>(&gotValue),
                                                               handle);
    *pOutValue = static_cast<system::SixAxisSensorUserCalibrationStage>(gotValue);

    NN_RESULT_DO(result);
    NN_RESULT_SUCCESS;
}

::nn::Result CancelSixAxisSensorUserCalibration(system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->CancelSixAxisSensorUserCalibration(handle));
    NN_RESULT_SUCCESS;
}

::nn::Result GetAnalogStickState(AnalogStickState* pOutValue, system::UniquePadId id, system::AnalogStickPosition position) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);
    NN_RESULT_DO(
        pProxy->GetAnalogStickState(::nn::sf::Out<AnalogStickState>(pOutValue),
                                    id,
                                    position));

    NN_RESULT_SUCCESS;
}


::nn::Result GetAnalogStickManualCalibrationStage(system::AnalogStickManualCalibrationStage* pOutValue,
                                                  system::UniquePadId id,
                                                  system::AnalogStickPosition position) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);
    auto gotValue = int64_t();
    NN_RESULT_DO(
        pProxy->GetAnalogStickManualCalibrationStage(::nn::sf::Out<int64_t>(&gotValue),
                                                     id,
                                                     position));
    *pOutValue = static_cast<system::AnalogStickManualCalibrationStage>(gotValue);
    NN_RESULT_SUCCESS;
}

::nn::Result IsAnalogStickButtonPressed(bool* pOutValue, system::UniquePadId id, system::AnalogStickPosition position) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);
    NN_RESULT_DO(
        pProxy->IsAnalogStickButtonPressed(::nn::sf::Out<bool>(pOutValue),
                                           id,
                                           position));
    NN_RESULT_SUCCESS;
}

::nn::Result IsAnalogStickInReleasePosition(bool* pOutValue, system::UniquePadId id, system::AnalogStickPosition position) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);
    NN_RESULT_DO(
        pProxy->IsAnalogStickInReleasePosition(::nn::sf::Out<bool>(pOutValue),
                                               id,
                                               position));
    NN_RESULT_SUCCESS;
}

::nn::Result IsAnalogStickInCircumference(bool* pOutValue, system::UniquePadId id, system::AnalogStickPosition position) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);
    NN_RESULT_DO(
        pProxy->IsAnalogStickInCircumference(::nn::sf::Out<bool>(pOutValue),
                                             id,
                                             position));
    NN_RESULT_SUCCESS;
}

::nn::Result UpdateControllerColor(const ::nn::util::Color4u8Type& mainColor,
                                   const ::nn::util::Color4u8Type& subColor,
                                   const system::UniquePadId& id) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidDebugServer>();
    CreateHidDebugServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->UpdateControllerColor(mainColor, subColor, id));
    NN_RESULT_SUCCESS;
}

::nn::Result UpdateDesignInfo(const ::nn::util::Color4u8Type& mainColor,
                              const ::nn::util::Color4u8Type& subColor,
                              const ::nn::util::Color4u8Type& thirdColor,
                              const ::nn::util::Color4u8Type& forthColor,
                              uint8_t variation,
                              const system::UniquePadId& id) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidDebugServer>();
    CreateHidDebugServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->UpdateDesignInfo(mainColor, subColor, thirdColor, forthColor, variation, id));
    NN_RESULT_SUCCESS;
}

::nn::Result ConnectUsbPadsAsync() NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidDebugServer>();
    CreateHidDebugServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->ConnectUsbPadsAsync());
    NN_RESULT_SUCCESS;
}

::nn::Result DisconnectUsbPadsAsync() NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidDebugServer>();
    CreateHidDebugServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->DisconnectUsbPadsAsync());
    NN_RESULT_SUCCESS;
}

::nn::Result IsSixAxisSensorAccurateUserCalibrationSupported(bool* pOutIsSupported,
                                                             const system::UniqueSixAxisSensorHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutIsSupported);

    ::nn::sf::SharedPointer<IHidSystemServer> pHidSystemServer;
    NN_RESULT_DO(CreateHidSystemServerProxy(&pHidSystemServer));

    NN_RESULT_DO(pHidSystemServer->IsSixAxisSensorAccurateUserCalibrationSupported(pOutIsSupported,
                                                                                   handle,
                                                                                   GetAppletResourceUserId()));

    NN_RESULT_SUCCESS;
}

::nn::Result StartSixAxisSensorAccurateUserCalibration(const system::UniqueSixAxisSensorHandle& handle) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidSystemServer> pHidSystemServer;
    NN_RESULT_DO(CreateHidSystemServerProxy(&pHidSystemServer));

    NN_RESULT_DO(pHidSystemServer->StartSixAxisSensorAccurateUserCalibration(handle,
                                                                             GetAppletResourceUserId()));

    NN_RESULT_SUCCESS;
}

::nn::Result GetSixAxisSensorAccurateUserCalibrationState(system::SixAxisSensorAccurateUserCalibrationState* pOutValue,
                                                          const system::UniqueSixAxisSensorHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);
    auto gotValue = system::SixAxisSensorAccurateUserCalibrationState();
    auto result = pProxy->GetSixAxisSensorAccurateUserCalibrationState(::nn::sf::Out<system::SixAxisSensorAccurateUserCalibrationState>(&gotValue),
                                                                       handle,
                                                                       GetAppletResourceUserId());
    *pOutValue = static_cast<system::SixAxisSensorAccurateUserCalibrationState>(gotValue);

    NN_RESULT_DO(result);
    NN_RESULT_SUCCESS;
}

::nn::Result CancelSixAxisSensorAccurateUserCalibration(const system::UniqueSixAxisSensorHandle& handle) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidSystemServer> pHidSystemServer;
    NN_RESULT_DO(CreateHidSystemServerProxy(&pHidSystemServer));

    NN_RESULT_DO(pHidSystemServer->CancelSixAxisSensorAccurateUserCalibration(handle,
                                                                              GetAppletResourceUserId()));

    NN_RESULT_SUCCESS;
}

::nn::Result GetUniquePadDriverState(PadDriverState* pOutValue,
                                     const system::UniquePadId& id) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidDebugServer>();
    CreateHidDebugServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->GetUniquePadDriverState(::nn::sf::Out<PadDriverState>(pOutValue), id));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSixAxisSensorDriverStates(int* pOutCount,
                                          SixAxisSensorDriverState* pOutStates,
                                          int count,
                                          const system::UniquePadId& id) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidDebugServer>();
    CreateHidDebugServerProxy(&pProxy);

    int64_t outCount;
    NN_RESULT_DO(pProxy->GetSixAxisSensorDriverStates(::nn::sf::Out<int64_t>(&outCount),
                                                      ::nn::sf::OutArray<SixAxisSensorDriverState>(pOutStates, static_cast<size_t>(count)),
                                                      id));
    *pOutCount = static_cast<int>(outCount);
    NN_RESULT_SUCCESS;
}

::nn::Result GetRxPacketHistory(RxPacketHistory* pOutValue,
                                const system::UniquePadId& id) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidDebugServer>();
    CreateHidDebugServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->GetRxPacketHistory(::nn::sf::Out<RxPacketHistory>(pOutValue), id));
    NN_RESULT_SUCCESS;
}


::nn::Result BindSerialFlashEventHandle(
    ::nn::os::SystemEventType* pEvent,
    ::nn::os::EventClearMode clearMode,
    const system::UniquePadId& id) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pEvent);

    auto pProxy = ::nn::sf::SharedPointer<IHidDebugServer>();
    CreateHidDebugServerProxy(&pProxy);

    auto handle = ::nn::sf::NativeHandle();
    NN_RESULT_DO(pProxy->AcquireSerialFlashEventHandle(&handle, id));

    ::nn::os::AttachReadableHandleToSystemEvent(pEvent,
                                                handle.GetOsHandle(),
                                                handle.IsManaged(),
                                                clearMode);

    // ハンドルの管理権を放棄
    handle.Detach();

    NN_RESULT_SUCCESS;
}

::nn::Result ReadSerialFlash(
    const uint32_t address,
    uint8_t* pOutBuffer,
    int size,
    const system::UniquePadId& id) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutBuffer);

    auto pProxy = ::nn::sf::SharedPointer<IHidDebugServer>();
    CreateHidDebugServerProxy(&pProxy);

    ::nn::os::TransferMemory transferBuffer(reinterpret_cast<void*>(pOutBuffer), size, ::nn::os::MemoryPermission_ReadWrite);
    auto nativeHandle = nn::sf::NativeHandle(transferBuffer.Detach(), true);

    NN_RESULT_DO(pProxy->ReadSerialFlash(
        address,
        std::move(nativeHandle),
        static_cast<uint64_t>(size),
        id));

    NN_RESULT_SUCCESS;
}

::nn::Result WriteSerialFlash(
    const uint32_t address,
    const uint8_t* pBuffer,
    int bufferSize,
    int writeSize,
    const system::UniquePadId& id) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pBuffer);

    auto pProxy = ::nn::sf::SharedPointer<IHidDebugServer>();
    CreateHidDebugServerProxy(&pProxy);

    ::nn::os::TransferMemory transferBuffer(const_cast<void*>(reinterpret_cast<const void*>(pBuffer)), bufferSize, ::nn::os::MemoryPermission_ReadOnly);
    auto nativeHandle = nn::sf::NativeHandle(transferBuffer.Detach(), true);

    NN_RESULT_DO(pProxy->WriteSerialFlash(
        address,
        std::move(nativeHandle),
        static_cast<uint64_t>(bufferSize),
        static_cast<uint64_t>(writeSize),
        id));

    NN_RESULT_SUCCESS;
}

::nn::Result GetSerialFlashResult(const system::UniquePadId& id) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidDebugServer>();
    CreateHidDebugServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->GetSerialFlashResult(id));

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
