﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_Windows.h>
#include <nn/hid/hid_TouchScreen.h>

#include "hid_ActivationCount.h"
#include "hid_ITouchScreenDriver.h"
#include "hid_Rectangle.h"
#include "hid_Vector.h"
#include "hid_WindowsProcedure-os.win.h"

namespace nn { namespace hid { namespace detail {

//!< タッチ入力の識別子を管理するためのクラスです。
class TouchInputIdManager final
{
    NN_DISALLOW_COPY(TouchInputIdManager);
    NN_DISALLOW_MOVE(TouchInputIdManager);

private:
    //!< タッチ入力の識別子のエントリを表す構造体です。
    struct Entry
    {
        bool hasValue;
        DWORD value;
    };

    //!< タッチ入力の識別子のエントリ
    Entry m_Entries[TouchStateCountMax];

public:
    TouchInputIdManager() NN_NOEXCEPT
    {
        this->UnregisterAllTouchInputId();
    }

    ~TouchInputIdManager() NN_NOEXCEPT { /* 何もしない */ }

    //!< タッチ入力の識別子を登録します。
    bool RegisterTouchInputId(DWORD value) NN_NOEXCEPT;

    //!< タッチ入力の識別子の登録を解除します。
    bool UnregisterTouchInputId(DWORD value) NN_NOEXCEPT;

    //!< インデックス指定によってタッチ入力の識別子の登録を解除します。
    bool UnregisterTouchInputIdByIndex(size_t index) NN_NOEXCEPT;

    //!< 全てのタッチ入力の識別子の登録を解除します。
    void UnregisterAllTouchInputId() NN_NOEXCEPT;

    //!< マウスのものを除いて全てのタッチ入力の識別子の登録を解除します。
    void UnregisterAllTouchInputIdExceptMouse() NN_NOEXCEPT;

    //!< タッチ入力の識別子を検索します。
    bool FindTouchInputId(size_t* pOutValue, DWORD value) const NN_NOEXCEPT;
};

class TouchScreenDriver final : public ITouchScreenDriver
{
    NN_DISALLOW_COPY(TouchScreenDriver);
    NN_DISALLOW_MOVE(TouchScreenDriver);

private:
    //!< このインスタンスからドライバがアクティブ化された回数
    ActivationCount m_ActivationCount;

    //!< 起床状態にあるか否か
    bool m_IsAwake;

    //!< タッチ入力識別子マネージャ
    TouchInputIdManager m_TouchInputIdManager;

    //!< サンプリング番号
    int64_t m_SamplingNumber;

    //!< タッチの識別子
    int32_t m_FingerId;

    //!< タイムスタンプ
    ::nn::TimeSpanType m_TimeStamps[TouchInputCountMax];

    //!< タッチ入力の読み出し先
    TOUCHINPUT m_TouchInputs[TouchInputCountMax];

    //!< 識別子
    Vector<size_t, TouchStateCountMax> m_TouchIds;

    //!< 新たな識別子
    Vector<size_t, TouchStateCountMax> m_NewIds;

    //!< 去った識別子
    Vector<size_t, TouchStateCountMax> m_OldIds;

    //!< タッチの配列
    TouchState m_Touches[TouchStateCountMax];

    //!< TouchScreen の HidShell 向けに共有される自動操作状態
    ::nn::hid::debug::TouchScreenAutoPilotState<TouchStateCountMax
        > m_AutoPilotState;

public:
    TouchScreenDriver() NN_NOEXCEPT;

    virtual ~TouchScreenDriver() NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result Activate() NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result Deactivate() NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result Wake() NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result Sleep() NN_NOEXCEPT NN_OVERRIDE;

    virtual void GetState(TouchScreenState<TouchStateCountMax>* pOutValue
                          ) NN_NOEXCEPT NN_OVERRIDE;

private:
    //!< マウス入力の状態を反映します。
    void UpdateMouseInputState(const Rectangle& rect) NN_NOEXCEPT;

    //!< タッチ入力の状態を反映します。
    void UpdateTouchInputState() NN_NOEXCEPT;

    //!< タッチ入力の Down イベントを処理します。
    void ProcessTouchInputDownEvent(const TOUCHINPUT& touchInput) NN_NOEXCEPT;

    //!< タッチ入力の Up イベントを処理します。
    void ProcessTouchInputUpEvent(const TOUCHINPUT& touchInput) NN_NOEXCEPT;

    //!< タッチ入力の Move イベントを処理します。
    void ProcessTouchInputMoveEvent(const TOUCHINPUT& touchInput) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
