﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nnd/ftm/ftm.h>

#include "hid_ActivationCount.h"
#include "hid_ITouchScreenDriver.h"
#include "hid_Vector.h"

namespace nn { namespace hid { namespace detail {

class TouchScreenDriver final : public ITouchScreenDriver
{
    NN_DISALLOW_COPY(TouchScreenDriver);
    NN_DISALLOW_MOVE(TouchScreenDriver);

private:
    //!< このインスタンスからドライバがアクティブ化された回数
    ActivationCount m_ActivationCount;

    //!< デバイスが利用可能か否か
    bool m_IsDeviceInitialized;

    //!< デバイスが利用可能か否か
    bool m_IsDeviceAvailable;

    //!< 起床状態にあるか否か
    bool m_IsAwake;

    //!< サンプリング番号
    int64_t m_SamplingNumber;

    //!< タッチの識別子
    int32_t m_FingerId;

    //!< タイムスタンプ
    ::nn::TimeSpan m_LastTime;

    //!< イベント FIFO 上のバイトデータ
    char m_ReadData[::nnd::ftm::FtmMaxEventReportByteSize *
                    ::nnd::ftm::FtmMaxEventReportCount];

    //!< イベントレポートの配列
    ::nnd::ftm::EventReport m_Reports[
        ::nnd::ftm::FtmMaxEventReportCount];

    //!< 識別子の変換表
    size_t m_TouchIdMap[TouchStateCountMax];

    //!< 識別子
    Vector<size_t, TouchStateCountMax> m_TouchIds;

    //!< 新たな識別子
    Vector<size_t, TouchStateCountMax> m_NewIds;

    //!< 去った識別子
    Vector<size_t, TouchStateCountMax> m_OldIds;

    //!< タッチの配列
    TouchState m_Touches[TouchStateCountMax];

public:
    TouchScreenDriver() NN_NOEXCEPT;

    virtual ~TouchScreenDriver() NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result Activate() NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result Deactivate() NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result Wake() NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result Sleep() NN_NOEXCEPT NN_OVERRIDE;

    virtual void GetState(
        TouchScreenState<TouchStateCountMax>* pOutValue
        ) NN_NOEXCEPT NN_OVERRIDE;

    void DeassertReset() NN_NOEXCEPT;

private:
    //!< イベントレポートを処理します。
    void ProcessReports(
        const ::nnd::ftm::EventReport reports[], int count) NN_NOEXCEPT;

    //!< TouchEnter イベントを処理します。
    void ProcessTouchEnterEvent(
        const ::nnd::ftm::TouchEventReport& report) NN_NOEXCEPT;

    //!< TouchLeave イベントを処理します。
    void ProcessTouchLeaveEvent(
        const ::nnd::ftm::TouchEventReport& report) NN_NOEXCEPT;

    //!< TouchMotion イベントを処理します。
    void ProcessTouchMotionEvent(
        const ::nnd::ftm::TouchEventReport& report) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
