﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/bpc/bpc.h>
#include <nn/bpc/bpc_PowerButton.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/system/hid_SleepButton.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "hid_SleepButtonDriver.horizon.bcp.h"

namespace nn { namespace hid { namespace detail {

SleepButtonDriver::SleepButtonDriver() NN_NOEXCEPT
    : m_ActivationCount()
    , m_SamplingNumber(0)
    , m_SystemEvent()
    , m_State(::nn::bpc::SleepButtonState_Released)
{
    // 何もしない
}

SleepButtonDriver::~SleepButtonDriver() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result SleepButtonDriver::Activate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultSleepButtonDriverActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        auto needsRollback = true;

        ::nn::bpc::InitializePowerButton();
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                ::nn::bpc::FinalizePowerButton();
            }
        };

        NN_RESULT_DO(::nn::bpc::GetSleepButtonState(&m_State));
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_State = ::nn::bpc::SleepButtonState_Released;
            }
        };

        NN_RESULT_DO(::nn::bpc::GetPowerEvent(
                         &m_SystemEvent,
                         ::nn::bpc::EventTarget_SleepButtonStateChanged));

        needsRollback = false;
    }

    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result SleepButtonDriver::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultSleepButtonDriverDeactivationLowerLimitOver());

    --m_ActivationCount;

    if (m_ActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施

        ::nn::os::DestroySystemEvent(&m_SystemEvent);

        m_State = ::nn::bpc::SleepButtonState_Released;

        ::nn::bpc::FinalizePowerButton();
    }

    NN_RESULT_SUCCESS;
}

void SleepButtonDriver::GetState(::nn::hid::system::SleepButtonState* pOutValue
                                 ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    // スリープボタンの入力状態を初期化
    *pOutValue = ::nn::hid::system::SleepButtonState();

    pOutValue->samplingNumber = m_SamplingNumber++;

    if (::nn::os::TryWaitSystemEvent(&m_SystemEvent))
    {
        ::nn::os::ClearSystemEvent(&m_SystemEvent);

        ::nn::bpc::GetSleepButtonState(&m_State);
    }

    pOutValue->buttons.Set<::nn::hid::system::SleepButton::Active>(
        m_State == ::nn::bpc::SleepButtonState_Pushed);
}

}}} // namespace nn::hid::detail
