﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/hid/hid_Xpad.h>
#include <nn/hid/hid_JoyXpad.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/hid_NpadSixAxisSensor.h>
#include <nn/util/util_BitPack.h>

#include "hid_XpadId.h"

#include "hid_SixAxisSensor.h" // 公開ヘッダ更新後、削除予定

namespace nn { namespace hid { namespace detail {

const int SixAxisSensorIdxCountMax = 3;  //!< 6軸センサー搭載位置の種類数

//!< BasicXpad の6軸センサー識別子番号 (JoyXpad も同じ番号を共用)
enum SixAxisSensorIdx
{
    SixAxisSensorIdx_Left   = 0,
    SixAxisSensorIdx_Right  = 1,
    SixAxisSensorIdx_Center = 2,
};

//!< Xpad の種類に応じた SixAxisSensorIdxT および TypeValue の定義
template<typename T>
struct XpadType
{
    typedef int SixAxisSensorIdxT;
    static const uint8_t TypeValue = 0;
};

//!< BasicXpad に対応する SixAxisSensorIdxT および TypeValue の定義
template<>
struct XpadType<BasicXpadId>
{
    typedef SixAxisSensorIdx SixAxisSensorIdxT;
    static const uint8_t TypeValue = 1;
};

//!< JoyXpad に対応する SixAxisSensorIdxT および TypeValue の定義
template<>
struct XpadType<JoyXpadId>
{
    typedef SixAxisSensorIdx SixAxisSensorIdxT;  // 6軸センサー識別子番号は BasicXpad と共通のものを利用
    static const uint8_t TypeValue = 2;
};

//<! SixAxisSensorHandle のビットフィールド定義です。
struct SixAxisSensorHandleField final
{
    //!< Xpad の種類
    typedef ::nn::util::BitPack32::Field<0, 8, uint8_t> XpadTypeValue;

    //!< プレイヤー番号
    typedef ::nn::util::BitPack32::Field<8, 8, uint8_t> PlayerNumber;

    //!< 6軸センサー番号
    typedef ::nn::util::BitPack32::Field<16, 8, uint8_t> SixAxisSensorIdx;
};

const int NpadSixAxisSensorTypeValueOffset = 3;

//!< Xpad の種類、プレイヤー番号、6軸センサー識別子番号から SixAxisSensorHandle を作成します。
SixAxisSensorHandle MakeSixAxisSensorHandle(uint8_t xpadTypeId, int playerNumber, int sixAxisSensorIdx) NN_NOEXCEPT;

//!< SixAxisSensorHandle から Xpad の種類 (に対応する整数値) を取得します。
uint8_t GetSixAxisSensorHandleXpadTypeValue(const SixAxisSensorHandle& handle) NN_NOEXCEPT;

//!< SixAxisSensorHandle から NpadStyleTag のインデックスを取得します。
uint8_t GetSixAxisSensorHandleNpadStyleIndex(const SixAxisSensorHandle& handle) NN_NOEXCEPT;

//!< SixAxisSensorHandle からプレイヤー番号を取得します。
int GetSixAxisSensorHandlePlayerNumber(const SixAxisSensorHandle& handle) NN_NOEXCEPT;

//!< SixAxisSensorHandle から NpadId を取得します。
NpadIdType GetSixAxisSensorHandleNpadIdType(const SixAxisSensorHandle& handle) NN_NOEXCEPT;

//!< SixAxisSensorHandle から6軸センサー識別子番号を取得します。
int GetSixAxisSensorHandleSixAxisSensorIdx(const SixAxisSensorHandle& handle) NN_NOEXCEPT;

//!< SixAxisSensorHandle を作成します。
template<typename T>
inline SixAxisSensorHandle MakeSixAxisSensorHandle(const T& xpadId, SixAxisSensorIdx idx) NN_NOEXCEPT
{
    uint8_t xpadTypeValue = XpadType<T>::TypeValue;
    int playerNumber = GetXpadIdPlayerNumber(xpadId);
    typename XpadType<T>::SixAxisSensorIdxT sixAxisSensorIdx = idx;

    return MakeSixAxisSensorHandle(xpadTypeValue, playerNumber, sixAxisSensorIdx);
}

//!< SixAxisSensorHandle に対応する Xpad の種類を判別します。
template<typename T>
inline bool IsSixAxisSensorHandleXpadTypeEqualTo(const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    return GetSixAxisSensorHandleXpadTypeValue(handle) == XpadType<T>::TypeValue;
}

//!< SixAxisSensorHandle に対応する Npad の種類を判別します。
inline bool IsSixAxisSensorHandleNpadTypeEqualTo(const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    NN_UNUSED(handle);
    // TODO : Npadの種類を判定します
    return true;
}

//!< SixAxisSensorHandle から XpadId を取得します。
template<typename T>
inline T GetSixAxisSensorHandleXpadId(const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    // NN_SDK_ASSERT_EQUAL(GetSixAxisSensorHandleXpadTypeValue(handle), XpadType<T>::TypeValue);
    int playerNumber = GetSixAxisSensorHandlePlayerNumber(handle);
    return MakeXpadId<T>(playerNumber);
}

::nn::Result GetSixAxisSensorHandles(SixAxisSensorHandle* pOutLeftHandle,
                                     SixAxisSensorHandle* pOutRightHandle,
                                     JoyXpadId xpadId) NN_NOEXCEPT;

::nn::Result GetSixAxisSensorHandle(SixAxisSensorHandle* pOutHandle,
                                    BasicXpadId xpadId) NN_NOEXCEPT;

::nn::Result GetSixAxisSensorHandle(SixAxisSensorHandle* pOutHandle,
                                    int* pOutCount,
                                    int size,
                                    NpadIdType npadId,
                                    NpadStyleSet style) NN_NOEXCEPT;

::nn::Result StartSixAxisSensor(const SixAxisSensorHandle& handle) NN_NOEXCEPT;

::nn::Result StopSixAxisSensor(const SixAxisSensorHandle& handle) NN_NOEXCEPT;

::nn::Result IsSixAxisSensorFusionEnabled(bool* pOutIsEnabled,
                                          const nn::hid::SixAxisSensorHandle& handle) NN_NOEXCEPT;

::nn::Result EnableSixAxisSensorFusion(const nn::hid::SixAxisSensorHandle& handle, bool enable) NN_NOEXCEPT;

::nn::Result SetSixAxisSensorFusionParameters(const nn::hid::SixAxisSensorHandle& handle,
                                              float revisePower,
                                              float reviseRange) NN_NOEXCEPT;

::nn::Result GetSixAxisSensorFusionParameters(float* pOutRevisePower,
                                              float* pOutReviseRange,
                                              const nn::hid::SixAxisSensorHandle& handle) NN_NOEXCEPT;

::nn::Result ResetSixAxisSensorFusionParameters(const nn::hid::SixAxisSensorHandle& handle) NN_NOEXCEPT;

::nn::Result SetAccelerometerParameters(nn::hid::SixAxisSensorHandle handle,
                                        float playRadius,
                                        float sensitivity) NN_NOEXCEPT;

::nn::Result GetAccelerometerParameters(float* pOutPlayRadius,
                                        float* pOutSensitivity,
                                        nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT;

::nn::Result ResetAccelerometerParameters(nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT;

::nn::Result SetAccelerometerPlayMode(nn::hid::SixAxisSensorHandle handle,
                                      nn::hid::AccelerometerPlayMode mode) NN_NOEXCEPT;

::nn::Result GetAccelerometerPlayMode(nn::hid::AccelerometerPlayMode* pOutMode,
                                      nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT;

::nn::Result ResetAccelerometerPlayMode(nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT;

::nn::Result SetGyroscopeZeroDriftMode(nn::hid::SixAxisSensorHandle handle,
                                       nn::hid::GyroscopeZeroDriftMode mode) NN_NOEXCEPT;

::nn::Result GetGyroscopeZeroDriftMode(nn::hid::GyroscopeZeroDriftMode* pOutMode,
                                       nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT;

::nn::Result ResetGyroscopeZeroDriftMode(nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT;

::nn::Result GetSixAxisSensorState(SixAxisSensorState* outValue,
                                   const SixAxisSensorHandle& handle) NN_NOEXCEPT;

::nn::Result GetSixAxisSensorStates(int* outGotCount,
                                    SixAxisSensorState* outStates,
                                    int count,
                                    const SixAxisSensorHandle& handle) NN_NOEXCEPT;

::nn::Result IsSixAxisSensorAtRest(bool* pOutIsAtRest,
                                   const nn::hid::SixAxisSensorHandle& handle) NN_NOEXCEPT;

}}} // namespace nn::hid::detail
