﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits> // for numeric_limits

#include <nn/hid/system/hid_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_MatrixApi.h>
#include <nn/util/util_VectorApi.h>

#include "hid_SixAxisSensorCalibrator.h"
#include "hid_SixAxisSensorMathUtility.h"

namespace {

void SetMax(::nn::util::Float3* pMaxCount,
            const ::nn::util::Float3& count)
{
    pMaxCount->x = (pMaxCount->x < count.x) ? count.x : pMaxCount->x;
    pMaxCount->y = (pMaxCount->y < count.y) ? count.y : pMaxCount->y;
    pMaxCount->z = (pMaxCount->z < count.z) ? count.z : pMaxCount->z;
}

void SetMin(::nn::util::Float3* pMinCount,
            const ::nn::util::Float3& count)
{
    pMinCount->x = (pMinCount->x > count.x) ? count.x : pMinCount->x;
    pMinCount->y = (pMinCount->y > count.y) ? count.y : pMinCount->y;
    pMinCount->z = (pMinCount->z > count.z) ? count.z : pMinCount->z;
}

bool IsInRange(const float& state1,
               const float& state2,
               const int32_t& range)
{
    float work = state1 - state2;

    return (work > -range &&
            work < range);
}

bool IsInRange(const ::nn::util::Float3& state1,
               const ::nn::util::Float3& state2,
               const int32_t& range)
{
    return (IsInRange(state1.x, state2.x, range) &&
            IsInRange(state1.y, state2.y, range) &&
            IsInRange(state1.z, state2.z, range));
}

nn::Result GetRotationMatrixForAccelerometerCalibration(::nn::util::Matrix4x4fType* pOutMatrix,
                                                        const nn::util::Float3& calibrationAccelerometer,
                                                        const nn::util::Float3& modelAccelerometer) NN_NOEXCEPT
{
    nn::util::Vector3fType typicalAcc;   // Typical 値から取得した加速度ベクトル
    nn::util::Vector3fType modelAcc;     // モデル値から取得した加速度ベクトル

    nn::util::VectorLoad(&typicalAcc, calibrationAccelerometer);
    nn::util::VectorLoad(&modelAcc, modelAccelerometer);

    float normProduct = nn::util::VectorLength(typicalAcc) * nn::util::VectorLength(modelAcc);

    if (normProduct == 0.0f)
    {
        NN_RESULT_THROW(::nn::hid::system::ResultSixAxisSensorWriteFailure());
    }

    nn::util::Vector3fType rotateAxisVector; // 回転軸方向ベクトル
    nn::util::Vector3fType tmpVector;        // 符号判定処理用ベクトル

    nn::util::VectorCross(&rotateAxisVector,
                          typicalAcc,
                          modelAcc);

    nn::util::VectorCross(&tmpVector,
                          rotateAxisVector,
                          typicalAcc);

    float sign = 1.0f;
    if (nn::util::VectorDot(tmpVector, modelAcc) > 0.0f)
    {
        sign = - 1.0f;
    }

    // Typical と モデル値のなす角を取得
    float cos = nn::util::VectorDot(typicalAcc, modelAcc) / normProduct;
    float sin = sign * nn::hid::detail::Sqrt(1 - cos * cos);

    // Rodrigues' rotation formula
    // return I + sin * R + (1-cos) * R * R
    nn::util::VectorNormalize(&rotateAxisVector, rotateAxisVector);

    nn::util::Matrix4x4fType matrixR;
    nn::util::MatrixSet(
        &matrixR,
        0.f, -nn::util::VectorGetZ(rotateAxisVector), nn::util::VectorGetY(rotateAxisVector), 0.f,
        nn::util::VectorGetZ(rotateAxisVector), 0.f, -nn::util::VectorGetX(rotateAxisVector), 0.f,
        -nn::util::VectorGetY(rotateAxisVector), nn::util::VectorGetX(rotateAxisVector), 0.f, 0.f,
        0.f, 0.f, 0.f, 0.0f);

    nn::util::Matrix4x4fType buffer;
    nn::util::MatrixIdentity(&buffer); // buffer = I

    nn::util::Matrix4x4fType tmp;
    nn::util::MatrixMultiply(&tmp, matrixR, sin);
    nn::util::MatrixAdd(&buffer, buffer, tmp); // buffer = I + sin * R

    nn::util::MatrixMultiply(&tmp, matrixR, matrixR);
    nn::util::MatrixMultiply(&tmp, tmp, 1.0f - cos);
    nn::util::MatrixAdd(&buffer, buffer, tmp); // buffer = I + sin * R + (1-cos) * R * R

    *pOutMatrix = buffer;

    NN_RESULT_SUCCESS;
}

void MultiplyRotateMatrix(nn::util::Float3* pOutVector,
                          const nn::util::Matrix4x4fType& rotateMatrix,
                          const nn::util::Float3& vector) NN_NOEXCEPT
{
    nn::util::Vector3fType vec3;
    nn::util::VectorLoad(&vec3, vector);

    // 4x4 の行列演算に当てはめるためベクトルを行列に変換
    nn::util::Matrix4x4fType buffer;
    nn::util::MatrixSet(&buffer,
        nn::util::VectorGetX(vec3), 0.f, 0.f, 0.f,
        nn::util::VectorGetY(vec3), 0.f, 0.f, 0.f,
        nn::util::VectorGetZ(vec3), 0.f, 0.f, 0.f,
        0.f, 0.f, 0.f, 0.f);

    nn::util::MatrixMultiply(&buffer, rotateMatrix, buffer);

    // 計算結果取得
    nn::util::Matrix4x3fType tmp4x3;
    nn::util::MatrixConvert(&tmp4x3, buffer); // 行優先 4x3 行列に変換

    // X
    nn::util::MatrixGetAxisX(&vec3, tmp4x3);
    pOutVector->x = nn::util::VectorGetX(vec3);

    // Y
    nn::util::MatrixGetAxisY(&vec3, tmp4x3);
    pOutVector->y = nn::util::VectorGetX(vec3);

    // Z
    nn::util::MatrixGetAxisZ(&vec3, tmp4x3);
    pOutVector->z = nn::util::VectorGetX(vec3);
}

void ReviseCalibrationValue(::nn::xcd::SensorCalibrationValue* pOutCalibrationValue,
                            const ::nn::util::Float3& accelerometer,
                            const ::nn::util::Float3& angularVelocity) NN_NOEXCEPT
{
    // TODO: HW 仕様に基づく測定値の妥当性チェック

    // 加速度 Z 軸は重力加速度方向なので除く
    pOutCalibrationValue->accelerometerOrigin.x = static_cast<int16_t>(accelerometer.x);
    pOutCalibrationValue->accelerometerOrigin.y = static_cast<int16_t>(accelerometer.y);

    pOutCalibrationValue->gyroscopeOrigin.x = static_cast<int16_t>(angularVelocity.x);
    pOutCalibrationValue->gyroscopeOrigin.y = static_cast<int16_t>(angularVelocity.y);
    pOutCalibrationValue->gyroscopeOrigin.z = static_cast<int16_t>(angularVelocity.z);
}

} // namespace

namespace nn { namespace hid { namespace detail {

SixAxisSensorCalibrator::SixAxisSensorCalibrator() NN_NOEXCEPT
{
    Reset();
}

void SixAxisSensorCalibrator::Update(const ::nn::xcd::SixAxisSensorState& state) NN_NOEXCEPT
{
    if(state.sampleNumber <= m_LastSixAxisSensorState.sampleNumber)
    {
        // 過去サンプルは無視
        return;
    }

    // 各軸におけるセンサー値の総和を計算
    m_AccelerometerCountSum[0] += state.accelerometer.x;
    m_AccelerometerCountSum[1] += state.accelerometer.y;
    m_AccelerometerCountSum[2] += state.accelerometer.z;
    m_AngularVelocityCountSum[0] += state.gyroscope.x;
    m_AngularVelocityCountSum[1] += state.gyroscope.y;
    m_AngularVelocityCountSum[2] += state.gyroscope.z;

    // 有効サンプル数を更新
    m_ValidSampleCount++;

    // センサー値の変動範囲を更新
    SetMax(&m_AngularVelocityCountMax, state.gyroscope);
    SetMin(&m_AngularVelocityCountMin, state.gyroscope);

    // 内部状態を更新
    m_LastSixAxisSensorState = state;


    // 種々の判定 ------------------

    // センサー値が許容範囲内か
    if (m_Config.isAtRest)
    {
        m_Config.isAtRest = (IsInRange(m_AngularVelocityCountMax,
                                       m_AngularVelocityCountMin,
                                       m_Config.angularVelocityCountRange));
    }

    // 水平に置いたときの加速度カウントの理論値と直近の加速度カウントのズレが許容範囲内か
    if (m_Config.isHorizontal)
    {
        nn::util::Float3 const config = {{{
            static_cast<float>(m_Config.horizontalStateAccelerometerCount.x),
            static_cast<float>(m_Config.horizontalStateAccelerometerCount.y),
            static_cast<float>(m_Config.horizontalStateAccelerometerCount.z)
        }}};
        m_Config.isHorizontal = IsInRange(config,
                                          m_LastSixAxisSensorState.accelerometer,
                                          m_Config.horizontalStateAccelerometerCountRange);
    }
}

bool SixAxisSensorCalibrator::IsCalibrated() const NN_NOEXCEPT
{
    return (m_ValidSampleCount >= m_Config.validSampleCountMax);
}

void SixAxisSensorCalibrator::GetMeanValues(::nn::util::Float3* pOutAccelerometerCount,
                                            ::nn::util::Float3* pOutAngularVelocityCount) const NN_NOEXCEPT
{
    if(!IsCalibrated())
    {
        return;
    }

    pOutAccelerometerCount->x = static_cast<float>(m_AccelerometerCountSum[0]) / m_ValidSampleCount;
    pOutAccelerometerCount->y = static_cast<float>(m_AccelerometerCountSum[1]) / m_ValidSampleCount;
    pOutAccelerometerCount->z = static_cast<float>(m_AccelerometerCountSum[2]) / m_ValidSampleCount;

    pOutAngularVelocityCount->x = static_cast<float>(m_AngularVelocityCountSum[0]) / m_ValidSampleCount;
    pOutAngularVelocityCount->y = static_cast<float>(m_AngularVelocityCountSum[1]) / m_ValidSampleCount;
    pOutAngularVelocityCount->z = static_cast<float>(m_AngularVelocityCountSum[2]) / m_ValidSampleCount;
}

void SixAxisSensorCalibrator::SetCalibrationParameters(const ::nn::xcd::SensorState& horizontalStateAccelerometerCount,
                                                       const int32_t& horizontalStateAccelerometerCountRange,
                                                       const int32_t& angularVelocityCountRange,
                                                       const int& validSampleCountMax,
                                                       const ::nn::xcd::SensorCalibrationValue& calibrationValue,
                                                       const ::nn::util::Float3& typicalAccelerometer) NN_NOEXCEPT
{
    m_Config.horizontalStateAccelerometerCount = horizontalStateAccelerometerCount;
    m_Config.horizontalStateAccelerometerCountRange = horizontalStateAccelerometerCountRange;
    m_Config.angularVelocityCountRange = angularVelocityCountRange;
    m_Config.validSampleCountMax = validSampleCountMax;
    m_Config.calibrationValue = calibrationValue;
    m_Config.typicalAccelerometer = typicalAccelerometer;
}

nn::Result SixAxisSensorCalibrator::GetUserCalibrationValue(::nn::xcd::SensorCalibrationValue* pOutCalibrationValue) const NN_NOEXCEPT
{
    // 測定値(⇔標本平均値)を取得する
    ::nn::util::Float3 meanAccelerometer;
    ::nn::util::Float3 meanAngularVelocity;

    GetMeanValues(&meanAccelerometer,
                  &meanAngularVelocity);

    // モデル値を Typical 値に変換する回転行列を計算する
    ::nn::util::Matrix4x4fType rotationMatrix;
    ::nn::util::Float3 modelAccelerometer;

    modelAccelerometer.x = static_cast<float>(m_Config.horizontalStateAccelerometerCount.x);
    modelAccelerometer.y = static_cast<float>(m_Config.horizontalStateAccelerometerCount.y);
    modelAccelerometer.z = static_cast<float>(m_Config.horizontalStateAccelerometerCount.z);

    NN_RESULT_DO(GetRotationMatrixForAccelerometerCalibration(&rotationMatrix,
                                                              m_Config.typicalAccelerometer,
                                                              modelAccelerometer));

    // 測定値を回転行列で変換する
    ::nn::util::Float3 modifiedMeanAccelerometer;
    ::nn::util::Float3 modifiedMeanAngularVelocity;

    MultiplyRotateMatrix(&modifiedMeanAccelerometer,
                         rotationMatrix,
                         meanAccelerometer);

    MultiplyRotateMatrix(&modifiedMeanAngularVelocity,
                         rotationMatrix,
                         meanAngularVelocity);

    // 変換後の標本平均値を元にキャリブレーション値を修正
    *pOutCalibrationValue = m_Config.calibrationValue;
    ReviseCalibrationValue(pOutCalibrationValue,
                           modifiedMeanAccelerometer,
                           modifiedMeanAngularVelocity);

    NN_RESULT_SUCCESS;
}

void SixAxisSensorCalibrator::Reset() NN_NOEXCEPT
{
    m_LastSixAxisSensorState = ::nn::xcd::SixAxisSensorState();
    m_LastSixAxisSensorState.sampleNumber = ::std::numeric_limits<int64_t>::min();

    m_AngularVelocityCountMax.x = ::std::numeric_limits<int16_t>::min();
    m_AngularVelocityCountMax.y = ::std::numeric_limits<int16_t>::min();
    m_AngularVelocityCountMax.z = ::std::numeric_limits<int16_t>::min();

    m_AngularVelocityCountMin.x = ::std::numeric_limits<int16_t>::max();
    m_AngularVelocityCountMin.y = ::std::numeric_limits<int16_t>::max();
    m_AngularVelocityCountMin.z = ::std::numeric_limits<int16_t>::max();

    for(int i = 0; i < 3; i++)
    {
        m_AccelerometerCountSum[i] = 0;
        m_AngularVelocityCountSum[i] = 0;
    }
    m_ValidSampleCount = 0;

    m_Config.horizontalStateAccelerometerCount = ::nn::xcd::SensorState();
    m_Config.horizontalStateAccelerometerCountRange = ::std::numeric_limits<int32_t>::max();
    m_Config.angularVelocityCountRange = ::std::numeric_limits<int32_t>::max();
    m_Config.validSampleCountMax = ::std::numeric_limits<int32_t>::max();
    m_Config.calibrationValue = ::nn::xcd::SensorCalibrationValue();
    m_Config.typicalAccelerometer = ::nn::util::Float3();
    m_Config.isAtRest = true;
    m_Config.isHorizontal = true;
}

bool SixAxisSensorCalibrator::IsAtRest() const NN_NOEXCEPT
{
    return m_Config.isAtRest;
}

bool SixAxisSensorCalibrator::IsHorizontal() const NN_NOEXCEPT
{
    return m_Config.isHorizontal;
}

}}} // namespace ::nn::hid::detail
