﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       6 軸センサーに関する API の宣言
 */

#pragma once

namespace nn { namespace hid {

// 公開ヘッダに移行後、削除予定
// 設定 API の内部実装を追加するための暫定実装のため

/**
 * @brief       加速度センサーの遊びの挙動モードの定義です。
 */
enum AccelerometerPlayMode
{
    AccelerometerPlayMode_Loose,     //!< Loose モードです。
    AccelerometerPlayMode_Tight,     //!< Tight モードです。
};

/**
 * @brief       加速度センサーの補正パラメータを設定します。
 *
 * @details     指定の handle と対応する Xpad の加速度センサーについて、遊びの半径と検出感度を調整します。
 *              遊びの半径を大きくすると、コントローラーが静止している際の値のブレを抑えることができます。
 *              検出感度の値を大きくすると、コントローラーを動かしている際の値のブレを抑えることができます。
 *              遊びの半径外は検出感度による重みづけのみがなされます。
 *              遊びの半径内は検出感度による重みづけに加え、遊びの半径内部に近づくに従って曲線的に動きが小さくなる補正が行われます。
 *              遊びの半径を 0.0f、検出感度を 1.0f とすれば、デバイスから取得した値が変更せずに使用されます。
 *              遊びの半径、検出感度のデフォルトの値はそれぞれ 0.0f, 1.0f となっています。
 *
 * @param[in]   handle                      6軸センサーのハンドル
 * @param[in]   playRadius                  遊びの半径
 * @param[in]   sensitivity                 検出感度を表します。0.0f を指定すると全く反応しなくなり、1.0f ではデバイスから取得した値の通りに反応します。
 *
 * @pre
 *              - playRadius >= 0.0f
 *              - sensitivity >= 0.0f && sensitivity <= 1.0f
 */
void SetAccelerometerParameters(const SixAxisSensorHandle& handle,
                                float playRadius,
                                float sensitivity) NN_NOEXCEPT;

/**
 * @brief       加速度センサーの補正パラメータを取得します。
 *
 * @details     指定の handle と対応する Xpad の加速度センサーについて、遊びの半径と検出感度の設定値を取得します。
 *
 * @param[out]   pOutPlayRadius              遊びの半径
 * @param[out]   pOutSensitivity             検出感度を表します。0.0f を指定するとまったく反応しなくなり、1.0f ではデバイスから取得した値どおりに反応します。
 * @param[in]    handle                      6軸センサーのハンドル
 *
 */
void GetAccelerometerParameters(float* pOutPlayRadius,
                                float* pOutSensitivity,
                                const SixAxisSensorHandle& handle) NN_NOEXCEPT;

/**
 * @brief       加速度センサーの補正パラメータを初期化します。
 *
 * @details     指定の handle と対応する Xpad の加速度センサーについて、遊びの半径と検出感度の設定値を初期化します。
 *              遊びの半径、検出感度のデフォルトの値はそれぞれ 0.0f, 1.0f となります。
 *
 * @param[in]    handle                      6軸センサーのハンドル
 */
void ResetAccelerometerParameters(const SixAxisSensorHandle& handle) NN_NOEXCEPT;

/**
 * @brief       加速度センサーについて遊びの挙動方法を設定します。
 *
 * @details     指定の handle と対応する Xpad の加速度センサーについて遊びの挙動方法を設定します。
 *              デフォルト値は AccelerometerPlayMode_Tight です。
 *
 *              AccelerometerPlayMode_Tight では、遊び半径の境界を超えるかどうかによって数値が変化するかどうかが明確に決まります。
 *              遊びの範囲内であれば数値は変化しません。
 *
 *              AccelerometerPlayMode_Loose では、遊びの範囲内であっても遊び半径の境界に近づくに従って数値は徐々に動き始め、滑らかな数値変化が行われます。
 *              あまり遊び半径を大きくすると、あたかも検出感度を弱くしたかのような挙動になります。
 *
 * @param[in]   handle                      6軸センサーのハンドル
 * @param[in]   mode                        遊びの挙動方法です
 *
 */
void SetAccelerometerPlayMode(const SixAxisSensorHandle& handle,
                              const AccelerometerPlayMode& mode) NN_NOEXCEPT;

/**
 * @brief       加速度センサーについて遊びの挙動方法を取得します。
 *
 * @details     指定の handle と対応する Xpad の加速度センサーについて遊びの挙動方法を取得します。
 *
 * @param[out]  pOutMode                    遊びの挙動方法です
 * @param[in]   handle                      6軸センサーのハンドル
 *
 */
void GetAccelerometerPlayMode(AccelerometerPlayMode* pOutMode,
                              const SixAxisSensorHandle& handle) NN_NOEXCEPT;

/**
 * @brief       加速度センサーについて遊びの挙動方法を初期化します。
 *
 * @details     指定の handle と対応する Xpad の加速度センサーについて遊びの挙動方法を初期化します。
 *
 * @param[in]   handle                      6軸センサーのハンドル
 *
 */
void ResetAccelerometerPlayMode(const SixAxisSensorHandle& handle) NN_NOEXCEPT;

}} // namespace nn::hid
