﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_ConsoleSixAxisSensor.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nn/hid/debug/hid_CaptureButton.h>
#include <nn/hid/debug/hid_DebugPad.h>
#include <nn/hid/debug/hid_HomeButton.h>
#include <nn/hid/debug/hid_Keyboard.h>
#include <nn/hid/debug/hid_Mouse.h>
#include <nn/hid/debug/hid_TouchScreen.h>
#include <nn/hid/debug/hid_Xpad.h>
#include <nn/hid/system/hid_Keyboard.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_Thread.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/settings_DebugPad.h>
#include <nn/settings/settings_Xpad.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/hid/hid_JoyXpad.h>

#include "hid_CaptureButtonManager.h"
#include "hid_ConsoleSixAxisSensorLifo.h"
#include "hid_DebugPadLifo.h"
#include "hid_DebugPadMappingManager.h"
#include "hid_HomeButtonManager.h"
#include "hid_IConsoleSixAxisSensorDriver.h"
#include "hid_IDebugPadDriver.h"
#include "hid_IKeyboardDriver.h"
#include "hid_IMouseDriver.h"
#include "hid_InputDetectorId.h"
#include "hid_InputDetectorManager.h"
#include "hid_IrSensorXcdDriver.h"
#include "hid_ITouchScreenDriver.h"
#include "hid_IXpadDriver.h"
#include "hid_KeyboardLifo.h"
#include "hid_LockableMutexType.h"
#include "hid_MouseLifo.h"
#include "hid_ResourceManager-os.win.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_SleepButtonManager.h"
#include "hid_StaticObject.h"
#include "hid_TouchScreenLifo.h"
#include "hid_UniquePadId.h"
#include "hid_XcdFirmwareUpdater.h"
#include "hid_XpadId.h"
#include "hid_XpadLifo.h"
#include "hid_XpadMappingManager.h"

namespace nn { namespace hid { namespace detail {

const nn::TimeSpan SharedTask::PadCommonTimerInterval = nn::TimeSpan::FromMilliSeconds(5);

SharedTask::SharedTask() NN_NOEXCEPT
{
    // 何もしない
}

SharedTask::~SharedTask() NN_NOEXCEPT
{
    // 何もしない
}

void SharedTask::SetNpadResourceManager(NpadResourceManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    m_pNpadResourceManager = pManager;
}

void SharedTask::SetUniquePadResourceManager(UniquePadResourceManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    pManager->SetSystemEvents(m_BluetoothFirmwareUpdateSystemEvent.GetBase(),
                              m_McuFirmwareUpdateSystemEvent.GetBase(),
                              m_FirmwareUpdateNewSampleSystemEvent.GetBase(),
                              m_FirmwareUpdateResetTimeoutEvent.GetBase());
    m_pUniquePadResourceManager = pManager;
}

void SharedTask::SetKeyboardManager(KeyboardManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    pManager->SetTimer(&m_KeyboardTimerEvent);
    m_pKeyboardManager = pManager;
}

void SharedTask::SetMouseManager(MouseManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    pManager->SetTimer(&m_MouseTimerEvent);
    m_pMouseManager = pManager;
}

void SharedTask::SetTouchScreenManager(TouchScreenManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    pManager->SetTimerEvent(m_TouchScreenTimerEvent.GetBase());
    m_pTouchScreenManager = pManager;
}

void SharedTask::SetHomeButtonManager(HomeButtonManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    pManager->SetTimer(&m_HomeButtonTimerEvent);
    m_pHomeButtonManager = pManager;
}

void SharedTask::SetSleepButtonManager(SleepButtonManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    pManager->SetTimer(&m_SleepButtonTimerEvent);
    m_pSleepButtonManager = pManager;
}

void SharedTask::SetCaptureButtonManager(CaptureButtonManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    pManager->SetTimer(&m_CaptureButtonTimerEvent);
    m_pCaptureButtonManager = pManager;
}

void SharedTask::SetConsoleSixAxisSensorManager(ConsoleSixAxisSensorManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    m_pConsoleSixAxisSensorManager = pManager;
}

void SharedTask::SetPlayReportManager(PlayReportManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    pManager->SetTimer(&m_PlayReportTimerEvent);
    m_pPlayReportManager = pManager;
}

void SharedTask::SetRegisteredDeviceManager(RegisteredDeviceManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    pManager->BindXcdRegistrationEvent(m_PairingCompleteEvent.GetBase());
    pManager->BindXcdConnectionTriggerTimeoutEvent(m_ConnectionTriggerTimeoutEvent.GetBase());
    m_pRegisteredDeviceManager = pManager;
}

void SharedTask::SetAudioControlManager(AudioControlManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    pManager->SetTimer(&m_AudioControlTimerEvent);
    m_pAudioControlManager = pManager;
}

::nn::Result SharedTask::ActivateKeyboard() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pKeyboardManager);

    auto needsRollback = true;

    NN_RESULT_DO(this->ActivateThread());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    ::std::lock_guard<decltype(m_KeyboardMutex)> locker(m_KeyboardMutex);
    NN_RESULT_DO(m_pKeyboardManager->Activate());

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivateKeyboard() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pKeyboardManager);

    {
        ::std::lock_guard<decltype(m_KeyboardMutex)> locker(m_KeyboardMutex);
        NN_RESULT_DO(m_pKeyboardManager->Deactivate());
    }

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::EnsureKeyboardAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pKeyboardManager);
    ::std::lock_guard<decltype(m_KeyboardMutex)> locker(m_KeyboardMutex);
    NN_RESULT_DO(m_pKeyboardManager->EnsureAppletResource(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::SetKeyboardAutoPilotState(
    const ::nn::hid::debug::KeyboardAutoPilotState& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pKeyboardManager);
    ::std::lock_guard<decltype(m_KeyboardMutex)> locker(m_KeyboardMutex);
    NN_RESULT_THROW(m_pKeyboardManager->SetAutoPilotState(value));
}

::nn::Result SharedTask::UnsetKeyboardAutoPilotState() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pKeyboardManager);
    ::std::lock_guard<decltype(m_KeyboardMutex)> locker(m_KeyboardMutex);
    NN_RESULT_THROW(m_pKeyboardManager->UnsetAutoPilotState());
}

::nn::Result SharedTask::SendKeyboardLockKeyEvent(
    ::nn::hid::system::KeyboardLockKeyEventSet value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pKeyboardManager);
    ::std::lock_guard<decltype(m_KeyboardMutex)> locker(m_KeyboardMutex);
    NN_RESULT_DO(m_pKeyboardManager->SendLockKeyEvent(value));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::ActivateMouse() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pMouseManager);

    auto needsRollback = true;

    NN_RESULT_DO(this->ActivateThread());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    ::std::lock_guard<decltype(m_MouseMutex)> locker(m_MouseMutex);
    NN_RESULT_DO(m_pMouseManager->Activate());

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivateMouse() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pMouseManager);

    {
        ::std::lock_guard<decltype(m_MouseMutex)> locker(m_MouseMutex);
        NN_RESULT_DO(m_pMouseManager->Deactivate());
    }

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::EnsureMouseAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pMouseManager);
    ::std::lock_guard<decltype(m_MouseMutex)> locker(m_MouseMutex);
    NN_RESULT_DO(m_pMouseManager->EnsureAppletResource(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::SetMouseAutoPilotState(
    const ::nn::hid::debug::MouseAutoPilotState& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pMouseManager);
    ::std::lock_guard<decltype(m_MouseMutex)> locker(m_MouseMutex);
    NN_RESULT_THROW(m_pMouseManager->SetAutoPilotState(value));
}

::nn::Result SharedTask::UnsetMouseAutoPilotState() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pMouseManager);
    ::std::lock_guard<decltype(m_MouseMutex)> locker(m_MouseMutex);
    NN_RESULT_THROW(m_pMouseManager->UnsetAutoPilotState());
}

::nn::Result SharedTask::ActivateGesture() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTouchScreenManager);

    auto needsRollback = true;

    NN_RESULT_DO(this->ActivateThread());

    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    {
        ::std::lock_guard<decltype(m_TouchScreenMutex)
                          > locker(m_TouchScreenMutex);
        NN_RESULT_DO(m_pTouchScreenManager->ActivateForGesture());
    }

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivateGesture() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTouchScreenManager);

    {
        ::std::lock_guard<decltype(m_TouchScreenMutex)
                          > locker(m_TouchScreenMutex);
        NN_RESULT_DO(m_pTouchScreenManager->DeactivateForGesture());
    }

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::EnsureGestureAppletResource(
    ::nn::applet::AppletResourceUserId aruid, int32_t version) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTouchScreenManager);
    ::std::lock_guard<decltype(m_TouchScreenMutex)> locker(m_TouchScreenMutex);
    NN_RESULT_DO(
        m_pTouchScreenManager->EnsureAppletResourceForGesture(aruid, version));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::ActivateTouchScreen() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTouchScreenManager);

    auto needsRollback = true;

    NN_RESULT_DO(this->ActivateThread());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    ::std::lock_guard<decltype(m_TouchScreenMutex)> locker(m_TouchScreenMutex);
    NN_RESULT_DO(m_pTouchScreenManager->ActivateForTouchScreen());

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivateTouchScreen() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTouchScreenManager);

    {
        ::std::lock_guard<decltype(m_TouchScreenMutex)
                          > locker(m_TouchScreenMutex);
        NN_RESULT_DO(m_pTouchScreenManager->DeactivateForTouchScreen());
    }

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::EnsureTouchScreenAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTouchScreenManager);
    ::std::lock_guard<decltype(m_TouchScreenMutex)> locker(m_TouchScreenMutex);
    NN_RESULT_DO(
        m_pTouchScreenManager->EnsureAppletResourceForTouchScreen(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::SetTouchScreenAutoPilotState(
    const ::nn::hid::debug::TouchScreenAutoPilotState<TouchStateCountMax
        >& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTouchScreenManager);
    ::std::lock_guard<decltype(m_TouchScreenMutex)> locker(m_TouchScreenMutex);
    NN_RESULT_THROW(m_pTouchScreenManager->SetAutoPilotState(value));
}

::nn::Result SharedTask::UnsetTouchScreenAutoPilotState() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTouchScreenManager);
    ::std::lock_guard<decltype(m_TouchScreenMutex)> locker(m_TouchScreenMutex);
    NN_RESULT_THROW(m_pTouchScreenManager->UnsetAutoPilotState());
}

::nn::Result SharedTask::AcquireHomeButtonEventHandle(
    ::nn::os::NativeHandle* pOutValue,
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pHomeButtonManager);
    NN_RESULT_THROW_UNLESS(
        pOutValue != nullptr, ResultHomeButtonEventNullHandleBuffer());
    ::std::lock_guard<decltype(m_HomeButtonMutex)> locker(m_HomeButtonMutex);
    NN_RESULT_DO(
        m_pHomeButtonManager->AcquireHomeButtonEventHandle(pOutValue, aruid));
    // ハンドルの取得直後はシグナル状態であることを保証
    m_pHomeButtonManager->SignalHomeButtonEvent();
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::ActivateHomeButton() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pHomeButtonManager);

    auto needsRollback = true;

    NN_RESULT_DO(this->ActivateThread());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    ::std::lock_guard<decltype(m_HomeButtonMutex)> locker(m_HomeButtonMutex);
    NN_RESULT_DO(m_pHomeButtonManager->Activate());

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivateHomeButton() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pHomeButtonManager);

    {
        ::std::lock_guard<decltype(m_HomeButtonMutex)
                          > locker(m_HomeButtonMutex);
        NN_RESULT_DO(m_pHomeButtonManager->Deactivate());
    }

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::EnsureHomeButtonAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pHomeButtonManager);
    ::std::lock_guard<decltype(m_HomeButtonMutex)> locker(m_HomeButtonMutex);
    NN_RESULT_DO(m_pHomeButtonManager->EnsureAppletResource(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::SetHomeButtonAutoPilotState(
    const ::nn::hid::debug::HomeButtonAutoPilotState& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pHomeButtonManager);
    ::std::lock_guard<decltype(m_HomeButtonMutex)> locker(m_HomeButtonMutex);
    NN_RESULT_THROW(m_pHomeButtonManager->SetAutoPilotState(value));
}

::nn::Result SharedTask::UnsetHomeButtonAutoPilotState() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pHomeButtonManager);
    ::std::lock_guard<decltype(m_HomeButtonMutex)> locker(m_HomeButtonMutex);
    NN_RESULT_THROW(m_pHomeButtonManager->UnsetAutoPilotState());
}

::nn::Result SharedTask::AcquireSleepButtonEventHandle(
    ::nn::os::NativeHandle* pOutValue,
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pSleepButtonManager);
    NN_RESULT_THROW_UNLESS(
        pOutValue != nullptr, ResultSleepButtonEventNullHandleBuffer());
    ::std::lock_guard<decltype(m_SleepButtonMutex)> locker(m_SleepButtonMutex);
    NN_RESULT_DO(
        m_pSleepButtonManager->AcquireSleepButtonEventHandle(pOutValue, aruid));
    // ハンドルの取得直後はシグナル状態であることを保証
    m_pSleepButtonManager->SignalSleepButtonEvent();
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::ActivateSleepButton() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pSleepButtonManager);

    auto needsRollback = true;

    NN_RESULT_DO(this->ActivateThread());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    ::std::lock_guard<decltype(m_SleepButtonMutex)> locker(m_SleepButtonMutex);
    NN_RESULT_DO(m_pSleepButtonManager->Activate());

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivateSleepButton() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pSleepButtonManager);

    {
        ::std::lock_guard<decltype(m_SleepButtonMutex)
                          > locker(m_SleepButtonMutex);
        NN_RESULT_DO(m_pSleepButtonManager->Deactivate());
    }

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::EnsureSleepButtonAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pSleepButtonManager);
    ::std::lock_guard<decltype(m_SleepButtonMutex)> locker(m_SleepButtonMutex);
    NN_RESULT_DO(m_pSleepButtonManager->EnsureAppletResource(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::SetSleepButtonAutoPilotState(
    const ::nn::hid::debug::SleepButtonAutoPilotState& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pSleepButtonManager);
    ::std::lock_guard<decltype(m_SleepButtonMutex)> locker(m_SleepButtonMutex);
    NN_RESULT_THROW(m_pSleepButtonManager->SetAutoPilotState(value));
}

::nn::Result SharedTask::UnsetSleepButtonAutoPilotState() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pSleepButtonManager);
    ::std::lock_guard<decltype(m_SleepButtonMutex)> locker(m_SleepButtonMutex);
    NN_RESULT_THROW(m_pSleepButtonManager->UnsetAutoPilotState());
}

::nn::Result SharedTask::AcquireCaptureButtonEventHandle(
    ::nn::os::NativeHandle* pOutValue,
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pCaptureButtonManager);
    NN_RESULT_THROW_UNLESS(
        pOutValue != nullptr, ResultCaptureButtonEventNullHandleBuffer());
    ::std::lock_guard<decltype(m_CaptureButtonMutex)
                      > locker(m_CaptureButtonMutex);
    NN_RESULT_DO(
        m_pCaptureButtonManager->AcquireCaptureButtonEventHandle(
            pOutValue, aruid));
    // ハンドルの取得直後はシグナル状態であることを保証
    m_pCaptureButtonManager->SignalCaptureButtonEvent();
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::ActivateCaptureButton() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pCaptureButtonManager);

    auto needsRollback = true;

    NN_RESULT_DO(this->ActivateThread());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    ::std::lock_guard<decltype(m_CaptureButtonMutex)
                      > locker(m_CaptureButtonMutex);
    NN_RESULT_DO(m_pCaptureButtonManager->Activate());

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivateCaptureButton() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pCaptureButtonManager);

    {
        ::std::lock_guard<decltype(m_CaptureButtonMutex)
                          > locker(m_CaptureButtonMutex);
        NN_RESULT_DO(m_pCaptureButtonManager->Deactivate());
    }

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::EnsureCaptureButtonAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pCaptureButtonManager);
    ::std::lock_guard<decltype(m_CaptureButtonMutex)
                      > locker(m_CaptureButtonMutex);
    NN_RESULT_DO(m_pCaptureButtonManager->EnsureAppletResource(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::SetCaptureButtonAutoPilotState(
    const ::nn::hid::debug::CaptureButtonAutoPilotState& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pCaptureButtonManager);
    ::std::lock_guard<decltype(m_CaptureButtonMutex)
                      > locker(m_CaptureButtonMutex);
    NN_RESULT_DO(m_pCaptureButtonManager->SetAutoPilotState(value));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::UnsetCaptureButtonAutoPilotState() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pCaptureButtonManager);
    ::std::lock_guard<decltype(m_CaptureButtonMutex)
                      > locker(m_CaptureButtonMutex);
    NN_RESULT_DO(m_pCaptureButtonManager->UnsetAutoPilotState());
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::ActivateConsoleSixAxisSensor() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pConsoleSixAxisSensorManager);

    auto needsRollback = true;

    NN_RESULT_DO(this->ActivateThread());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    ::std::lock_guard<decltype(m_ConsoleSixAxisSensorMutex)> locker(m_ConsoleSixAxisSensorMutex);
    NN_RESULT_DO(m_pConsoleSixAxisSensorManager->ActivateForConsoleSixAxisSensor());

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivateConsoleSixAxisSensor() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pConsoleSixAxisSensorManager);

    {
        ::std::lock_guard<decltype(m_ConsoleSixAxisSensorMutex)
                          > locker(m_ConsoleSixAxisSensorMutex);
        NN_RESULT_DO(m_pConsoleSixAxisSensorManager->DeactivateForConsoleSixAxisSensor());
    }

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::GetConsoleSixAxisSensorCountStates(
    int* pOutCount,
    ::nn::hid::debug::ConsoleSixAxisSensorCountState* pOutValues,
    int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutValues);
    NN_UNUSED(count);
    NN_UNUSED(pOutCount);
    NN_UNUSED(pOutValues);

    // 何もしない

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::GetConsoleSixAxisSensorSamplingFrequency(
    int64_t* pOutValue,
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pConsoleSixAxisSensorManager);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_UNUSED(aruid);

    {
        ::std::lock_guard<decltype(m_ConsoleSixAxisSensorMutex)
        > locker(m_ConsoleSixAxisSensorMutex);
        *pOutValue = m_pConsoleSixAxisSensorManager->GetSamplingFrequency();
    }

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::EnsureConsoleSixAxisSensorAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pConsoleSixAxisSensorManager);
    ::std::lock_guard<decltype(m_ConsoleSixAxisSensorMutex)> locker(m_ConsoleSixAxisSensorMutex);
    NN_RESULT_DO(
        m_pConsoleSixAxisSensorManager->EnsureAppletResourceForConsoleSixAxisSensor(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::GetConsoleSixAxisSensorCalibrationValues(
    tmp::ConsoleSixAxisSensorCalibrationValues* pOutValues) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValues);
    NN_RESULT_DO(
        m_pConsoleSixAxisSensorManager->GetConsoleSixAxisSensorCalibrationValues(pOutValues));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::IsConsoleSixAxisSensorUserCalibrationSupported(
    bool* pOutIsSupported) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pConsoleSixAxisSensorManager);
    NN_SDK_REQUIRES_NOT_NULL(pOutIsSupported);

    NN_RESULT_DO(
        m_pConsoleSixAxisSensorManager
            ->IsConsoleSixAxisSensorUserCalibrationSupported(pOutIsSupported));

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::ResetConsoleSixAxisSensorCalibrationValues() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pConsoleSixAxisSensorManager);
    NN_RESULT_DO(m_pConsoleSixAxisSensorManager->ResetConsoleSixAxisSensorCalibrationValues());
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::StartConsoleSixAxisSensorUserCalibration() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pConsoleSixAxisSensorManager);
    NN_RESULT_DO(m_pConsoleSixAxisSensorManager->StartConsoleSixAxisSensorUserCalibration());
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::CancelConsoleSixAxisSensorUserCalibration() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pConsoleSixAxisSensorManager);
    NN_RESULT_DO(m_pConsoleSixAxisSensorManager->CancelConsoleSixAxisSensorUserCalibration());
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::GetSixAxisSensorAccurateUserCalibrationState(
    ::nn::hid::system::SixAxisSensorAccurateUserCalibrationState* pOutState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pConsoleSixAxisSensorManager);
    NN_RESULT_DO(
        m_pConsoleSixAxisSensorManager->GetSixAxisSensorAccurateUserCalibrationState(pOutState));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::SetStateBufferMemoryHandle(
    const ::nn::applet::AppletResourceUserId& aruid,
    ::nn::os::NativeHandle&& transferMemoryHandle,
    size_t size, bool isManaged) NN_NOEXCEPT
{
    NN_UNUSED(aruid);
    NN_UNUSED(transferMemoryHandle);
    NN_UNUSED(size);
    NN_UNUSED(isManaged);

    NN_SDK_REQUIRES_NOT_NULL(m_pConsoleSixAxisSensorManager);
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::SetWorkBufferMemoryHandle(
    const ::nn::applet::AppletResourceUserId& aruid,
    ::nn::os::NativeHandle&& transferMemoryHandle,
    size_t size, bool isManaged) NN_NOEXCEPT
{
    NN_UNUSED(aruid);
    NN_UNUSED(transferMemoryHandle);
    NN_UNUSED(size);
    NN_UNUSED(isManaged);

    NN_SDK_REQUIRES_NOT_NULL(m_pConsoleSixAxisSensorManager);
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::InitializeSevenSixAxisSensor(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    NN_SDK_REQUIRES_NOT_NULL(m_pConsoleSixAxisSensorManager);
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::FinalizeSevenSixAxisSensor(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    NN_SDK_REQUIRES_NOT_NULL(m_pConsoleSixAxisSensorManager);
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::ActivateAudioControl() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAudioControlManager);

    auto needsRollback = true;

    NN_RESULT_DO(this->ActivateThread());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    ::std::lock_guard<decltype(m_AudioControlMutex)> locker(m_AudioControlMutex);
    NN_RESULT_DO(m_pAudioControlManager->Activate());

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivateAudioControl() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAudioControlManager);

    {
        ::std::lock_guard<decltype(m_AudioControlMutex)> locker(m_AudioControlMutex);
        NN_RESULT_DO(m_pAudioControlManager->Deactivate());
    }

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

void SharedTask::SetInputDetectorManager(InputDetectorManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    m_pInputDetectorManager = pManager;
}

void SharedTask::SetAbstractedPadXcd(AbstractedPadXcd* pXcd, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pXcd);
    NN_SDK_REQUIRES_GREATER(count, 0);

    m_pAbstractedPadXcd = pXcd;
    m_AbstractedPadXcdCount = count;
}

void SharedTask::SetAbstractedPadXcdManager(AbstractedPadXcdManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    m_pAbstractedPadXcdManager = pManager;
}

void SharedTask::SetAbstractedAutoPilotPadManager(
    AbstractedAutoPilotPadManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    m_pAbstractedAutoPilotPadManager = pManager;
}

void SharedTask::SetAbstractedAutoPilotVirtualPadManager(
    AbstractedAutoPilotVirtualPadManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    m_pAbstractedAutoPilotVirtualPadManager = pManager;
}

void SharedTask::SetAbstractedPadHidShellManager(
    AbstractedPadHidShellManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    m_pAbstractedPadHidShellManager = pManager;
}

::nn::Result SharedTask::ActivateInputDetector() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pInputDetectorManager);

    ::std::lock_guard<decltype(m_InputDetectorMutex)> locker(m_InputDetectorMutex);
    NN_RESULT_DO(m_pInputDetectorManager->Activate());

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivateInputDetector() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pInputDetectorManager);

    {
        ::std::lock_guard<decltype(m_InputDetectorMutex)
        > locker(m_InputDetectorMutex);
        NN_RESULT_DO(m_pInputDetectorManager->Deactivate());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::NotifyInputDetector(const system::InputSourceIdSet& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pInputDetectorManager);
    ::std::lock_guard<decltype(m_InputDetectorMutex)> locker(m_InputDetectorMutex);
    NN_RESULT_DO(m_pInputDetectorManager->Notify(id));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::EnsureInputDetectorAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pInputDetectorManager);
    ::std::lock_guard<decltype(m_InputDetectorMutex)> locker(m_InputDetectorMutex);
    NN_RESULT_DO(m_pInputDetectorManager->EnsureAppletResource(aruid));
    NN_RESULT_SUCCESS;
}

void SharedTask::ActivatePlayReport() NN_NOEXCEPT
{
    m_pPlayReportManager->Activate();
}

void SharedTask::DeactivatePlayReport() NN_NOEXCEPT
{
    m_pPlayReportManager->Deactivate();
}

void SharedTask::ActivateRegisteredDeviceManagement() NN_NOEXCEPT
{
    m_pRegisteredDeviceManager->Activate();
}

void SharedTask::DeactivateRegisteredDeviceManagement() NN_NOEXCEPT
{
    m_pRegisteredDeviceManager->Deactivate();
}

Result SharedTask::AcquirePlayReportControllerUsageUpdateEventHandle(
    ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    NN_RESULT_DO(m_pPlayReportManager->AcquireUsageUpdateEventHandle(pOutValue));
    NN_RESULT_SUCCESS;
}

int SharedTask::GetPlayReportControllerUsages(
    system::PlayReportControllerUsage* pOutValues, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValues);
    NN_SDK_REQUIRES_GREATER(count, 0);

    return m_pPlayReportManager->GetControllerUsages(pOutValues, count);
}

Result SharedTask::AcquirePlayReportRegisteredDevicesEventHandle(
    ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    NN_RESULT_DO(m_pRegisteredDeviceManager->AcquireUpdateForPlayReportEventHandle(pOutValue));
    NN_RESULT_SUCCESS;
}

int SharedTask::GetRegisteredDevices(
    system::RegisteredDeviceOld* pOutValues, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValues);
    NN_SDK_REQUIRES_GREATER(count, 0);

    ::std::lock_guard<decltype(m_RegisteredDeviceMutex)
                      > locker(m_RegisteredDeviceMutex);
    return m_pRegisteredDeviceManager->GetRegisteredDevices(pOutValues, count);
}

int SharedTask::GetRegisteredDevices(
    system::RegisteredDevice* pOutValues, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValues);
    NN_SDK_REQUIRES_GREATER(count, 0);

    ::std::lock_guard<decltype(m_RegisteredDeviceMutex)
                      > locker(m_RegisteredDeviceMutex);
    return m_pRegisteredDeviceManager->GetRegisteredDevices(pOutValues, count);
}

void SharedTask::AddRegisteredDevices(const system::RegisteredDevice& device) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_RegisteredDeviceMutex)
                      > locker(m_RegisteredDeviceMutex);
    m_pRegisteredDeviceManager->AddRegisteredDevice(device);
}

Result SharedTask::AcquireConnectionTriggerTimeoutEvent(
    ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::lock_guard<decltype(m_RegisteredDeviceMutex)
                      > locker(m_RegisteredDeviceMutex);
    NN_RESULT_DO(m_pRegisteredDeviceManager->AcquireConnectionTriggerTimeoutEvent(pOutValue));
    NN_RESULT_SUCCESS;
}

Result SharedTask::SendConnectionTrigger(::nn::bluetooth::Address& address) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_RegisteredDeviceMutex)
                      > locker(m_RegisteredDeviceMutex);
    NN_RESULT_DO(m_pRegisteredDeviceManager->SendConnectionTrigger(address));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::AcquireAudioControlEventHandle(
    ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAudioControlManager);
    NN_RESULT_THROW_UNLESS(
        pOutValue != nullptr, ResultAudioControlEventNullHandleBuffer());
    ::std::lock_guard<decltype(m_AudioControlMutex)> locker(m_AudioControlMutex);
    NN_RESULT_DO(
        m_pAudioControlManager->AcquireAudioControlEventHandle(pOutValue));
    // ハンドルの取得直後はシグナル状態であることを保証
    m_pAudioControlManager->SignalAudioControlEvent();
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::GetAudioControlStates(
    int* pOutCount,
    system::AudioControlState outStates[], int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAudioControlManager);
    NN_RESULT_THROW_UNLESS(
        outStates != nullptr, ResultAudioControlEventNullHandleBuffer());
    ::std::lock_guard<decltype(m_AudioControlMutex)> locker(m_AudioControlMutex);
    NN_RESULT_DO(
        m_pAudioControlManager->GetStates(pOutCount, outStates, count));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::IsUsbFullKeyControllerEnabled(bool* pOutEnabled) NN_NOEXCEPT
{
    NN_RESULT_DO(m_pAbstractedPadXcdManager->IsUsbFullKeyControllerEnabled(pOutEnabled));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::EnableUsbFullKeyController(bool enabled) NN_NOEXCEPT
{
    NN_RESULT_DO(m_pAbstractedPadXcdManager->EnableUsbFullKeyController(enabled));
    NN_RESULT_SUCCESS;
}

Result SharedTask::AcquireDeviceRegisteredEventForControllerSupport(
    ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::lock_guard<decltype(m_RegisteredDeviceMutex)
                      > locker(m_RegisteredDeviceMutex);
    NN_RESULT_DO(
        m_pRegisteredDeviceManager->AcquireDeviceRegisteredEventForControllerSupport(pOutValue));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::ActivateAbstractedXcdManager() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAbstractedPadXcdManager);
    auto needsRollback = true;

    // デフォルトはシグナル状態
    nn::os::SignalSystemEvent(m_XcdDeviceUpdateEvent.GetBase());
    // XCD にSystemEvent を登録
    nn::xcd::BindLinkUpdateEvent(m_XcdDeviceUpdateEvent.GetBase());

    NN_RESULT_DO(m_pAbstractedPadXcdManager->Activate());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            m_pAbstractedPadXcdManager->Deactivate();
        }
    };

    NN_RESULT_DO(this->ActivateThread());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivateAbstractedXcdManager() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAbstractedPadXcdManager);

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_DO(this->m_pAbstractedPadXcdManager->Deactivate());

    // XCD に登録されているSystemEvent をアンバインド
    nn::xcd::UnbindLinkUpdateEvent(m_XcdDeviceUpdateEvent.GetBase());

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::SetAbstractedPadAutoPilotStateWithBasicXpad(
    int playerNumber,
    const ::nn::hid::debug::BasicXpadAutoPilotState& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(playerNumber, 0, AbstractedAutoPilotPadCount);
    NN_RESULT_THROW(
        m_pAbstractedAutoPilotPadManager->SetAutoPilotState(playerNumber, value));
}

::nn::Result SharedTask::UnsetAbstractedPadAutoPilotState(int playerNumber) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(playerNumber, 0, AbstractedAutoPilotPadCount);
    NN_RESULT_THROW(
        m_pAbstractedAutoPilotPadManager->UnsetAutoPilotState(playerNumber));
}

::nn::Result SharedTask::SetAbstractedPadAutoPilotVirtualPadState(
    const ::nn::hid::debug::AbstractedVirtualPadId& virtualPadId,
    const ::nn::hid::debug::AbstractedPadState& value) NN_NOEXCEPT
{
    NN_RESULT_THROW(
        m_pAbstractedAutoPilotVirtualPadManager->SetAutoPilotState(virtualPadId, value));
}

::nn::Result SharedTask::UnsetAbstractedPadAutoPilotVirtualPadState(
    const ::nn::hid::debug::AbstractedVirtualPadId& virtualPadId) NN_NOEXCEPT
{
    NN_RESULT_THROW(
        m_pAbstractedAutoPilotVirtualPadManager->UnsetAutoPilotState(virtualPadId));

}

::nn::Result SharedTask::UnsetAllAbstractedPadAutoPilotVirtualPadState() NN_NOEXCEPT
{
    NN_RESULT_THROW(
        m_pAbstractedAutoPilotVirtualPadManager->UnsetAllAutoPilotState());
}

::nn::Result SharedTask::ActivatePadCommonTimer() NN_NOEXCEPT
{
    auto needsRollback = true;

    NN_RESULT_DO(this->ActivatePadCommonTimerEvent());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivatePadCommonTimerEvent();
        }
    };

    NN_RESULT_DO(this->ActivateThread());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivatePadCommonTimer() NN_NOEXCEPT
{
    NN_RESULT_DO(this->DeactivatePadCommonTimerEvent());

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

void SharedTask::LinkTimerEvent(::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
{
    m_XcdDeviceUpdateEvent.Link(pMultiWait);
    m_TouchScreenTimerEvent.Link(pMultiWait);
    m_BluetoothFirmwareUpdateSystemEvent.Link(pMultiWait);
    m_McuFirmwareUpdateSystemEvent.Link(pMultiWait);
    m_FirmwareUpdateNewSampleSystemEvent.Link(pMultiWait);
    m_FirmwareUpdateResetTimeoutEvent.Link(pMultiWait);
    m_PairingCompleteEvent.Link(pMultiWait);
    m_ConnectionTriggerTimeoutEvent.Link(pMultiWait);
}

void SharedTask::LinkTimers(SynchronizedTimerGroup* group) NN_NOEXCEPT
{
    m_KeyboardTimerEvent.Link(group);
    m_MouseTimerEvent.Link(group);
    m_HomeButtonTimerEvent.Link(group);
    m_SleepButtonTimerEvent.Link(group);
    m_CaptureButtonTimerEvent.Link(group);
    m_PlayReportTimerEvent.Link(group);
    m_ConsoleSixAxisSensorTimerEvent.Link(group);
    m_PadCommonTimerEvent.Link(group);
    m_AudioControlTimerEvent.Link(group);
}

void SharedTask::UnlinkTimerEvent() NN_NOEXCEPT
{
    m_XcdDeviceUpdateEvent.Unlink();
    m_TouchScreenTimerEvent.Unlink();
    m_BluetoothFirmwareUpdateSystemEvent.Unlink();
    m_McuFirmwareUpdateSystemEvent.Unlink();
    m_FirmwareUpdateNewSampleSystemEvent.Unlink();
    m_FirmwareUpdateResetTimeoutEvent.Unlink();
    m_PairingCompleteEvent.Unlink();
    m_ConnectionTriggerTimeoutEvent.Unlink();
    m_AudioControlTimerEvent.Unlink();
}

void SharedTask::UnlinkTimers() NN_NOEXCEPT
{
    m_KeyboardTimerEvent.Unlink();
    m_MouseTimerEvent.Unlink();
    m_HomeButtonTimerEvent.Unlink();
    m_SleepButtonTimerEvent.Unlink();
    m_CaptureButtonTimerEvent.Unlink();
    m_ConsoleSixAxisSensorTimerEvent.Unlink();
    m_PlayReportTimerEvent.Unlink();
    m_PadCommonTimerEvent.Unlink();
}

void SharedTask::DoSampling(
    const ::nn::os::MultiWaitHolderType* waitId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTouchScreenManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pConsoleSixAxisSensorManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pPlayReportManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pRegisteredDeviceManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pAbstractedPadXcdManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pAudioControlManager);

    if (waitId == m_XcdDeviceUpdateEvent.GetWaitId())
    {
        m_XcdDeviceUpdateEvent.Clear();
        m_pAbstractedPadXcdManager->UpdateDevices();
        m_pUniquePadResourceManager->HandleDeviceUpdate();

        return;
    }

    if (waitId == m_TouchScreenTimerEvent.GetWaitId())
    {
        m_TouchScreenTimerEvent.Clear();
        // TouchScreen の状態をサンプリング
        ::std::lock_guard<decltype(m_TouchScreenMutex)
                          > locker(m_TouchScreenMutex);
        m_pTouchScreenManager->Sample();
        return;
    }

    if (waitId == m_BluetoothFirmwareUpdateSystemEvent.GetWaitId())
    {
        m_BluetoothFirmwareUpdateSystemEvent.Clear();
        m_pUniquePadResourceManager->HandleBluetoothFirmwareUpdate();
        return;
    }

    if (waitId == m_McuFirmwareUpdateSystemEvent.GetWaitId())
    {
        m_McuFirmwareUpdateSystemEvent.Clear();
        m_pUniquePadResourceManager->HandleMcuFirmwareUpdate();
        return;
    }

    if (waitId == m_FirmwareUpdateNewSampleSystemEvent.GetWaitId())
    {
        m_FirmwareUpdateNewSampleSystemEvent.Clear();
        m_pUniquePadResourceManager->HandleNewSample();
        return;
    }

    if (waitId == m_FirmwareUpdateResetTimeoutEvent.GetWaitId())
    {
        m_FirmwareUpdateResetTimeoutEvent.Clear();
        m_pUniquePadResourceManager->HandleResetTimedOut();
        return;
    }

    if (waitId == m_PairingCompleteEvent.GetWaitId())
    {
        m_PairingCompleteEvent.Clear();
        ::std::lock_guard<decltype(m_RegisteredDeviceMutex)
                          > locker(m_RegisteredDeviceMutex);
        m_pRegisteredDeviceManager->UpdateRegisteredDevice();
        return;
    }

    if (waitId == m_ConnectionTriggerTimeoutEvent.GetWaitId())
    {
        m_ConnectionTriggerTimeoutEvent.Clear();
        ::std::lock_guard<decltype(m_RegisteredDeviceMutex)
                          > locker(m_RegisteredDeviceMutex);
        m_pRegisteredDeviceManager->SignalConnectionTriggerTimeout();
        m_pUniquePadResourceManager->HandleConnectionTriggerTimeout();
        return;
    }
}

void SharedTask::DoSamplingForTimer(
    const SynchronizedTimer* waitId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pConsoleSixAxisSensorManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pPlayReportManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pRegisteredDeviceManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pNpadResourceManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pUniquePadResourceManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pAudioControlManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pAbstractedPadHidShellManager);

    if (this->ResolveKeyboardTimerEvent(waitId))
    {
        return;
    }

    if (this->ResolveMouseTimerEvent(waitId))
    {
        return;
    }

    if (this->ResolveHomeButtonTimerEvent(waitId))
    {
        return;
    }

    if (this->ResolveSleepButtonTimerEvent(waitId))
    {
        return;
    }

    if (this->ResolveCaptureButtonTimerEvent(waitId))
    {
        return;
    }

    if (waitId == &m_ConsoleSixAxisSensorTimerEvent)
    {
        m_ConsoleSixAxisSensorTimerEvent.Clear();
        // ConsoleSixAxisSensor の状態をサンプリング
        ::std::lock_guard<decltype(m_ConsoleSixAxisSensorMutex)
                          > locker(m_ConsoleSixAxisSensorMutex);
        m_pConsoleSixAxisSensorManager->Sample();
        return;
    }

    if (waitId == &m_PlayReportTimerEvent)
    {
        m_PlayReportTimerEvent.Clear();
        m_pPlayReportManager->CheckForUpdates();
    }

    if (waitId == &m_PadCommonTimerEvent)
    {
        m_PadCommonTimerEvent.Clear();

        m_pAbstractedPadHidShellManager->Recieve();
        m_pAbstractedPadXcdManager->HandlePeriodicalEvent();
        for (int i = 0; i < m_AbstractedPadXcdCount; ++i)
        {
            m_pAbstractedPadXcd[i].Update();
        }
        m_pNpadResourceManager->HandlePeriodcialEvent();
        m_pUniquePadResourceManager->HandlePeriodicalEvent();
        m_pAbstractedPadHidShellManager->Send();

        return;
    }

    if (waitId == &m_AudioControlTimerEvent)
    {
        m_AudioControlTimerEvent.Clear();
        ::std::lock_guard<decltype(m_AudioControlMutex)> locker(m_AudioControlMutex);
        m_pAudioControlManager->Sample();
        return;
    }
}

::nn::Result SharedTask::ActivatePadCommonTimerEvent() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_PadCommonTimerEventActivationCount.IsMax(),
                           ResultGamePadDriverActivationUpperLimitOver());

    if (m_PadCommonTimerEventActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施
        m_PadCommonTimerEvent.Enable(PadCommonTimerInterval);
    }

    ++m_PadCommonTimerEventActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivatePadCommonTimerEvent() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_PadCommonTimerEventActivationCount.IsZero(),
                           ResultGamePadDriverDeactivationLowerLimitOver());

    --m_PadCommonTimerEventActivationCount;

    if (m_PadCommonTimerEventActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施
        m_PadCommonTimerEvent.Disable();
    }

    NN_RESULT_SUCCESS;
}

bool SharedTask::ResolveKeyboardTimerEvent(
    const SynchronizedTimer* waitId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pKeyboardManager);

    if (waitId != &m_KeyboardTimerEvent)
    {
        return false;
    }

    m_KeyboardTimerEvent.Clear();

    ::std::lock_guard<decltype(m_KeyboardMutex)> locker(m_KeyboardMutex);

    m_pKeyboardManager->Sample();

    return true;
}

bool SharedTask::ResolveMouseTimerEvent(
    const SynchronizedTimer* waitId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pMouseManager);

    if (waitId != &m_MouseTimerEvent)
    {
        return false;
    }

    m_MouseTimerEvent.Clear();

    ::std::lock_guard<decltype(m_MouseMutex)> locker(m_MouseMutex);

    m_pMouseManager->Sample();

    return true;
}

bool SharedTask::ResolveHomeButtonTimerEvent(
    const SynchronizedTimer* waitId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pHomeButtonManager);

    if (waitId != &m_HomeButtonTimerEvent)
    {
        return false;
    }

    m_HomeButtonTimerEvent.Clear();

    ::std::lock_guard<decltype(m_HomeButtonMutex)> locker(m_HomeButtonMutex);

    m_pHomeButtonManager->Sample();

    return true;
}

bool SharedTask::ResolveSleepButtonTimerEvent(
    const SynchronizedTimer* waitId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pSleepButtonManager);

    if (waitId != &m_SleepButtonTimerEvent)
    {
        return false;
    }

    m_SleepButtonTimerEvent.Clear();

    ::std::lock_guard<decltype(m_SleepButtonMutex)> locker(m_SleepButtonMutex);

    m_pSleepButtonManager->Sample();

    return true;
}

bool SharedTask::ResolveCaptureButtonTimerEvent(
    const SynchronizedTimer* waitId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pCaptureButtonManager);

    if (waitId != &m_CaptureButtonTimerEvent)
    {
        return false;
    }

    m_CaptureButtonTimerEvent.Clear();

    ::std::lock_guard<decltype(m_CaptureButtonMutex)
                      > locker(m_CaptureButtonMutex);

    m_pCaptureButtonManager->Sample();

    return true;
}

DebugPadTask::DebugPadTask() NN_NOEXCEPT
{
    // 何もしない
}

DebugPadTask::~DebugPadTask() NN_NOEXCEPT
{
    // 何もしない
}

void DebugPadTask::SetDebugPadManager(DebugPadManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    pManager->SetTimer(&m_DebugPadTimerEvent);
    m_pDebugPadManager = pManager;
}

void DebugPadTask::SetWindowsGenericPadBrokerTask(
    WindowsGenericPadBrokerTask* pWindowsGenericPadBrokerTask) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pWindowsGenericPadBrokerTask);
    m_pWindowsGenericPadBrokerTask = pWindowsGenericPadBrokerTask;
}

::nn::Result DebugPadTask::ActivateDebugPad() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDebugPadManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pWindowsGenericPadBrokerTask);

    auto needsRollback = true;

    NN_RESULT_DO(
        m_pWindowsGenericPadBrokerTask->ActivateWindowsGenericPadBroker());

    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            m_pWindowsGenericPadBrokerTask->DeactivateWindowsGenericPadBroker();
        }
    };

    NN_RESULT_DO(this->ActivateThread());

    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    ::std::lock_guard<decltype(m_DebugPadMutex)> locker(m_DebugPadMutex);

    NN_RESULT_DO(m_pDebugPadManager->Activate());

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result DebugPadTask::DeactivateDebugPad() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDebugPadManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pWindowsGenericPadBrokerTask);

    {
        ::std::lock_guard<decltype(m_DebugPadMutex)> locker(m_DebugPadMutex);

        NN_RESULT_DO(m_pDebugPadManager->Deactivate());
    }

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_DO(
        m_pWindowsGenericPadBrokerTask->DeactivateWindowsGenericPadBroker());

    NN_RESULT_SUCCESS;
}

::nn::Result DebugPadTask::EnsureDebugPadAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDebugPadManager);
    ::std::lock_guard<decltype(m_DebugPadMutex)> locker(m_DebugPadMutex);
    NN_RESULT_DO(m_pDebugPadManager->EnsureAppletResource(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result DebugPadTask::SetDebugPadAutoPilotState(
    const ::nn::hid::debug::DebugPadAutoPilotState& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDebugPadManager);
    ::std::lock_guard<decltype(m_DebugPadMutex)> locker(m_DebugPadMutex);
    NN_RESULT_THROW(m_pDebugPadManager->SetAutoPilotState(value));
}

::nn::Result DebugPadTask::UnsetDebugPadAutoPilotState() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDebugPadManager);
    ::std::lock_guard<decltype(m_DebugPadMutex)> locker(m_DebugPadMutex);
    NN_RESULT_THROW(m_pDebugPadManager->UnsetAutoPilotState());
}

void DebugPadTask::LinkTimerEvent(
    ::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
{
    NN_UNUSED(pMultiWait);
}

void DebugPadTask::UnlinkTimerEvent() NN_NOEXCEPT
{
    // 何もしない
}

void DebugPadTask::DoSampling(
    const ::nn::os::MultiWaitHolderType* waitId) NN_NOEXCEPT
{
    NN_UNUSED(waitId);
}

void DebugPadTask::LinkTimers(SynchronizedTimerGroup* group) NN_NOEXCEPT
{
    m_DebugPadTimerEvent.Link(group);
}

void DebugPadTask::UnlinkTimers() NN_NOEXCEPT
{
    m_DebugPadTimerEvent.Unlink();
}

void DebugPadTask::DoSamplingForTimer(
    const SynchronizedTimer* waitId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDebugPadManager);

    if (waitId == &m_DebugPadTimerEvent)
    {
        m_DebugPadTimerEvent.Clear();
        ::std::lock_guard<decltype(m_DebugPadMutex)> locker(m_DebugPadMutex);
        m_pDebugPadManager->Sample();
        return;
    }
}

XpadTask::XpadTask() NN_NOEXCEPT
{
    // 何もしない
}

XpadTask::~XpadTask() NN_NOEXCEPT
{
    // 何もしない
}

void XpadTask::SetXpadManager(XpadManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    pManager->SetTimer(&m_XpadTimerEvent);
    m_pXpadManager = pManager;
}

void XpadTask::SetWindowsGenericPadBrokerTask(
    WindowsGenericPadBrokerTask* pWindowsGenericPadBrokerTask) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pWindowsGenericPadBrokerTask);
    m_pWindowsGenericPadBrokerTask = pWindowsGenericPadBrokerTask;
}

::nn::Result XpadTask::ActivateBasicXpad() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pXpadManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pWindowsGenericPadBrokerTask);

    auto needsRollback = true;

    NN_RESULT_DO(
        m_pWindowsGenericPadBrokerTask->ActivateWindowsGenericPadBroker());

    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            m_pWindowsGenericPadBrokerTask->DeactivateWindowsGenericPadBroker();
        }
    };

    NN_RESULT_DO(this->ActivateThread());

    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    ::std::lock_guard<decltype(m_XpadMutex)> locker(m_XpadMutex);

    NN_RESULT_DO(m_pXpadManager->ActivateBasicXpad());

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result XpadTask::DeactivateBasicXpad() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pXpadManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pWindowsGenericPadBrokerTask);

    {
        ::std::lock_guard<decltype(m_XpadMutex)> locker(m_XpadMutex);

        NN_RESULT_DO(m_pXpadManager->DeactivateBasicXpad());
    }

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_DO(
        m_pWindowsGenericPadBrokerTask->DeactivateWindowsGenericPadBroker());

    NN_RESULT_SUCCESS;
}

void XpadTask::EnableBasicXpadSampling() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pXpadManager);
    ::std::lock_guard<decltype(m_XpadMutex)> locker(m_XpadMutex);
    m_pXpadManager->EnableSampling();
}

::nn::Result XpadTask::EnsureBasicXpadAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pXpadManager);
    ::std::lock_guard<decltype(m_XpadMutex)> locker(m_XpadMutex);
    NN_RESULT_DO(m_pXpadManager->EnsureBasicXpadAppletResource(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result XpadTask::SetBasicXpadAutoPilotState(
    const ::nn::hid::debug::BasicXpadAutoPilotState& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pXpadManager);
    ::std::lock_guard<decltype(m_XpadMutex)> locker(m_XpadMutex);
    NN_RESULT_THROW(m_pXpadManager->SetBasicXpadAutoPilotState(value));
}

::nn::Result XpadTask::UnsetBasicXpadAutoPilotState() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pXpadManager);
    ::std::lock_guard<decltype(m_XpadMutex)> locker(m_XpadMutex);
    NN_RESULT_THROW(m_pXpadManager->UnsetBasicXpadAutoPilotState());
}

::nn::Result XpadTask::ActivateJoyXpad() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pXpadManager);

    auto needsRollback = true;

    NN_RESULT_DO(this->ActivateThread());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    ::std::lock_guard<decltype(m_XpadMutex)> locker(m_XpadMutex);
    NN_RESULT_DO(m_pXpadManager->ActivateJoyXpad());

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result XpadTask::DeactivateJoyXpad() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pXpadManager);

    {
        ::std::lock_guard<decltype(m_XpadMutex)> locker(m_XpadMutex);
        NN_RESULT_DO(m_pXpadManager->DeactivateJoyXpad());
    }

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

void XpadTask::EnableJoyXpadSampling() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pXpadManager);
    ::std::lock_guard<decltype(m_XpadMutex)> locker(m_XpadMutex);
    m_pXpadManager->EnableSampling();
}

::nn::Result XpadTask::GetJoyXpadLifoHandle(
    ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(m_pXpadManager);
    ::std::lock_guard<decltype(m_XpadMutex)> locker(m_XpadMutex);
    NN_RESULT_THROW(m_pXpadManager->GetJoyXpadLifoHandle(pOutValue));
}

::nn::Result XpadTask::ActivateSixAxisSensor() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pXpadManager);
    NN_RESULT_DO(m_pXpadManager->ActivateSixAxisSensor());
    NN_RESULT_SUCCESS;
}

::nn::Result XpadTask::DeactivateSixAxisSensor() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pXpadManager);
    NN_RESULT_DO(m_pXpadManager->DeactivateSixAxisSensor());
    NN_RESULT_SUCCESS;
}

::nn::Result XpadTask::GetSixAxisSensorLifoHandle(
    ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(m_pXpadManager);
    ::std::lock_guard<decltype(m_XpadMutex)> locker(m_XpadMutex);
    NN_RESULT_THROW(m_pXpadManager->GetSixAxisSensorLifoHandle(pOutValue));
}

::nn::Result XpadTask::ActivateJoySixAxisSensor() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pXpadManager);
    NN_RESULT_DO(m_pXpadManager->ActivateJoySixAxisSensor());
    NN_RESULT_SUCCESS;
}

::nn::Result XpadTask::DeactivateJoySixAxisSensor() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pXpadManager);
    NN_RESULT_DO(m_pXpadManager->DeactivateJoySixAxisSensor());
    NN_RESULT_SUCCESS;
}

::nn::Result XpadTask::GetJoySixAxisSensorLifoHandle(
    ::nn::os::NativeHandle* outValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES_NOT_NULL(m_pXpadManager);
    ::std::lock_guard<decltype(m_XpadMutex)> locker(m_XpadMutex);
    NN_RESULT_THROW(m_pXpadManager->GetJoySixAxisSensorLifoHandle(outValue));
}

void XpadTask::LinkTimerEvent(::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
{
    NN_UNUSED(pMultiWait);
}

void XpadTask::UnlinkTimerEvent() NN_NOEXCEPT
{
    // 何もしない
}

void XpadTask::DoSampling(
    const ::nn::os::MultiWaitHolderType* waitId) NN_NOEXCEPT
{
    NN_UNUSED(waitId);
}

void XpadTask::LinkTimers(SynchronizedTimerGroup* group) NN_NOEXCEPT
{
    m_XpadTimerEvent.Link(group);
}

void XpadTask::UnlinkTimers() NN_NOEXCEPT
{
    m_XpadTimerEvent.Unlink();
}

void XpadTask::DoSamplingForTimer(const SynchronizedTimer* waitId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pXpadManager);

    if (waitId == &m_XpadTimerEvent)
    {
        m_XpadTimerEvent.Clear();
        ::std::lock_guard<decltype(m_XpadMutex)> locker(m_XpadMutex);
        m_pXpadManager->Sample();
        return;
    }
}

AbstractedGenericPadTask::AbstractedGenericPadTask() NN_NOEXCEPT
    : m_pWindowsGenericPadBrokerTask(nullptr)
    , m_pAbstractedGenericPad(nullptr)
{
    // 何もしない
}

AbstractedGenericPadTask::~AbstractedGenericPadTask() NN_NOEXCEPT
{
    // 何もしない
}

void AbstractedGenericPadTask::SetWindowsGenericPadBrokerTask(
    WindowsGenericPadBrokerTask* pWindowsGenericPadBrokerTask) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pWindowsGenericPadBrokerTask);
    m_pWindowsGenericPadBrokerTask = pWindowsGenericPadBrokerTask;
}

::nn::Result AbstractedGenericPadTask::ActivateAbstractedGenericPad() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAbstractedGenericPad);
    NN_SDK_REQUIRES_NOT_NULL(m_pWindowsGenericPadBrokerTask);

    auto needsRollback = true;

    NN_RESULT_DO(
        m_pWindowsGenericPadBrokerTask->ActivateWindowsGenericPadBroker());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            m_pWindowsGenericPadBrokerTask->DeactivateWindowsGenericPadBroker();
        }
    };

    NN_RESULT_DO(this->ActivateThread());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    ::std::lock_guard<decltype(m_AbstractedGenericPadMutex)
                      > locker(m_AbstractedGenericPadMutex);
    NN_RESULT_DO(m_pAbstractedGenericPad->Activate());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            m_pAbstractedGenericPad->Deactivate();
        }
    };

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result AbstractedGenericPadTask::DeactivateAbstractedGenericPad() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAbstractedGenericPad);
    NN_SDK_REQUIRES_NOT_NULL(m_pWindowsGenericPadBrokerTask);

    {
        ::std::lock_guard<decltype(m_AbstractedGenericPadMutex)
                          > locker(m_AbstractedGenericPadMutex);
        NN_RESULT_DO(m_pAbstractedGenericPad->Deactivate());
    }

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_DO(
        m_pWindowsGenericPadBrokerTask->DeactivateWindowsGenericPadBroker());

    NN_RESULT_SUCCESS;
}

void AbstractedGenericPadTask::LinkTimerEvent(
    ::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
{
    m_AbstractedGenericPadTimerEvent.Link(pMultiWait);
}

void AbstractedGenericPadTask::UnlinkTimerEvent() NN_NOEXCEPT
{
    m_AbstractedGenericPadTimerEvent.Unlink();
}

void AbstractedGenericPadTask::DoSampling(
    const ::nn::os::MultiWaitHolderType* waitId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAbstractedGenericPad);

    if (waitId == m_AbstractedGenericPadTimerEvent.GetWaitId())
    {
        m_AbstractedGenericPadTimerEvent.Clear();
        ::std::lock_guard<decltype(m_AbstractedGenericPadMutex)
                          > locker(m_AbstractedGenericPadMutex);
        m_pAbstractedGenericPad->Update();
        return;
    }
}

void AbstractedGenericPadTask::SetAbstractedGenericPad(
    AbstractedGenericPad* pGenericPad) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pGenericPad);
    pGenericPad->SetTimerEvent(m_AbstractedGenericPadTimerEvent.GetBase());
    m_pAbstractedGenericPad = pGenericPad;
}


XcdTask::XcdTask() NN_NOEXCEPT
{
    // 何もしない
}

XcdTask::~XcdTask() NN_NOEXCEPT
{
    // 何もしない
}

void XcdTask::SetXcdDriver(XcdDriver* pXcdDriver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pXcdDriver);

    m_pXcdDriver = pXcdDriver;
}

void XcdTask::LinkTimerEvent(::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
{
    m_pXcdDriver->LinkMultiWait(pMultiWait);

    // Xcd 内部で多重待ちオブジェクトへのイベントのリンクを行うので、ここで初期化する
    ::std::lock_guard<decltype(m_XcdMutex)> locker(m_XcdMutex);
    m_pXcdDriver->Activate();
}

void XcdTask::UnlinkTimerEvent() NN_NOEXCEPT
{
    {
        ::std::lock_guard<decltype(m_XcdMutex)> locker(m_XcdMutex);
        m_pXcdDriver->Deactivate();
    }

    m_pXcdDriver->UnlinkMultiWait();
}

void XcdTask::DoSampling(
    const ::nn::os::MultiWaitHolderType* waitId) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_XcdMutex)> lock(m_XcdMutex);

    if (m_pXcdDriver->Proceed(waitId) == true)
    {
        return;
    }
}

void XcdTask::InvokeEventLoop() NN_NOEXCEPT
{
    while (!nn::os::TryWaitEvent(m_TerminationEvent.GetBase()))
    {
        auto* pWaitId = m_pXcdDriver->WaitEvent();
        if (pWaitId == nullptr)
        {
            std::lock_guard<decltype(m_XcdMutex)> lock(m_XcdMutex);
            m_pXcdDriver->ProceedPeriodic();
        }
        else
        {
            std::lock_guard<decltype(m_XcdMutex)> lock(m_XcdMutex);
            m_pXcdDriver->Proceed(pWaitId);
        }
    }
}

::nn::Result XcdTask::ActivateXcdDriver() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pXcdDriver);

    auto needsRollback = true;

    NN_RESULT_DO(this->ActivateThread());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result XcdTask::DeactivateXcdDriver() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pXcdDriver);

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

WindowsGenericPadBrokerTask::WindowsGenericPadBrokerTask() NN_NOEXCEPT
    : m_BrokerCompletionEvent(::nn::os::EventClearMode_ManualClear)
{
    // 何もしない
}

WindowsGenericPadBrokerTask::~WindowsGenericPadBrokerTask() NN_NOEXCEPT
{
    // 何もしない
}

void WindowsGenericPadBrokerTask::SetWindowsGenericPadBroker(
    WindowsGenericPadBroker* pWindowsGenericPadBroker) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pWindowsGenericPadBroker);
    pWindowsGenericPadBroker->SetTimerEvent(m_BrokerTimerEvent.GetBase());
    m_pWindowsGenericPadBroker = pWindowsGenericPadBroker;
}

::nn::Result WindowsGenericPadBrokerTask::ActivateWindowsGenericPadBroker(
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pWindowsGenericPadBroker);

    NN_RESULT_DO(this->ActivateThread());

    {
        ::std::lock_guard<decltype(m_EventMutex)> locker(m_EventMutex);

        m_BrokerActivationEvent.Signal();

        m_BrokerCompletionEvent.Wait();

        m_BrokerCompletionEvent.Clear();
    }

    NN_RESULT_SUCCESS;
}

::nn::Result WindowsGenericPadBrokerTask::DeactivateWindowsGenericPadBroker(
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pWindowsGenericPadBroker);

    {
        ::std::lock_guard<decltype(m_EventMutex)> locker(m_EventMutex);

        m_BrokerDeactivationEvent.Signal();

        m_BrokerCompletionEvent.Wait();

        m_BrokerCompletionEvent.Clear();
    }

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

void WindowsGenericPadBrokerTask::LinkTimerEvent(
    ::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pMultiWait);

    m_BrokerActivationEvent.Link(pMultiWait);

    m_BrokerDeactivationEvent.Link(pMultiWait);

    m_BrokerTimerEvent.Link(pMultiWait);
}

void WindowsGenericPadBrokerTask::UnlinkTimerEvent() NN_NOEXCEPT
{
    m_BrokerTimerEvent.Unlink();

    m_BrokerDeactivationEvent.Unlink();

    m_BrokerActivationEvent.Unlink();
}

void WindowsGenericPadBrokerTask::DoSampling(
    const ::nn::os::MultiWaitHolderType* waitId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pWindowsGenericPadBroker);

    if (waitId == m_BrokerActivationEvent.GetWaitId())
    {
        m_BrokerActivationEvent.Clear();

        // ブローカのアクティブ化を実行
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            m_pWindowsGenericPadBroker->Activate());

        m_BrokerCompletionEvent.Signal();

        return;
    }

    if (waitId == m_BrokerDeactivationEvent.GetWaitId())
    {
        m_BrokerDeactivationEvent.Clear();

        // ブローカの非アクティブ化を実行
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            m_pWindowsGenericPadBroker->Deactivate());

        m_BrokerCompletionEvent.Signal();

        return;
    }

    if (waitId == m_BrokerTimerEvent.GetWaitId())
    {
        m_BrokerTimerEvent.Clear();

        // ブローカの定期処理を実行
        m_pWindowsGenericPadBroker->Sample();

        return;
    }
}

ResourceManager::ResourceManager() NN_NOEXCEPT
{
    // WindowsGenericPadBroker タスクを初期化
    this->InitializeWindowsGenericPadBrokerTask();

    // 共有タスクを初期化
    this->InitializeSharedTask();

    // DebugPad タスクを初期化
    this->InitializeDebugPadTask();

    m_SixAxisSensorAppletSettingManager.SetAppletResourceManager(
        &m_AppletResourceManager, &m_AppletResourceMutex);

    // AbstractedGenericPad を初期化
    InitializeAbstractedPad();
    // Xpad を初期化
    InitializeXpad();
    // Npad を初期化
    InitializeNpad();
    // UniquePad を初期化
    InitializeUniquePad();
    // AbstractedPadManager を初期化
    InitializeAbstractedPadManager();

    // Xcd タスクを初期化
    m_XcdTask.SetThreadPriority(NN_SYSTEM_THREAD_PRIORITY(hid, XcdDriver));
    m_XcdTask.SetThreadNamePointer(NN_SYSTEM_THREAD_NAME(hid, XcdDriver));
    m_XcdTask.SetThreadStack(s_XcdTaskThreadStack, sizeof(s_XcdTaskThreadStack));
    m_XcdTask.SetXcdDriver(&m_XcdDriver);

    // XCD をアクティブ化 (暫定的にResourceManagerのコンストラクタ内部で実施)
    m_XcdTask.ActivateXcdDriver();

    // AbstractedXcdManager をアクティブ化
    m_SharedTask.ActivateAbstractedXcdManager();

    // Npad をアクティブ化
    m_NpadResourceManager.Activate();

    // UniquePad をアクティブ化
    m_UniquePadResourceManager.Activate();

    // Pad 共通のスレッドをアクティブ化
    m_SharedTask.ActivatePadCommonTimer();
}

ResourceManager::~ResourceManager() NN_NOEXCEPT
{
    // Pad 共通のスレッドを非アクティブ化
    m_SharedTask.DeactivatePadCommonTimer();

    // UniquePad を非アクティブ化
    m_UniquePadResourceManager.Deactivate();

    // Npad を非アクティブ化
    m_NpadResourceManager.Deactivate();

    // AbstractedXcdManager を非アクティブ化
    m_SharedTask.DeactivateAbstractedXcdManager();

    // XCD を終了 (暫定的にResourceManagerのデストラクタ内部で実施)
    m_XcdTask.DeactivateXcdDriver();
}

DebugPadTask& ResourceManager::GetDebugPadTask() NN_NOEXCEPT
{
    return m_DebugPadTask;
}

SharedTask& ResourceManager::GetGestureTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

SharedTask& ResourceManager::GetKeyboardTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

SharedTask& ResourceManager::GetMouseTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

SharedTask& ResourceManager::GetTouchScreenTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

SharedTask& ResourceManager::GetHomeButtonTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

SharedTask& ResourceManager::GetSleepButtonTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

SharedTask& ResourceManager::GetCaptureButtonTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

SharedTask& ResourceManager::GetConsoleSixAxisSensorTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

XpadTask& ResourceManager::GetXpadTask(int playerNumber) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(playerNumber, 0, XpadPlayerCountMax);
    return m_XpadTasks[playerNumber];
}

SharedTask& ResourceManager::GetInputDetectorTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

SharedTask& ResourceManager::GetAbstractedAutoPilotPadTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

SharedTask& ResourceManager::GetAbstractedPadXcdTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

SharedTask& ResourceManager::GetPlayReportTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

SharedTask& ResourceManager::GetRegisteredDeviceTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

SharedTask& ResourceManager::GetAudioControlTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

::nn::Result ResourceManager::AcquireJoyDetachOnBluetoothOffEventHandle(
    ::nn::os::NativeHandle* pOutValue,
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_RESULT_DO(
        m_InterruptSceneNotifier
            .AcquireJoyDetachOnBluetoothOffEventHandle(pOutValue, aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result ResourceManager::ActivatePadCommonTimer() NN_NOEXCEPT
{
    NN_RESULT_DO(m_SharedTask.ActivatePadCommonTimer());
    NN_RESULT_SUCCESS;
}

::nn::Result ResourceManager::DeactivatePadCommonTimer() NN_NOEXCEPT
{
    NN_RESULT_DO(m_SharedTask.DeactivatePadCommonTimer());
    NN_RESULT_SUCCESS;
}

NpadResourceManager& ResourceManager::GetNpadResourceManager() NN_NOEXCEPT
{
    return m_NpadResourceManager;
}

UniquePadResourceManager& ResourceManager::GetUniquePadResourceManager() NN_NOEXCEPT
{
    return m_UniquePadResourceManager;
}

AbstractedPadManager& ResourceManager::GetAbstractedPadManager() NN_NOEXCEPT
{
    return m_AbstractedPadManager;
}

PalmaResourceManager& ResourceManager::GetPalmaResourceManager() NN_NOEXCEPT
{
    return m_PalmaResourceManager;
}

void ResourceManager::SetAppletResourceUserId(
    ::nn::applet::AppletResourceUserId value) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_AppletResourceMutex)
                      > locker(m_AppletResourceMutex);
    m_AppletResourceManager.SetAppletResourceUserId(value);
    m_UniquePadResourceManager.NotifyAppletResourceUserId(value);
}

void ResourceManager::EnableHandheldHids() NN_NOEXCEPT
{
    m_HandheldManager.SetHandheldEnabled(true);
}

void ResourceManager::DisableHandheldHids() NN_NOEXCEPT
{
    m_HandheldManager.SetHandheldEnabled(false);
}

::nn::Result ResourceManager::RegisterAppletResourceUserId(
    ::nn::applet::AppletResourceUserId aruid,
    bool enablesInput) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_AppletResourceMutex)
                      > locker(m_AppletResourceMutex);
    NN_RESULT_DO(
        m_AppletResourceManager
            .RegisterAppletResourceUserId(aruid, enablesInput));
    NN_RESULT_SUCCESS;
}

void ResourceManager::UnregisterAppletResourceUserId(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_AppletResourceMutex)
                      > locker(m_AppletResourceMutex);
    m_AppletResourceManager.UnregisterAppletResourceUserId(aruid);
}

::nn::Result ResourceManager::SetAppletResourceEntry(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_AppletResourceMutex)
                      > locker(m_AppletResourceMutex);
    NN_RESULT_DO(m_AppletResourceManager.SetAppletResourceEntry(aruid));
    NN_RESULT_SUCCESS;
}

void ResourceManager::ResetAppletResourceEntry(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_AppletResourceMutex)
                      > locker(m_AppletResourceMutex);
    m_AppletResourceManager.ResetAppletResourceEntry(aruid);
}

void ResourceManager::EnableAppletToGetInput(
    ::nn::applet::AppletResourceUserId aruid,
    bool enablesInput) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_AppletResourceMutex)
                      > locker(m_AppletResourceMutex);
    m_AppletResourceManager.EnableAppletResourceInput(aruid, enablesInput);
}

void ResourceManager::EnableAppletToGetSixAxisSensor(
    ::nn::applet::AppletResourceUserId aruid,
    bool enablesInput) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_AppletResourceMutex)
                      > locker(m_AppletResourceMutex);
    m_AppletResourceManager
        .EnableAppletResourceSixAxisSensor(aruid, enablesInput);
}


::nn::Result ResourceManager::GetAppletResourceSharedMemoryHandle(
    ::nn::os::NativeHandle* pOutValue,
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    ::std::lock_guard<decltype(m_AppletResourceMutex)
                      > locker(m_AppletResourceMutex);
    NN_RESULT_DO(
        m_AppletResourceManager
            .GetAppletResourceSharedMemoryHandle(pOutValue, aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result ResourceManager::ActivateSharedAppletResource() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_AppletResourceMutex)
                      > locker(m_AppletResourceMutex);
    NN_RESULT_DO(m_AppletResourceManager.ActivateSharedAppletResource());
    NN_RESULT_SUCCESS;
}

::nn::Result ResourceManager::DeactivateSharedAppletResource() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_AppletResourceMutex)
                      > locker(m_AppletResourceMutex);
    NN_RESULT_DO(m_AppletResourceManager.DeactivateSharedAppletResource());
    NN_RESULT_SUCCESS;
}

const SixAxisSensorProcessor& ResourceManager::GetSixAxisSensorProcessor(
    const SixAxisSensorHandle& handle) const NN_NOEXCEPT
{
    auto pProcessor = m_NpadResourceManager.GetSixAxisSensor(handle);
    NN_SDK_REQUIRES_NOT_NULL(pProcessor);
    return *pProcessor;
}

SixAxisSensorAppletSettingManager&
ResourceManager::GetSixAxisSensorAppletSettingManager() NN_NOEXCEPT
{
    return m_SixAxisSensorAppletSettingManager;
}

ConsoleSixAxisSensorAppletSettingManager&
ResourceManager::GetConsoleSixAxisSensorAppletSettingManager() NN_NOEXCEPT
{
    return m_ConsoleSixAxisSensorAppletSettingManager;
}

SharedTask& ResourceManager::GetFirmwareUpdaterTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

XcdDriver& ResourceManager::GetXcdDriver() NN_NOEXCEPT
{
    return m_XcdDriver;
}

::nn::Result ResourceManager::AssertValidBasicXpadPlayerNumber(
    int playerNumber) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        0 <= playerNumber && playerNumber < XpadPlayerCountMax,
        ResultBasicXpadPlayerNumberOutOfRange());
    NN_RESULT_SUCCESS;
}

::nn::Result ResourceManager::AssertValidJoyXpadPlayerNumber(
    int playerNumber) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        0 <= playerNumber && playerNumber < XpadPlayerCountMax,
        ResultJoyXpadPlayerNumberOutOfRange());
    NN_RESULT_SUCCESS;
}

::nn::Result ResourceManager::AcquireXpadIdEventHandle(
    ::nn::os::NativeHandle* outHandle,
    uint64_t clientId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outHandle);
    ::std::lock_guard<decltype(m_XpadIdManagerMutex)
                      > locker(m_XpadIdManagerMutex);
    NN_RESULT_DO(m_XpadIdManager.AcquireXpadIdEventHandle(outHandle, clientId));
    // ハンドルの取得直後はシグナル状態であることを保証
    m_XpadIdManager.SignalXpadIdSystemEvent();
    NN_RESULT_SUCCESS;
}

::nn::Result ResourceManager::ReleaseXpadIdEventHandle(
    uint64_t clientId) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_XpadIdManagerMutex)
                      > locker(m_XpadIdManagerMutex);
    NN_RESULT_THROW(m_XpadIdManager.ReleaseXpadIdEventHandle(clientId));
}

::nn::Result ResourceManager::GetXpadIds(
    int* outCount, BasicXpadId* outXpadIds, int count) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_XpadIdManagerMutex)
                      > locker(m_XpadIdManagerMutex);
    NN_RESULT_THROW(
        m_XpadIdManager.GetBasicXpadIds(outCount, outXpadIds, count));
}

::nn::Result ResourceManager::GetXpadIds(
    int* outCount, JoyXpadId* outXpadIds, int count) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_XpadIdManagerMutex)
                      > locker(m_XpadIdManagerMutex);
    NN_RESULT_THROW(
        m_XpadIdManager.GetJoyXpadIds(outCount, outXpadIds, count));
}

DebugPadMappingManager& ResourceManager::GetDebugPadMappingManager() NN_NOEXCEPT
{
    return m_DebugPadMappingManager;
}

XpadMappingManager& ResourceManager::GetXpadMappingManager(
    int playerNumber) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(playerNumber, 0, XpadPlayerCountMax);
    return m_XpadMappingManagers[playerNumber];
}

::nn::Result ResourceManager::ActivateAbstractedGenericPad() NN_NOEXCEPT
{
    for (auto& task : m_AbstractedGenericPadTask)
    {
        NN_RESULT_DO(task.ActivateAbstractedGenericPad());
    }
    NN_RESULT_SUCCESS;
}

::nn::Result ResourceManager::DeactivateAbstractedGenericPad() NN_NOEXCEPT
{
    for (auto& task : m_AbstractedGenericPadTask)
    {
        NN_RESULT_DO(task.DeactivateAbstractedGenericPad());
    }
    NN_RESULT_SUCCESS;
}

void ResourceManager::InitializeSharedTask() NN_NOEXCEPT
{
    m_SharedTask.SetThreadStack(
        s_SharedTaskThreadStack, sizeof(s_SharedTaskThreadStack));
    m_SharedTask.SetThreadPriority(
        NN_SYSTEM_THREAD_PRIORITY(hid, CommonSampler));
    m_SharedTask.SetThreadNamePointer(
        NN_SYSTEM_THREAD_NAME(hid, CommonSampler));

    m_KeyboardManager.SetDriver(&m_KeyboardDriver);
    m_KeyboardManager.SetAppletResourceManager(
        &m_AppletResourceManager, &m_AppletResourceMutex);
    m_KeyboardManager.SetInputDetectorManager(
        &m_InputDetectorManager, &m_InputDetectorMutex);
    m_SharedTask.SetKeyboardManager(&m_KeyboardManager);

    m_MouseManager.SetDriver(&m_MouseDriver);
    m_MouseManager.SetAppletResourceManager(
        &m_AppletResourceManager, &m_AppletResourceMutex);
    m_MouseManager.SetInputDetectorManager(
        &m_InputDetectorManager, &m_InputDetectorMutex);
    m_SharedTask.SetMouseManager(&m_MouseManager);

    m_TouchScreenManager.SetDriver(&m_TouchScreenDriver);
    m_TouchScreenManager.SetAppletResourceManager(
        &m_AppletResourceManager, &m_AppletResourceMutex);
    m_TouchScreenManager.SetHandheldManager(&m_HandheldManager);
    m_TouchScreenManager.SetInputDetectorManager(
        &m_InputDetectorManager, &m_InputDetectorMutex);
    m_SharedTask.SetTouchScreenManager(&m_TouchScreenManager);

    m_HomeButtonManager.SetAppletResourceManager(
        &m_AppletResourceManager, &m_AppletResourceMutex);
    m_HomeButtonManager.SetInputDetectorManager(
        &m_InputDetectorManager, &m_InputDetectorMutex);
    m_SharedTask.SetHomeButtonManager(&m_HomeButtonManager);

    m_SleepButtonManager.SetDriver(&m_SleepButtonDriver);
    m_SleepButtonManager.SetAppletResourceManager(
        &m_AppletResourceManager, &m_AppletResourceMutex);
    m_SleepButtonManager.SetInputDetectorManager(
        &m_InputDetectorManager, &m_InputDetectorMutex);
    m_SharedTask.SetSleepButtonManager(&m_SleepButtonManager);

    m_CaptureButtonManager.SetAppletResourceManager(
        &m_AppletResourceManager, &m_AppletResourceMutex);
    m_CaptureButtonManager.SetInputDetectorManager(
        &m_InputDetectorManager, &m_InputDetectorMutex);

    m_SharedTask.SetCaptureButtonManager(&m_CaptureButtonManager);

    m_SharedTask.SetPlayReportManager(&m_PlayReportManager);
    m_SharedTask.ActivatePlayReport();

    m_RegisteredDeviceManager.SetOvlnSenderManager(&m_OvlnSender);
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedPadXcd); i++)
    {
        m_RegisteredDeviceManager.AddIAbstractedPad(&m_AbstractedPadXcd[i]);
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedGenericPad); i++)
    {
        m_RegisteredDeviceManager.AddIAbstractedPad(&m_AbstractedGenericPad[i]);
    }
    m_SharedTask.SetRegisteredDeviceManager(&m_RegisteredDeviceManager);
    m_SharedTask.ActivateRegisteredDeviceManagement();

    m_ConsoleSixAxisSensorManager.SetDriver(&m_ConsoleSixAxisSensorDriver);
    m_ConsoleSixAxisSensorManager.SetAppletResourceManager(
        &m_AppletResourceManager, &m_AppletResourceMutex);
    m_ConsoleSixAxisSensorManager.SetSixAxisSensorAppletSettingManager(
        &m_SixAxisSensorAppletSettingManager);
    m_ConsoleSixAxisSensorManager.SetConsoleSixAxisSensorAppletSettingManager(
        &m_ConsoleSixAxisSensorAppletSettingManager);
    m_ConsoleSixAxisSensorManager.SetHandheldManager(&m_HandheldManager);
    m_ConsoleSixAxisSensorManager.SetInputDetectorManager(
        &m_InputDetectorManager, &m_InputDetectorMutex);
    m_ConsoleSixAxisSensorManager.SetConsoleSixAxisSensorProcessor(
        &m_ConsoleSixAxisSensorProcessor);
    m_ConsoleSixAxisSensorAppletSettingManager.SetAppletResourceManager(
        &m_AppletResourceManager, &m_AppletResourceMutex);

    m_SharedTask.SetConsoleSixAxisSensorManager(&m_ConsoleSixAxisSensorManager);

    m_InputDetectorManager.SetAppletResourceManager(
        &m_AppletResourceManager, &m_AppletResourceMutex);
    m_SharedTask.SetInputDetectorManager(&m_InputDetectorManager);

    m_AudioControlManager.SetDriver(&m_AudioControlDriver);
    m_SharedTask.SetAudioControlManager(&m_AudioControlManager);
}

void ResourceManager::InitializeDebugPadTask() NN_NOEXCEPT
{
    m_DebugPadTask.SetThreadStack(
        s_DebugPadTaskThreadStack, sizeof(s_DebugPadTaskThreadStack));
    m_DebugPadTask.SetThreadPriority(
        NN_SYSTEM_THREAD_PRIORITY(hid, DebugPadSampler));
    m_DebugPadTask.SetThreadNamePointer(
        NN_SYSTEM_THREAD_NAME(hid, DebugPadSampler));

    m_DebugPadDriver.SetMappingManager(&m_DebugPadMappingManager);
    m_DebugPadManager.SetDriver(&m_DebugPadDriver);
    m_DebugPadManager.SetAppletResourceManager(
        &m_AppletResourceManager, &m_AppletResourceMutex);
    m_DebugPadManager.SetInputDetectorManager(
        &m_InputDetectorManager, &m_InputDetectorMutex);
    m_DebugPadTask.SetDebugPadManager(&m_DebugPadManager);
    m_DebugPadTask.SetWindowsGenericPadBrokerTask(
        &m_WindowsGenericPadBrokerTask);
}

void ResourceManager::InitializeAbstractedPad() NN_NOEXCEPT
{
    for (int i = 0; i < AbstractedGenericPadCountMax; i++)
    {
        m_AbstractedGenericPad[i].SetHidShellPort(i);
        m_AbstractedGenericPad[i].SetMappingManager(&m_XpadMappingManagers[i]);
        m_AbstractedGenericPad[i].SetAbstractedPadIdPublisher(
            &m_AbstractedIdPublisher);
        m_AbstractedGenericPadTask[i].SetThreadStack(
            s_AbstractedGenericPadTaskThreadStack[i],
            sizeof(s_AbstractedGenericPadTaskThreadStack[i]));
        m_AbstractedGenericPadTask[i].SetWindowsGenericPadBrokerTask(
            &m_WindowsGenericPadBrokerTask);
        m_AbstractedGenericPadTask[i].SetThreadPriority(
        NN_SYSTEM_THREAD_PRIORITY(hid, NpadSampler));
        m_AbstractedGenericPadTask[i].SetThreadNamePointer(
            NN_SYSTEM_THREAD_NAME(hid, NpadSampler));
        m_AbstractedGenericPadTask[i].SetAbstractedGenericPad(
            &m_AbstractedGenericPad[i]);
    }

    m_AbstractedPadXcdManager.SetAbstractedPadXcds(
        m_AbstractedPadXcd, NN_ARRAY_SIZE(m_AbstractedPadXcd));
    m_AbstractedPadXcdManager.SetAbstractedPadIdPublisher(
        &m_AbstractedIdPublisher);
    m_SharedTask.SetAbstractedPadXcd(
        m_AbstractedPadXcd, NN_ARRAY_SIZE(m_AbstractedPadXcd));
    m_SharedTask.SetAbstractedPadXcdManager(&m_AbstractedPadXcdManager);

    m_AbstractedAutoPilotPadManager.SetAbstractedAutoPilotPads(
        m_AbstractedAutoPilotPad, NN_ARRAY_SIZE(m_AbstractedAutoPilotPad));
    for (auto& pad : m_AbstractedAutoPilotPad)
    {
        pad.SetAbstractedPadIdPublisher(&m_AbstractedIdPublisher);
    }
    m_SharedTask.SetAbstractedAutoPilotPadManager(&m_AbstractedAutoPilotPadManager);

    m_AbstractedAutoPilotVirtualPadManager.SetAbstractedAutoPilotVirtualPads(
        m_AbstractedAutoPilotVirtualPad,
        NN_ARRAY_SIZE(m_AbstractedAutoPilotVirtualPad));
    for (auto &pad : m_AbstractedAutoPilotVirtualPad)
    {
        pad.SetAbstractedPadIdPublisher(&m_AbstractedIdPublisher);
    }
    m_SharedTask.SetAbstractedAutoPilotVirtualPadManager(
        &m_AbstractedAutoPilotVirtualPadManager);

    m_AbstractedPadHidShellManager.SetAbstractedPadManager(
        &m_AbstractedPadManager);
    m_AbstractedPadHidShellManager.SetAbstractedAutoPilotVirtualPadManager(
        &m_AbstractedAutoPilotVirtualPadManager);
    m_SharedTask.SetAbstractedPadHidShellManager(
        &m_AbstractedPadHidShellManager);
}

void ResourceManager::InitializeXpad() NN_NOEXCEPT
{

    // Xpad タスクを初期化
    for (int i = 0; i < XpadPlayerCountMax; ++i)
    {
        m_XpadDrivers[i].SetHidShellPort(i);
        m_XpadDrivers[i].SetMappingManager(&m_XpadMappingManagers[i]);
        m_XpadManagers[i].SetDriver(&m_XpadDrivers[i]);
        m_XpadManagers[i].SetAppletResourceManager(
            &m_AppletResourceManager, &m_AppletResourceMutex);
        m_XpadManagers[i].SetXpadId(MakeXpadId<BasicXpadId>(i));
        m_XpadTasks[i].SetThreadStack(
            s_XpadTaskThreadStacks[i], sizeof(s_XpadTaskThreadStacks[i]));
        m_XpadTasks[i].SetThreadPriority(
            NN_SYSTEM_THREAD_PRIORITY(hid, XpadSampler));
        m_XpadTasks[i].SetThreadNamePointer(
            NN_SYSTEM_THREAD_NAME(hid, XpadSampler));
        m_XpadTasks[i].SetXpadManager(&m_XpadManagers[i]);
        m_XpadTasks[i].SetWindowsGenericPadBrokerTask(
            &m_WindowsGenericPadBrokerTask);
    }

    // 基本的な構成を持つ Xpad の解決された ID の初期値
    const BasicXpadId BasicXpadIds[] =
    {
        MakeXpadId<BasicXpadId>(0),
        MakeXpadId<BasicXpadId>(1),
        MakeXpadId<BasicXpadId>(2),
        MakeXpadId<BasicXpadId>(3)
    };

    // JoyXpad の解決された ID の初期値
    const JoyXpadId JoyXpadIds[] =
    {
        MakeXpadId<JoyXpadId>(0),
        MakeXpadId<JoyXpadId>(1),
        MakeXpadId<JoyXpadId>(2),
        MakeXpadId<JoyXpadId>(3)
    };

    // Xpad ID マネージャを初期化
    m_XpadIdManager.SetBasicXpadIds(BasicXpadIds, NN_ARRAY_SIZE(BasicXpadIds));
    m_XpadIdManager.SetJoyXpadIds(JoyXpadIds, NN_ARRAY_SIZE(JoyXpadIds));
}

void ResourceManager::InitializeNpad() NN_NOEXCEPT
{
    m_NpadResourceManager.Initialize(&m_CaptureButtonManager,
        &m_HomeButtonManager,
        &m_OvlnSender,
        &m_PlayReportManager,
        &m_AppletResourceManager,
        &m_AppletResourceMutex,
        &m_SixAxisSensorAppletSettingManager,
        &m_ConsoleSixAxisSensorManager,
        &m_HandheldManager,
        &m_InputDetectorManager,
        &m_InputDetectorMutex,
        &m_UniquePadResourceManager,
        &m_InterruptSceneNotifier,
        &m_GcAdapterDriver,
        &m_PalmaResourceManager);
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedGenericPad); i++)
    {
        m_NpadResourceManager.AddIAbstractedPad(&m_AbstractedGenericPad[i]);
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedPadXcd); i++)
    {
        m_NpadResourceManager.AddIAbstractedPad(&m_AbstractedPadXcd[i]);
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedAutoPilotPad); i++)
    {
        m_NpadResourceManager.AddIAbstractedPad(&m_AbstractedAutoPilotPad[i]);
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedAutoPilotVirtualPad); i++)
    {
        m_NpadResourceManager.AddIAbstractedPad(&m_AbstractedAutoPilotVirtualPad[i]);
    }
    m_SharedTask.SetNpadResourceManager(&m_NpadResourceManager);
}

void ResourceManager::InitializeUniquePad() NN_NOEXCEPT
{
    m_UniquePadResourceManager.Initialize(&m_HandheldManager,
                                          &m_InterruptSceneNotifier);
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedGenericPad); i++)
    {
        m_UniquePadResourceManager.AddIAbstractedPad(&m_AbstractedGenericPad[i]);
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedPadXcd); i++)
    {
        m_UniquePadResourceManager.AddIAbstractedPad(&m_AbstractedPadXcd[i]);
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedAutoPilotPad); i++)
    {
        m_UniquePadResourceManager.AddIAbstractedPad(&m_AbstractedAutoPilotPad[i]);
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedAutoPilotVirtualPad); i++)
    {
        m_UniquePadResourceManager.AddIAbstractedPad(&m_AbstractedAutoPilotVirtualPad[i]);
    }
    m_SharedTask.SetUniquePadResourceManager(&m_UniquePadResourceManager);
}

void ResourceManager::InitializeWindowsGenericPadBrokerTask() NN_NOEXCEPT
{
    m_WindowsGenericPadBrokerTask.SetThreadStack(
        s_WindowsGenericPadBrokerTaskThreadStack,
        sizeof(s_WindowsGenericPadBrokerTaskThreadStack));
    m_WindowsGenericPadBrokerTask.SetThreadPriority(
        NN_SYSTEM_THREAD_PRIORITY(hid, XpadSampler));
    m_WindowsGenericPadBrokerTask.SetThreadNamePointer(
        NN_SYSTEM_THREAD_NAME(hid, XpadSampler));

    m_WindowsGenericPadBrokerTask.SetWindowsGenericPadBroker(
        &m_WindowsGenericPadBroker);
}

void ResourceManager::InitializeAbstractedPadManager() NN_NOEXCEPT
{
    m_AbstractedPadManager.Initialize();
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedGenericPad); i++)
    {
        m_AbstractedPadManager.AddIAbstractedPad(&m_AbstractedGenericPad[i]);
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedPadXcd); i++)
    {
        m_AbstractedPadManager.AddIAbstractedPad(&m_AbstractedPadXcd[i]);
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedAutoPilotPad); i++)
    {
        m_AbstractedPadManager.AddIAbstractedPad(&m_AbstractedAutoPilotPad[i]);
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedAutoPilotVirtualPad); i++)
    {
        m_AbstractedPadManager.AddIAbstractedPad(&m_AbstractedAutoPilotVirtualPad[i]);
    }
}

NN_ALIGNAS(0x1000) char ResourceManager::s_SharedTaskThreadStack[0x1000];

NN_ALIGNAS(0x1000) char ResourceManager::s_DebugPadTaskThreadStack[0x1000];

NN_ALIGNAS(0x1000) char ResourceManager::s_XpadTaskThreadStacks[
    XpadPlayerCountMax][0x1000];

NN_ALIGNAS(0x1000) char ResourceManager::s_XcdTaskThreadStack[0x2000];

NN_ALIGNAS(0x1000) char ResourceManager::
    s_WindowsGenericPadBrokerTaskThreadStack[0x1000];

NN_ALIGNAS(0x1000) char ResourceManager::
    s_AbstractedGenericPadTaskThreadStack[AbstractedGenericPadCountMax][0x1000];

const char* const ResourceManager::s_UniquePadSamplerThreadName = "UniquePadSampler";

ResourceManager& GetResourceManager() NN_NOEXCEPT
{
    return StaticObject<ResourceManager>::Get();
}

}}} // namespace nn::hid::detail
