﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_ConsoleSixAxisSensor.h>
#include <nn/hid/hid_SevenSixAxisSensor.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nn/hid/hid_Xpad.h>
#include <nn/hid/debug/hid_CaptureButton.h>
#include <nn/hid/debug/hid_ConsoleSixAxisSensor.h>
#include <nn/hid/debug/hid_DebugPad.h>
#include <nn/hid/debug/hid_FirmwareUpdate.h>
#include <nn/hid/debug/hid_HomeButton.h>
#include <nn/hid/debug/hid_Keyboard.h>
#include <nn/hid/debug/hid_Mouse.h>
#include <nn/hid/debug/hid_SleepButton.h>
#include <nn/hid/debug/hid_TouchScreen.h>
#include <nn/hid/debug/hid_Xpad.h>
#include <nn/hid/system/hid_Common.h>
#include <nn/hid/system/hid_InputDetection.h>
#include <nn/hid/system/hid_Keyboard.h>
#include <nn/hid/system/hid_UniquePad.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_LightEvent.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_Thread.h>

#include "hid_AbstractedAutoPilotPad.h"
#include "hid_AbstractedAutoPilotPadManager.h"
#include "hid_AbstractedAutoPilotVirtualPad.h"
#include "hid_AbstractedAutoPilotVirtualPadManager.h"
#include "hid_AbstractedFiftyPad.h"
#include "hid_AbstractedGcAdapterPad.h"
#include "hid_AbstractedGcAdapterPadManager.h"
#include "hid_AbstractedPadManager.h"
#include "hid_AbstractedPadUsb.h"
#include "hid_AbstractedPadUsbDriver.h"
#include "hid_AbstractedPadXcd.h"
#include "hid_AbstractedPadXcdManager.h"
#include "hid_AbstractedPalmaPad.h"
#include "hid_AbstractedPalmaPadManager.h"
#include "hid_ActivationCount.h"
#include "hid_AppletResourceManager.h"
#include "hid_AudioControlManager.h"
#include "hid_CaptureButtonManager.h"
#include "hid_ConsoleSixAxisSensorAppletSettingManager.h"
#include "hid_ConsoleSixAxisSensorManager.h"
#include "hid_ConsoleSixAxisSensorProcessor.h"
#include "hid_DebugPadManager.h"
#include "hid_GcAdapterDriver.h"
#include "hid_HandheldManager.h"
#include "hid_HomeButtonManager.h"
#include "hid_IConsoleSixAxisSensorDriver.h"
#include "hid_IDebugPadDriver.h"
#include "hid_IKeyboardDriver.h"
#include "hid_IMouseDriver.h"
#include "hid_InputDetectorManager.h"
#include "hid_InterruptSceneNotifier.h"
#include "hid_IrSensorXcdDriver.h"
#include "hid_ITouchScreenDriver.h"
#include "hid_IXpadDriver.h"
#include "hid_KeyboardManager.h"
#include "hid_MouseManager.h"
#include "hid_MultiWaitEvent.h"
#include "hid_NpadResourceManager.h"
#include "hid_OvlnSenderManager.h"
#include "hid_PalmaResourceManager.h"
#include "hid_PlayReportManager.h"
#include "hid_RailManager.h"
#include "hid_RegisteredDeviceManager.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_SixAxisSensorAppletSettingManager.h"
#include "hid_SixAxisSensorSanjiroFilter.h"
#include "hid_SixAxisSensorXcdDriver.h"
#include "hid_SleepButtonManager.h"
#include "hid_Task.h"
#include "hid_TouchScreenManager.h"
#include "hid_UniquePadResourceManager.h"
#include "hid_VibratorXcdDriver.h"
#include "hid_VibratorGcDriver.h"
#include "hid_XpadIdManager.h"
#include "hid_XpadManager.h"
#include "hid_XcdDriver.h"
#include "hid_SynchronizedTimer.h"

#include "hid_AhidManager-os.horizon.h"
#include "hid_DebugPadDriver-os.horizon.h"
#include "hid_KeyboardDriver-os.horizon.h"
#include "hid_MouseDriver-os.horizon.h"
#include "hid_AudioControlDriver.h"
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    #include "hid_SleepButtonDriver.horizon.bcp.h"
    #include "hid_TouchScreenDriver-os.horizon.ftm.h"
    #include "hid_ConsoleSixAxisSensorDriver.horizon.lsm6ds3.h"
#else
    #include "hid_SleepButtonDriver.horizon.stub.h"
    #include "hid_TouchScreenDriver-os.horizon.stub.h"
    #include "hid_ConsoleSixAxisSensorDriver.horizon.stub.h"
#endif
#include "hid_XpadDriver.horizon.h"

namespace nn { namespace hid { namespace detail {

//!< 共有タスクを扱うクラスです。
class SharedTask final : public Task
{
private:
    //!< DebugPad 操作排他用のミューテックス
    ::nn::os::SdkMutex m_DebugPadMutex;

    //!< Keyboard 操作排他用のミューテックス
    ::nn::os::SdkMutex m_KeyboardMutex;

    //!< Mouse 操作排他用のミューテックス
    ::nn::os::SdkMutex m_MouseMutex;

    //!< Xpad 操作排他用のミューテックス
    ::nn::os::SdkMutex m_XpadMutex;

    //!< InputDetector 操作排他用のミューテックス
    ::nn::os::SdkMutex m_InputDetectorMutex;

    //!< ホームボタン操作排他用のミューテックス
    ::nn::os::SdkMutex m_HomeButtonMutex;

    //!< スリープボタン操作排他用のミューテックス
    ::nn::os::SdkMutex m_SleepButtonMutex;

    //!< 撮影ボタン操作排他用のミューテックス
    ::nn::os::SdkMutex m_CaptureButtonMutex;

    //!< RegisteredDevice 操作排他用のミューテックス
    ::nn::os::SdkMutex m_RegisteredDeviceMutex;

    //!< レール操作排他用のミューテックス
    ::nn::os::SdkMutex m_RailMutex;

    //!< オーディオコントロール操作排他用のミューテックス
    ::nn::os::SdkMutex m_AudioControlMutex;

    //!< DebugPad のタイマーイベント
    SynchronizedTimer m_DebugPadTimerEvent;

    //!< Keyboard のタイマーイベント
    SynchronizedTimer m_KeyboardTimerEvent;

    //!< Mouse のタイマーイベント
    SynchronizedTimer m_MouseTimerEvent;

    //!< Xpad のタイマーイベント
    SynchronizedTimer m_XpadTimerEvent;

    //!< Npad / UniquePad 共通のタイマーイベント
    SynchronizedTimer m_PadCommonTimerEvent;

    //!< Pad 共通のタイマーイベントがアクティブ化された回数
    ActivationCount m_PadCommonTimerEventActivationCount;

    //!< Pad 共通のタイマーイベントの動作間隔
    static const nn::TimeSpan PadCommonTimerInterval;

    //!< ホームボタンのタイマーイベント
    SynchronizedTimer m_HomeButtonTimerEvent;

    //!< スリープボタンのタイマーイベント
    SynchronizedTimer m_SleepButtonTimerEvent;

    //!< 撮影ボタンのタイマーイベント
    SynchronizedTimer m_CaptureButtonTimerEvent;

    //!< プレイレポートのタイマーイベント
    SynchronizedTimer m_PlayReportTimerEvent;

    //!< Xcd の接続状態変化検知用のシステムイベント
    MultiWaitSystemEvent m_XcdDeviceUpdateEvent;

    //!< Xcd の有線ペアリング完了通知用のシステムイベント
    MultiWaitSystemEvent m_PairingCompleteEvent;

    //!< Xcd のコントローラーへの接続タイムアウト通知用のシステムイベント
    MultiWaitSystemEvent m_ConnectionTriggerTimeoutEvent;

    //!< Xcd のレール接続状態変化の通知を受けるためのシステムイベント
    MultiWaitSystemEvent m_RailUpdateEvent;

    //!< Bluetooth ファームウェア更新の完了検知用のシステムイベント
    MultiWaitSystemEvent m_BluetoothFirmwareUpdateSystemEvent;

    //!< Mcu ファームウェア更新の完了検知用のシステムイベント
    MultiWaitSystemEvent m_McuFirmwareUpdateSystemEvent;

    //!< ファームウェア更新中のサンプル受信検知用のシステムイベント
    MultiWaitSystemEvent m_FirmwareUpdateNewSampleSystemEvent;

    //!< ファームウェア更新中のリセットタイムアウト検知用のタイマーイベント
    MultiWaitTimerEvent m_FirmwareUpdateResetTimeoutEvent;

    //!< オーディオコントロールのタイマーイベント
    SynchronizedTimer m_AudioControlTimerEvent;

    //!< DebugPad マネージャ
    DebugPadManager* m_pDebugPadManager;

    //!< Keyboard マネージャ
    KeyboardManager* m_pKeyboardManager;

    //!< Mouse マネージャ
    MouseManager* m_pMouseManager;

    //!< Xpad マネージャ
    XpadManager* m_XpadManagers;

    //!< Xpad マネージャの数
    int m_XpadManagerCount;

    //!< InputDetector マネージャ
    InputDetectorManager* m_pInputDetectorManager;

    //!< ホームボタンマネージャ
    HomeButtonManager* m_pHomeButtonManager;

    //!< スリープボタンマネージャ
    SleepButtonManager* m_pSleepButtonManager;

    //!< 撮影ボタンマネージャ
    CaptureButtonManager* m_pCaptureButtonManager;

    //!< プレイレポートマネージャー
    PlayReportManager* m_pPlayReportManager;

    //!< 登録情報マネージャー
    RegisteredDeviceManager* m_pRegisteredDeviceManager;

    //!< Npad リソースマネージャー
    NpadResourceManager* m_pNpadResourceManager;

    //!< UniquePad リソースマネージャー
    UniquePadResourceManager* m_pUniquePadResourceManager;

    //!< Palma リソースマネージャー
    PalmaResourceManager* m_pPalmaResourceManager;

    //!< Xcd の抽象化インタフェース
    AbstractedPadXcd* m_pAbstractedPadXcd;

    //!< Xcd の抽象化 インタフェースの数
    int m_AbstractedPadXcdCount;

    //!< Xcd の割り当てマネージャー
    AbstractedPadXcdManager* m_pAbstractedPadXcdManager;

    //!< Palma の抽象化インタフェース
    AbstractedPalmaPad* m_pAbstractedPalmaPad;

    //!< Palma の抽象化 インタフェースの数
    int m_AbstractedPalmaPadCount;

    //!< Palma の割り当てマネージャー
    AbstractedPalmaPadManager* m_pAbstractedPalmaPadManager;

    //!< 自動入力の抽象化 Pad を管理するマネージャー
    AbstractedAutoPilotPadManager* m_pAbstractedAutoPilotPadManager;

    //!< 自動入力の抽象化 Pad を管理するマネージャー
    AbstractedAutoPilotVirtualPadManager* m_pAbstractedAutoPilotVirtualPadManager;

    //!< Gc コントローラーアダプターを管理するマネージャー
    AbstractedGcAdapterPadManager* m_pAbstractedGcAdapterPadManager;

    //!< Fifty の抽象化インタフェース
    AbstractedFiftyPad* m_pAbstractedFiftyPad;

    //!< レールマネージャー
    RailManager* m_pRailManager;

    //!< オーディオコントロールマネージャ
    AudioControlManager* m_pAudioControlManager;

public:
    SharedTask() NN_NOEXCEPT;

    virtual ~SharedTask() NN_NOEXCEPT NN_OVERRIDE;

    //!< Npad リソースマネージャを設定します。
    void SetNpadResourceManager(NpadResourceManager* pManager) NN_NOEXCEPT;

    //!< UniquePad リソースマネージャを設定します。
    void SetUniquePadResourceManager(UniquePadResourceManager* pManager) NN_NOEXCEPT;

    //!< Palma リソースマネージャを設定します。
    void SetPalmaResourceManager(PalmaResourceManager* pManager) NN_NOEXCEPT;

    //!< DebugPad マネージャを設定します。
    void SetDebugPadManager(DebugPadManager* pManager) NN_NOEXCEPT;

    //!< Keyboard マネージャを設定します。
    void SetKeyboardManager(KeyboardManager* pManager) NN_NOEXCEPT;

    //!< Mouse マネージャを設定します。
    void SetMouseManager(MouseManager* pManager) NN_NOEXCEPT;

    //!< Xpad マネージャを設定します。
    void SetXpadManagers(XpadManager* managers, int count) NN_NOEXCEPT;

    //!< InputDetector マネージャを設定します。
    void SetInputDetectorManager(InputDetectorManager* pManager) NN_NOEXCEPT;

    //!< ホームボタンマネージャを設定します。
    void SetHomeButtonManager(HomeButtonManager* pManager) NN_NOEXCEPT;

    //!< スリープボタンマネージャを設定します。
    void SetSleepButtonManager(SleepButtonManager* pManager) NN_NOEXCEPT;

    //!< 撮影ボタンマネージャを設定します。
    void SetCaptureButtonManager(CaptureButtonManager* pManager) NN_NOEXCEPT;

    //!< プレイレポートマネージャーを設定します。
    void SetPlayReportManager(PlayReportManager* pManager) NN_NOEXCEPT;

    //!< 登録情報マネージャーを設定します。
    void SetRegisteredDeviceManager(RegisteredDeviceManager* pManager) NN_NOEXCEPT;

    //!< Xcd の抽象化インタフェースを設定します。
    void SetAbstractedPadXcd(AbstractedPadXcd* pXcd, int count) NN_NOEXCEPT;

    //!< Xcd の割り当てマネージャーを設定します。
    void SetAbstractedPadXcdManager(AbstractedPadXcdManager* pManager) NN_NOEXCEPT;

    //!< Palma の抽象化インタフェースを設定します。
    void SetAbstractedPalmaPad(AbstractedPalmaPad* pPalma, int count) NN_NOEXCEPT;

    //!< Palma の割り当てマネージャーを設定します。
    void SetAbstractedPalmaPadManager(AbstractedPalmaPadManager* pManager) NN_NOEXCEPT;

    //!< 自動入力の抽象化 Pad を管理するマネージャーを設定します。
    void SetAbstractedAutoPilotPadManager(AbstractedAutoPilotPadManager* pManager) NN_NOEXCEPT;

    //!< 自動入力の抽象化 Pad を管理するマネージャーを設定します。
    void SetAbstractedAutoPilotVirtualPadManager(
        AbstractedAutoPilotVirtualPadManager* pManager) NN_NOEXCEPT;

    //!< Gc コントローラーアダプターの抽象化 Pad を管理するマネージャーを設定します。
    void SetAbstractedGcAdapterPadManager(AbstractedGcAdapterPadManager* pManager) NN_NOEXCEPT;

    //!< Fifty の抽象化インタフェースを設定します。
    void SetAbstractedFiftyPad(AbstractedFiftyPad* pFifty) NN_NOEXCEPT;

    //!< レールマネージャーを設定します。
    void SetRailManager(RailManager* pManager) NN_NOEXCEPT;

    //!< オーディオコントロールマネージャーを設定します。
    void SetAudioControlManager(AudioControlManager* pManager) NN_NOEXCEPT;

    //!< DebugPad をアクティブ化します。
    ::nn::Result ActivateDebugPad() NN_NOEXCEPT;

    //!< DebugPad を非アクティブ化します。
    ::nn::Result DeactivateDebugPad() NN_NOEXCEPT;

    //!< DebugPad のアプレットリソースの状態を保証します。
    ::nn::Result EnsureDebugPadAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< DebugPad に自動操作状態を設定します。
    ::nn::Result SetDebugPadAutoPilotState(
        const ::nn::hid::debug::DebugPadAutoPilotState& value) NN_NOEXCEPT;

    //!< DebugPad の自動操作状態を解除します。
    ::nn::Result UnsetDebugPadAutoPilotState() NN_NOEXCEPT;

    //!< Keyboard をアクティブ化します。
    ::nn::Result ActivateKeyboard() NN_NOEXCEPT;

    //!< Keyboard を非アクティブ化します。
    ::nn::Result DeactivateKeyboard() NN_NOEXCEPT;

    //!< Keyboard のアプレットリソースの状態を保証します。
    ::nn::Result EnsureKeyboardAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Keyboard に自動操作状態を設定します。
    ::nn::Result SetKeyboardAutoPilotState(
        const ::nn::hid::debug::KeyboardAutoPilotState& value) NN_NOEXCEPT;

    //!< Keyboard の自動操作状態を解除します。
    ::nn::Result UnsetKeyboardAutoPilotState() NN_NOEXCEPT;

    //!< Keyboard のロックキーイベントを送信します。
    ::nn::Result SendKeyboardLockKeyEvent(
        ::nn::hid::system::KeyboardLockKeyEventSet value) NN_NOEXCEPT;

    //!< Mouse をアクティブ化します。
    ::nn::Result ActivateMouse() NN_NOEXCEPT;

    //!< Mouse を非アクティブ化します。
    ::nn::Result DeactivateMouse() NN_NOEXCEPT;

    //!< Mouse のアプレットリソースの状態を保証します。
    ::nn::Result EnsureMouseAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Mouse に自動操作状態を設定します。
    ::nn::Result SetMouseAutoPilotState(
        const ::nn::hid::debug::MouseAutoPilotState& value) NN_NOEXCEPT;

    //!< Mouse の自動操作状態を解除します。
    ::nn::Result UnsetMouseAutoPilotState() NN_NOEXCEPT;

    //!< コントロールマネージャーをアクティブ化します。
    ::nn::Result ActivateAbstractedPadXcdManager() NN_NOEXCEPT;

    //!< コントロールマネージャーを非アクティブ化します。
    ::nn::Result DeactivateAbstractedPadXcdManager() NN_NOEXCEPT;

    //!< Xpad をアクティブ化します。
    ::nn::Result ActivateXpad() NN_NOEXCEPT;

    //!< Xpad を非アクティブ化します。
    ::nn::Result DeactivateXpad() NN_NOEXCEPT;

    //!< Xpad のサンプリングを有効化します。
    void EnableXpadSampling() NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad のアプレットリソースの状態を保証します。
    ::nn::Result EnsureBasicXpadAppletResource(
        ::nn::applet::AppletResourceUserId aruid,
        int playerNumber) NN_NOEXCEPT;

    //!< JoyXpad の LIFO のハンドルを取得します。
    ::nn::Result GetJoyXpadLifoHandle(
        ::nn::os::NativeHandle* pOutValue, int playerNumber) NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad に自動操作状態を設定します。
    ::nn::Result SetBasicXpadAutoPilotState(
        int playerNumber,
        const ::nn::hid::debug::BasicXpadAutoPilotState& value) NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad の自動操作状態を解除します。
    ::nn::Result UnsetBasicXpadAutoPilotState(int playerNumber) NN_NOEXCEPT;

    //!< 抽象化されたコントローラーに BasicXapd 相当の自動操作状態を設定します。
    ::nn::Result SetAbstractedPadAutoPilotStateWithBasicXpad(
        int playerNumber,
        const ::nn::hid::debug::BasicXpadAutoPilotState& value) NN_NOEXCEPT;

    //!< 抽象化されたコントローラーの自動操作状態を解除します。
    ::nn::Result UnsetAbstractedPadAutoPilotState(int playerNumber) NN_NOEXCEPT;

    //!< 抽象化されたコントローラーに自動操作状態を設定します。
    ::nn::Result SetAbstractedPadAutoPilotVirtualPadState(
        const ::nn::hid::debug::AbstractedVirtualPadId& virtualPadId,
        const ::nn::hid::debug::AbstractedPadState& value) NN_NOEXCEPT;

    //!< 抽象化されたコントローラーの自動操作状態を解除します。
    ::nn::Result UnsetAbstractedPadAutoPilotVirtualPadState(
        const ::nn::hid::debug::AbstractedVirtualPadId& virtualPadId) NN_NOEXCEPT;

    //<! 抽象化されたコントローラーの自動操作状態を全て解除します。
    ::nn::Result UnsetAllAbstractedPadAutoPilotVirtualPadState() NN_NOEXCEPT;

    //!< 6軸センサーをアクティブ化します。
    ::nn::Result ActivateSixAxisSensor() NN_NOEXCEPT;

    //!< 6軸センサーを非アクティブ化します。
    ::nn::Result DeactivateSixAxisSensor() NN_NOEXCEPT;

    //!< 6軸センサーの LIFO のハンドルを取得します。
    ::nn::Result GetSixAxisSensorLifoHandle(
        ::nn::os::NativeHandle* pOutValue, int playerNumber) NN_NOEXCEPT;

    //!< 6軸センサーをアクティブ化します。
    ::nn::Result ActivateJoySixAxisSensor() NN_NOEXCEPT;

    //!< 6軸センサーを非アクティブ化します。
    ::nn::Result DeactivateJoySixAxisSensor() NN_NOEXCEPT;

    //!< 6軸センサーの LIFO のハンドルを取得します。
    ::nn::Result GetJoySixAxisSensorLifoHandle(
        ::nn::os::NativeHandle* pOutValue, int playerNumber) NN_NOEXCEPT;

    //!< Pad 共通のタイマーを初期化します
    ::nn::Result ActivatePadCommonTimer() NN_NOEXCEPT;

    //!< Pad 共通のタイマーを初期化します
    ::nn::Result DeactivatePadCommonTimer() NN_NOEXCEPT;

    //!< ペアリング更新時に通知を受けるイベントを設定します。
    void BindPairingEvent(::nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

    //!< ペアリング更新時に通知を受けるイベントを解除します。
    void UnbindPairingEvent(::nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

    //!< 有線ペアリングされたデバイスを列挙し、オーバーレイ通知を発行します
    void HandlePairing() NN_NOEXCEPT;

    //!< InputDetector をアクティブ化します。
    ::nn::Result ActivateInputDetector() NN_NOEXCEPT;

    //!< InputDetector を非アクティブ化します。
    ::nn::Result DeactivateInputDetector() NN_NOEXCEPT;

    //!< InputDetector に入力変化を通知します。
    ::nn::Result NotifyInputDetector(const ::nn::hid::system::InputSourceIdSet& id) NN_NOEXCEPT;

    //!< InputDetector のアプレットリソースの状態を保証します。
    ::nn::Result EnsureInputDetectorAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< ホームボタンの入力通知のハンドルを確保します。
    ::nn::Result AcquireHomeButtonEventHandle(
        ::nn::os::NativeHandle* pOutValue,
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< ホームボタンをアクティブ化します。
    ::nn::Result ActivateHomeButton() NN_NOEXCEPT;

    //!< ホームボタンを非アクティブ化します。
    ::nn::Result DeactivateHomeButton() NN_NOEXCEPT;

    //!< ホームボタンのアプレットリソースの状態を保証します。
    ::nn::Result EnsureHomeButtonAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< ホームボタンに自動操作状態を設定します。
    ::nn::Result SetHomeButtonAutoPilotState(
        const ::nn::hid::debug::HomeButtonAutoPilotState& value) NN_NOEXCEPT;

    //!< ホームボタンの自動操作状態を解除します。
    ::nn::Result UnsetHomeButtonAutoPilotState() NN_NOEXCEPT;

    //!< スリープボタンの入力通知のハンドルを確保します。
    ::nn::Result AcquireSleepButtonEventHandle(
        ::nn::os::NativeHandle* pOutValue,
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< スリープボタンをアクティブ化します。
    ::nn::Result ActivateSleepButton() NN_NOEXCEPT;

    //!< スリープボタンを非アクティブ化します。
    ::nn::Result DeactivateSleepButton() NN_NOEXCEPT;

    //!< スリープボタンのアプレットリソースの状態を保証します。
    ::nn::Result EnsureSleepButtonAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< スリープボタンに自動操作状態を設定します。
    ::nn::Result SetSleepButtonAutoPilotState(
        const ::nn::hid::debug::SleepButtonAutoPilotState& value) NN_NOEXCEPT;

    //!< スリープボタンの自動操作状態を解除します。
    ::nn::Result UnsetSleepButtonAutoPilotState() NN_NOEXCEPT;

    //!< 撮影ボタンの入力通知のハンドルを確保します。
    ::nn::Result AcquireCaptureButtonEventHandle(
        ::nn::os::NativeHandle* pOutValue,
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 撮影ボタンをアクティブ化します。
    ::nn::Result ActivateCaptureButton() NN_NOEXCEPT;

    //!< 撮影ボタンを非アクティブ化します。
    ::nn::Result DeactivateCaptureButton() NN_NOEXCEPT;

    //!< 撮影ボタンのアプレットリソースの状態を保証します。
    ::nn::Result EnsureCaptureButtonAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 撮影ボタンに自動操作状態を設定します。
    ::nn::Result SetCaptureButtonAutoPilotState(
        const ::nn::hid::debug::CaptureButtonAutoPilotState& value) NN_NOEXCEPT;

    //!< 撮影ボタンの自動操作状態を解除します。
    ::nn::Result UnsetCaptureButtonAutoPilotState() NN_NOEXCEPT;

    //!< Npad を使用するかどうか
    void IsNpadUsed(bool isUsed) NN_NOEXCEPT;

    //!< プレイレポート送信用のサービスをアクティブ化します。
    void ActivatePlayReport() NN_NOEXCEPT;

    //!< プレイレポート送信用のサービスをを非アクティブにします。
    void DeactivatePlayReport() NN_NOEXCEPT;

    //!< コントローラーの登録処理のサービスをアクティブ化します。
    void ActivateRegisteredDeviceManagement() NN_NOEXCEPT;

    //!< コントローラーの登録処理のサービスをを非アクティブにします。
    void DeactivateRegisteredDeviceManagement() NN_NOEXCEPT;

    //!< コントローラーの利用状態が変化したときに受ける通知をイベントオブジェクトにバインドします
    Result AcquirePlayReportControllerUsageUpdateEventHandle(
        ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT;

    //!< 現在のコントローラーの利用状態の取得を行います。
    int GetPlayReportControllerUsages(
        system::PlayReportControllerUsage* pOutValues, int count) NN_NOEXCEPT;

    //!< コントローラーの登録情報が変化したときに受ける通知をイベントオブジェクトにバインドします
    Result AcquirePlayReportRegisteredDevicesEventHandle(
        ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT;

    //!< 現在のコントローラーの登録情報の取得を行います。【互換性維持用】
    int GetRegisteredDevices(system::RegisteredDeviceOld* pOutValues, int count) NN_NOEXCEPT;

    //!< 現在のコントローラーの登録情報の取得を行います。
    int GetRegisteredDevices(system::RegisteredDevice* pOutValues, int count) NN_NOEXCEPT;

    //!< コントローラーの登録情報を追加します。
    void AddRegisteredDevices(const system::RegisteredDevice& device) NN_NOEXCEPT;

    //!< コントローラーへの接続処理がタイムアウトした際の通知をうけるイベントハンドルを取得します
    ::nn::Result AcquireConnectionTriggerTimeoutEvent(::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT;

    //!< コントローラーに対して接続を試みます
    ::nn::Result SendConnectionTrigger(nn::bluetooth::Address& address) NN_NOEXCEPT;

    //!< コントローラーが追加登録されたことをコントローラーサポートアプレットに通知するためのイベントをバインドします
    Result AcquireDeviceRegisteredEventForControllerSupport(
        ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT;

    //!< Pro Controller 有線 USB 通信機能が有効かどうかを取得します。
    ::nn::Result IsUsbFullKeyControllerEnabled(bool* pOutEnabled) NN_NOEXCEPT;

    //!< Pro Controller 有線 USB 通信機能を有効にします。
    ::nn::Result EnableUsbFullKeyController(bool enabled) NN_NOEXCEPT;

    //!< Xcd デバイスをすべて Reset します。
    ::nn::Result ResetDevices() NN_NOEXCEPT;

    ::nn::Result ActivateAudioControl() NN_NOEXCEPT;

    ::nn::Result DeactivateAudioControl() NN_NOEXCEPT;

    //!< オーディオコントロールの入力通知のハンドルを確保します。
    ::nn::Result AcquireAudioControlEventHandle(
        ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT;

    ::nn::Result GetAudioControlStates(
        int* pOutCount, system::AudioControlState outStates[], int count
        ) NN_NOEXCEPT;

protected:
    //!< タイマーイベントをリンクします。
    virtual void LinkTimerEvent(
        ::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT NN_OVERRIDE;

    //!< タイマーイベントをアンリンクします。
    virtual void UnlinkTimerEvent() NN_NOEXCEPT NN_OVERRIDE;

    //!< サンプリング処理を実行します。
    virtual void DoSampling(
        const ::nn::os::MultiWaitHolderType* waitId) NN_NOEXCEPT NN_OVERRIDE;

    virtual void LinkTimers(SynchronizedTimerGroup* group) NN_NOEXCEPT NN_OVERRIDE;

    virtual void UnlinkTimers() NN_NOEXCEPT NN_OVERRIDE;

    virtual void DoSamplingForTimer(const SynchronizedTimer* waitId) NN_NOEXCEPT NN_OVERRIDE;

private:
    //!< Pad 共通のタイマーイベントをアクティブ化します。
    ::nn::Result ActivatePadCommonTimerEvent() NN_NOEXCEPT;

    //!< Pad 共通のタイマーイベントをアクティブ化します。
    ::nn::Result DeactivatePadCommonTimerEvent() NN_NOEXCEPT;

    //!< Pad 共通のタイマーイベントを解決します。
    bool ResolvePadCommonTimerEvent(
        const SynchronizedTimer* waitId) NN_NOEXCEPT;

    //!< DebugPad のタイマーイベントを解決します。
    bool ResolveDebugPadTimerEvent(const SynchronizedTimer* waitId) NN_NOEXCEPT;

    //!< Keyboard のタイマーイベントを解決します。
    bool ResolveKeyboardTimerEvent(const SynchronizedTimer* waitId) NN_NOEXCEPT;

    //!< Mouse のタイマーイベントを解決します。
    bool ResolveMouseTimerEvent(const SynchronizedTimer* waitId) NN_NOEXCEPT;

    //!< Xpad のタイマーイベントを解決します。
    bool ResolveXpadTimerEvent(const SynchronizedTimer* waitId) NN_NOEXCEPT;

    //!< ホームボタンのタイマーイベントを解決します。
    bool ResolveHomeButtonTimerEvent(
        const SynchronizedTimer* waitId) NN_NOEXCEPT;

    //!< スリープボタンのタイマーイベントを解決します。
    bool ResolveSleepButtonTimerEvent(
        const SynchronizedTimer* waitId) NN_NOEXCEPT;

    //!< 撮影ボタンのタイマーイベントを解決します。
    bool ResolveCaptureButtonTimerEvent(
        const SynchronizedTimer* waitId) NN_NOEXCEPT;

    //!< プレイレポートのタイマーイベントを解決します。
    bool ResolvePlayReportTimerEvent(
        const SynchronizedTimer* waitId) NN_NOEXCEPT;

    //!< オーディオコントロールのタイマーイベントを解決します。
    bool ResolveAudioControlTimerEvent(
        const SynchronizedTimer* waitId) NN_NOEXCEPT;
};

//!< ahid のタスクを扱うクラスです。
class AhidTask final : public Task
{
private:
    //!< ahid 操作排他用のミューテックス
    static ::nn::os::SdkMutexType s_AhidMutex;

    //!< ahid のタイマーイベント
    SynchronizedTimer m_AhidTimerEvent;

    //!< ahid マネージャ
    AhidManager* m_pAhidManager;

public:
    AhidTask() NN_NOEXCEPT;

    virtual ~AhidTask() NN_NOEXCEPT NN_OVERRIDE;

    //!< ahid マネージャを設定します。
    void SetAhidManager(AhidManager* pManager) NN_NOEXCEPT;

    //!< ahid をアクティブ化します。
    ::nn::Result ActivateAhid() NN_NOEXCEPT;

    //!< ahid を非アクティブ化します。
    ::nn::Result DeactivateAhid() NN_NOEXCEPT;

protected:
    //!< タイマーイベントをリンクします。
    virtual void LinkTimerEvent(
        ::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT NN_OVERRIDE;

    //!< タイマーイベントをアンリンクします。
    virtual void UnlinkTimerEvent() NN_NOEXCEPT NN_OVERRIDE;

    //!< サンプリング処理を実行します。
    virtual void DoSampling(
        const ::nn::os::MultiWaitHolderType* waitId) NN_NOEXCEPT NN_OVERRIDE;

    //!< 同期タイマーイベントをリンクします。
    virtual void LinkTimers(
        SynchronizedTimerGroup* group) NN_NOEXCEPT NN_OVERRIDE;

    //!< 同期タイマーイベントをアンリンクします。
    virtual void UnlinkTimers() NN_NOEXCEPT NN_OVERRIDE;

    //!< 同期タイマーイベントに基づくサンプリング処理を実行します。
    virtual void DoSamplingForTimer(
        const SynchronizedTimer* waitId) NN_NOEXCEPT NN_OVERRIDE;
};

//!< TouchScreen のタスクを扱うクラスです。
class TouchScreenTask final : public Task
{
private:
    //!< TouchScreen 操作排他用のミューテックス
    ::nn::os::SdkMutex m_TouchScreenMutex;

    //!< TouchScreen のタイマーイベント
    MultiWaitTimerEvent m_TouchScreenTimerEvent;

    //!< TouchScreen マネージャ
    TouchScreenManager* m_pTouchScreenManager;

public:
    TouchScreenTask() NN_NOEXCEPT;

    virtual ~TouchScreenTask() NN_NOEXCEPT NN_OVERRIDE;

    //!< TouchScreen マネージャを設定します。
    void SetTouchScreenManager(TouchScreenManager* pManager) NN_NOEXCEPT;

    //!< Gesture をアクティブ化します。
    ::nn::Result ActivateGesture() NN_NOEXCEPT;

    //!< Gesture を非アクティブ化します。
    ::nn::Result DeactivateGesture() NN_NOEXCEPT;

    //!< Gesture のアプレットリソースの状態を保証します。
    ::nn::Result EnsureGestureAppletResource(
        ::nn::applet::AppletResourceUserId aruid, int32_t version) NN_NOEXCEPT;

    //!< TouchScreen をアクティブ化します。
    ::nn::Result ActivateTouchScreen() NN_NOEXCEPT;

    //!< TouchScreen を非アクティブ化します。
    ::nn::Result DeactivateTouchScreen() NN_NOEXCEPT;

    //!< TouchScreen のアプレットリソースの状態を保証します。
    ::nn::Result EnsureTouchScreenAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< TouchScreen に自動操作状態を設定します。
    ::nn::Result SetTouchScreenAutoPilotState(
        const ::nn::hid::debug::TouchScreenAutoPilotState<
            TouchStateCountMax>& value) NN_NOEXCEPT;

    //!< TouchScreen の自動操作状態を解除します。
    ::nn::Result UnsetTouchScreenAutoPilotState() NN_NOEXCEPT;

    //!< TouchScreen を起床状態に遷移させます。
    ::nn::Result WakeTouchScreenUp() NN_NOEXCEPT;

    //!< TouchScreen をスリープ状態に遷移させます。
    ::nn::Result PutTouchScreenToSleep() NN_NOEXCEPT;

protected:
    //!< タイマーイベントをリンクします。
    virtual void LinkTimerEvent(
        ::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT NN_OVERRIDE;

    //!< タイマーイベントをアンリンクします。
    virtual void UnlinkTimerEvent() NN_NOEXCEPT NN_OVERRIDE;

    //!< サンプリング処理を実行します。
    virtual void DoSampling(
        const ::nn::os::MultiWaitHolderType* waitId) NN_NOEXCEPT NN_OVERRIDE;
};

//!< ConsoleSixAxisSensor のタスクを扱うクラスです。
class ConsoleSixAxisSensorTask final : public Task
{
private:
    //!< ConsoleSixAxisSensor 操作排他用のミューテックス
    ::nn::os::SdkMutex m_ConsoleSixAxisSensorMutex;

    //!< ConsoleSixAxisSensor のタイマーイベント
    MultiWaitTimerEvent m_ConsoleSixAxisSensorTimerEvent;

    //!< ConsoleSixAxisSensor マネージャ
    ConsoleSixAxisSensorManager* m_pConsoleSixAxisSensorManager;

public:
    ConsoleSixAxisSensorTask() NN_NOEXCEPT;

    virtual ~ConsoleSixAxisSensorTask() NN_NOEXCEPT NN_OVERRIDE;

    //!< ConsoleSixAxisSensor マネージャを設定します。
    void SetConsoleSixAxisSensorManager(ConsoleSixAxisSensorManager* pManager) NN_NOEXCEPT;

    //!< ConsoleSixAxisSensor をアクティブ化します。
    ::nn::Result ActivateConsoleSixAxisSensor() NN_NOEXCEPT;

    //!< ConsoleSixAxisSensor を非アクティブ化します。
    ::nn::Result DeactivateConsoleSixAxisSensor() NN_NOEXCEPT;

    //!< ConsoleSixAxisSensor のアプレットリソースの状態を保証します。
    ::nn::Result EnsureConsoleSixAxisSensorAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< ConsoleSixAxisSensor を起床状態に遷移させます。
    ::nn::Result WakeConsoleSixAxisSensorUp() NN_NOEXCEPT;

    //!< ConsoleSixAxisSensor をスリープ状態に遷移させます。
    ::nn::Result PutConsoleSixAxisSensorToSleep() NN_NOEXCEPT;

    //!< ConsoleSixAxisSensor の生値を取得します。
    ::nn::Result GetConsoleSixAxisSensorCountStates(
        int* pOutCount,
        ::nn::hid::debug::ConsoleSixAxisSensorCountState* pOutValues,
        int count) NN_NOEXCEPT;

    //!< ConsoleSixAxisSensor のサンプリング間隔を取得します。
    ::nn::Result GetConsoleSixAxisSensorSamplingFrequency(
        int64_t* pOutValue, ::nn::applet::AppletResourceUserId aruid
        ) NN_NOEXCEPT;

    //!< ConsoleSixAxisSensor のキャリブレーション値を取得します。
    ::nn::Result GetConsoleSixAxisSensorCalibrationValues(
        tmp::ConsoleSixAxisSensorCalibrationValues* pOutValues) NN_NOEXCEPT;

    //!< 本体 6 軸センサーのユーザーキャリブレーション機能のサポート有無を取得します。
    ::nn::Result IsConsoleSixAxisSensorUserCalibrationSupported(
        bool* pOutIsSupported) NN_NOEXCEPT;

    //!< 工程出荷時の本体 6 軸センサーのキャリブレーション値を書き込みます。
    ::nn::Result ResetConsoleSixAxisSensorCalibrationValues() NN_NOEXCEPT;

    //!< 本体 6 軸センサーのユーザーキャリブレーション処理を開始します。
    ::nn::Result StartConsoleSixAxisSensorUserCalibration() NN_NOEXCEPT;

    //!< 進行中の本体 6 軸センサーのユーザーキャリブレーション処理をキャンセルします。
    ::nn::Result CancelConsoleSixAxisSensorUserCalibration() NN_NOEXCEPT;

    //!< 6 軸センサーの高精度ユーザーキャリブレーションの進捗を取得します。
    ::nn::Result GetSixAxisSensorAccurateUserCalibrationState(
        ::nn::hid::system::SixAxisSensorAccurateUserCalibrationState* pOutState
        ) NN_NOEXCEPT;

    //!< アプリから割り当てられた入力状態格納用バッファの Handle を設定します。
    ::nn::Result SetStateBufferMemoryHandle(
        const ::nn::applet::AppletResourceUserId& aruid,
        ::nn::os::NativeHandle&& transferMemoryHandle,
        size_t size, bool isManaged) NN_NOEXCEPT;

    //!< アプリから割り当てられたワークバッファの Handle を設定します。
    ::nn::Result SetWorkBufferMemoryHandle(
        const ::nn::applet::AppletResourceUserId& aruid,
        ::nn::os::NativeHandle&& transferMemoryHandle,
        size_t size, bool isManaged) NN_NOEXCEPT;

    //!< Seven 向け SixAxisSensor を初期化します。
    ::nn::Result InitializeSevenSixAxisSensor(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Seven 向け SixAxisSensor を終了します。
    ::nn::Result FinalizeSevenSixAxisSensor(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

protected:
    //!< タイマーイベントをリンクします。
    virtual void LinkTimerEvent(
        ::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT NN_OVERRIDE;

    //!< タイマーイベントをアンリンクします。
    virtual void UnlinkTimerEvent() NN_NOEXCEPT NN_OVERRIDE;

    //!< サンプリング処理を実行します。
    virtual void DoSampling(
        const ::nn::os::MultiWaitHolderType* waitId) NN_NOEXCEPT NN_OVERRIDE;
};

//!< XcdDriver のタスクを扱うクラスです。
class XcdTask final : public Task
{
private:
    //!< Xcd 操作排他用のミューテックス
    ::nn::os::SdkMutex m_XcdMutex;

    //!< Xcd ドライバ
    XcdDriver* m_pXcdDriver;

public:
    XcdTask() NN_NOEXCEPT;

    virtual ~XcdTask() NN_NOEXCEPT NN_OVERRIDE;

    //!< XpadId マネージャを設定します。
    void SetXcdDriver(XcdDriver* pXcdDriver) NN_NOEXCEPT;

    //!< Xcd ドライバをアクティブ化します。
    ::nn::Result ActivateXcdDriver() NN_NOEXCEPT;

    //!< Xcd ドライバを非アクティブ化します。
    ::nn::Result DeactivateXcdDriver() NN_NOEXCEPT;

    //!< Xcd ドライバをサスペンドします。
    ::nn::Result Suspend() NN_NOEXCEPT;

    //!< Xcd ドライバをレジュームします。
    ::nn::Result Resume() NN_NOEXCEPT;

    //!< Xcd ドライバの有線デバイス登録を有効にします。
    ::nn::Result EnableWiredDeviceRegistration() NN_NOEXCEPT;

    //!< Xcd ドライバの有線デバイス登録を無効にします。
    ::nn::Result DisableWiredDeviceRegistration() NN_NOEXCEPT;

protected:
    //!< イベントをリンクします。(名称はいったんTaskを引き継ぎます)
    virtual void LinkTimerEvent(
        ::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT NN_OVERRIDE;

    //!< イベントをアンリンクします。(名称はいったんTaskを引き継ぎます)
    virtual void UnlinkTimerEvent() NN_NOEXCEPT NN_OVERRIDE;

    //!< 定期処理を実行します。(名称はいったんTaskを引き継ぎます)
    virtual void DoSampling(
        const ::nn::os::MultiWaitHolderType* waitId) NN_NOEXCEPT NN_OVERRIDE;

    //!< イベントの受付を開始します。
    virtual void InvokeEventLoop() NN_NOEXCEPT NN_OVERRIDE;
};

//!< リソースマネージャを扱うクラスです。
class ResourceManager final
{
    NN_DISALLOW_COPY(ResourceManager);
    NN_DISALLOW_MOVE(ResourceManager);

private:
    //!< Xpad のプレイヤーの最大数
    static const int XpadPlayerCountMax = 4;

    //!< 共有タスクのスレッドのスタック領域
    static NN_ALIGNAS(0x1000) char s_SharedTaskThreadStack[0x2000];

    //!< ahid タスクのスレッドのスタック領域
    static NN_ALIGNAS(0x1000) char s_AhidTaskThreadStacks[
        2 * AhidPortCount + 1][0x1000];

    //!< TouchScreen タスクのスレッドのスタック領域
    static NN_ALIGNAS(0x1000) char s_TouchScreenTaskThreadStack[0x1000];

    //!< ConsoleSixAxisSensor タスクのスレッドのスタック領域
    static NN_ALIGNAS(0x1000) char s_ConsoleSixAxisSensorTaskThreadStack[0x2000];

    //!< XCD を駆動させるスレッドのスタック領域
    static NN_ALIGNAS(0x1000) char s_XcdTaskThreadStack[0x3000];

    //!< 共有タスク
    SharedTask m_SharedTask;

    //!< ahid タスク
    AhidTask m_AhidTasks[2 * AhidPortCount + 1];

    //!< TouchScreen タスク
    TouchScreenTask m_TouchScreenTask;

    //!< ConsoleSixAxisSensor タスク
    ConsoleSixAxisSensorTask m_ConsoleSixAxisSensorTask;

    //!< Xcd タスク
    XcdTask m_XcdTask;

    //!< DebugPad ドライバ
    DebugPadDriver m_DebugPadDriver;

    //!< Keyboard ドライバ
    KeyboardDriver m_KeyboardDriver;

    //!< Mouse ドライバ
    MouseDriver m_MouseDriver;

    //!< TouchScreen ドライバ
    TouchScreenDriver m_TouchScreenDriver;

    //!< ConsoleSixAxisSensor ドライバ
    ConsoleSixAxisSensorDriver m_ConsoleSixAxisSensorDriver;

    //!< Xpad ドライバ
    XpadDriver m_XpadDrivers[XpadPlayerCountMax];

    //!< スリープボタンドライバ
    SleepButtonDriver m_SleepButtonDriver;

    //!< オーディオコントロールドライバ
    AudioControlDriver m_AudioControlDriver;

    //!< ahid マネージャ
    AhidManager m_AhidManagers[2 * AhidPortCount + 1];

    //!< DebugPad マネージャ
    DebugPadManager m_DebugPadManager;

    //!< Keyboard マネージャ
    KeyboardManager m_KeyboardManager;

    //!< Mouse マネージャ
    MouseManager m_MouseManager;

    //!< TouchScreen マネージャ
    TouchScreenManager m_TouchScreenManager;

    //!< ConsoleSixAxisSensor マネージャ
    ConsoleSixAxisSensorManager m_ConsoleSixAxisSensorManager;

    //!< Xpad マネージャ
    XpadManager m_XpadManagers[XpadPlayerCountMax];

    //!< Xpad ID マネージャのミューテックス
    ::nn::os::SdkMutex m_XpadIdManagerMutex;

    //!< Xpad ID マネージャ
    XpadIdManager m_XpadIdManager;

    //!< InputDetector 操作排他用のミューテックス
    ::nn::os::SdkMutex m_InputDetectorMutex;

    //!< InputDetector マネージャ
    InputDetectorManager m_InputDetectorManager;

    //!< ホームボタンマネージャ
    HomeButtonManager m_HomeButtonManager;

    //!< スリープボタンマネージャ
    SleepButtonManager m_SleepButtonManager;

    //!< 撮影ボタンマネージャ
    CaptureButtonManager m_CaptureButtonManager;

    //!< アプレットリソース操作排他用のミューテックス
    ::nn::os::SdkRecursiveMutex m_AppletResourceMutex;

    //!< アプレットリソースマネージャ
    AppletResourceManager m_AppletResourceManager;

    //!< SixAxisSensor の設定値管理マネージャ
    SixAxisSensorAppletSettingManager m_SixAxisSensorAppletSettingManager;

    //!< ConsoleSixAxisSensor の設定値管理マネージャ
    ConsoleSixAxisSensorAppletSettingManager m_ConsoleSixAxisSensorAppletSettingManager;

    //!< Handheld マネージャ
    HandheldManager m_HandheldManager;

    //!< オーディオコントロールマネージャ
    AudioControlManager m_AudioControlManager;

    //!< ConsoleSixAxisSensorProcessor
    ConsoleSixAxisSensorProcessor m_ConsoleSixAxisSensorProcessor;

    //!< SixAxisSensorSanjiroFilter
    SixAxisSensorSanjiroFilter m_SixAxisSensorSanjiroFilter;

    //!< Xcd ドライバ
    XcdDriver m_XcdDriver;

    //!< オーバーレイ通知を送信するためのマネージャー
    OvlnSenderManager m_OvlnSender;

    //!< 割り込みシーン通知をハンドリングするクラス
    InterruptSceneNotifier m_InterruptSceneNotifier;

    //!< プレイレポートを送信するためのマネージャー
    PlayReportManager m_PlayReportManager;

    //!< コントローラーの登録情報を管理するためのマネージャー
    RegisteredDeviceManager m_RegisteredDeviceManager;

    //!< 自動入力の抽象化インタフェース
    AbstractedAutoPilotPad m_AbstractedAutoPilotPad[AbstractedAutoPilotPadCount];

    //!< 自動入力の抽象化 Pad を管理するマネージャー
    AbstractedAutoPilotPadManager m_AbstractedAutoPilotPadManager;

    //!< 自動入力の抽象化インタフェース
    AbstractedAutoPilotVirtualPad m_AbstractedAutoPilotVirtualPad[AbstractedAutoPilotVirtualPadCount];

    //!< 自動入力の抽象化 Pad を管理するマネージャー
    AbstractedAutoPilotVirtualPadManager m_AbstractedAutoPilotVirtualPadManager;

    //!< Usb コントローラー の抽象化インタフェース
    AbstractedPadUsb m_AbstractedPadUsb[AbstractedPadUsbDriver::PortCount];

    //!< Usb コントローラー の割り当てマネージャー
    AbstractedPadUsbDriver m_AbstractedPadUsbDriver;

    //!< Xcd の抽象化インタフェース
    AbstractedPadXcd m_AbstractedPadXcd[AbstractedPadXcdCountMax];

    //!< Xcd の割り当てマネージャー
    AbstractedPadXcdManager m_AbstractedPadXcdManager;

    //!< Gc コントローラーアダプターの抽象化インタフェース
    AbstractedGcAdapterPad m_AbstractedGcAdapterPads[AbstractedGcAdapterPadCountMax];

    //!< Gc コントローラーアダプターを管理するマネージャー
    AbstractedGcAdapterPadManager m_AbstractedGcAdapterPadManager;

    //!< Palma の抽象化インタフェース
    AbstractedPalmaPad m_AbstractedPalmaPad[AbstractedPalmaPadCountMax];

    //!< Palma の割り当てマネージャー
    AbstractedPalmaPadManager m_AbstractedPalmaPadManager;

    //!< AbstractedPadId 発行マネージャー
    AbstractedPadIdPublisher m_AbstractedIdPublisher;

    //!< Fifty の抽象化インタフェース
    AbstractedFiftyPad m_AbstractedFiftyPad;

    //!< Npad のリソースマネージャー
    NpadResourceManager m_NpadResourceManager;

    //!< UniquePad のリソースマネージャー
    UniquePadResourceManager m_UniquePadResourceManager;

    //!< Palma のリソースマネージャー
    PalmaResourceManager m_PalmaResourceManager;

    //!< レールマネージャー
    RailManager m_RailManager;

    //!< Gc コントローラーアダプターのドライバー
    GcAdapterDriver m_GcAdapterDriver;

    //!< AbstractedPad のマネージャー
    AbstractedPadManager m_AbstractedPadManager;

public:
    ResourceManager() NN_NOEXCEPT;

    //!< DebugPad タスクを返します。
    SharedTask& GetDebugPadTask() NN_NOEXCEPT;

    //!< Keyboard タスクを返します。
    SharedTask& GetKeyboardTask() NN_NOEXCEPT;

    //!< Mouse タスクを返します。
    SharedTask& GetMouseTask() NN_NOEXCEPT;

    //!< ホームボタンタスクを返します。
    SharedTask& GetHomeButtonTask() NN_NOEXCEPT;

    //!< スリープボタンタスクを返します。
    SharedTask& GetSleepButtonTask() NN_NOEXCEPT;

    //!< 撮影ボタンタスクを返します。
    SharedTask& GetCaptureButtonTask() NN_NOEXCEPT;

    //!< Gesture タスクを返します。
    TouchScreenTask& GetGestureTask() NN_NOEXCEPT;

    //!< TouchScreen タスクを返します。
    TouchScreenTask& GetTouchScreenTask() NN_NOEXCEPT;

    //!< ConsoleSixAxisSensor タスクを返します。
    ConsoleSixAxisSensorTask& GetConsoleSixAxisSensorTask() NN_NOEXCEPT;

    //!< Xpad タスクを返します。
    SharedTask& GetXpadTask() NN_NOEXCEPT;

    //!< InputDetector タスクを返します。
    SharedTask& GetInputDetectorTask() NN_NOEXCEPT;

    //!< ahid タスクを返します。
    AhidTask (&GetAhidTasks() NN_NOEXCEPT)[2 * AhidPortCount + 1];

    //!< Xpad Control タスクを返します。
    SharedTask& GetXpadControlTask() NN_NOEXCEPT;

    //!< Xcd タスクを返します。
    XcdTask& GetXcdTask() NN_NOEXCEPT;

    //!< AbstractedPadXcd タスクを返します。
    SharedTask& GetAbstractedPadXcdTask() NN_NOEXCEPT;

    //!< AbstractedAutoPilotPad タスクを返します。
    SharedTask& GetAbstractedAutoPilotPadTask() NN_NOEXCEPT;

    //!< プレイレポートタスク返します。
    SharedTask& GetPlayReportTask() NN_NOEXCEPT;

    //!< 登録デバイスタスク返します。
    SharedTask& GetRegisteredDeviceTask() NN_NOEXCEPT;

    //!< Xcd ファームウェアアップデータタスク を返します。
    SharedTask& GetFirmwareUpdaterTask() NN_NOEXCEPT;

    //!< オーディオコントロールタスク を返します。
    SharedTask& GetAudioControlTask() NN_NOEXCEPT;

    //!< Npad のリソースマネージャーを返します。
    NpadResourceManager& GetNpadResourceManager() NN_NOEXCEPT;

    //!< UniquePad のリソースマネージャーを返します。
    UniquePadResourceManager& GetUniquePadResourceManager() NN_NOEXCEPT;

    //!< AbstractedPad のマネージャを返します。
    AbstractedPadManager& GetAbstractedPadManager() NN_NOEXCEPT;

    //!< Palma のリソースマネージャーを返します。
    PalmaResourceManager& GetPalmaResourceManager() NN_NOEXCEPT;

    //!< ARUID を設定します。
    void SetAppletResourceUserId(
        ::nn::applet::AppletResourceUserId value) NN_NOEXCEPT;

    //!< 携帯機で利用できる Hid を有効にします。
    void EnableHandheldHids() NN_NOEXCEPT;

    //!< 携帯機で利用できる Hid を無効にします。
    void DisableHandheldHids() NN_NOEXCEPT;

    //!< アプレットリソースユーザ ID を登録します。
    ::nn::Result RegisterAppletResourceUserId(
        ::nn::applet::AppletResourceUserId aruid,
        bool enablesInput) NN_NOEXCEPT;

    //!< アプレットリソースユーザ ID の登録を解除します。
    void UnregisterAppletResourceUserId(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< アプレットリソースのエントリを設定します。
    ::nn::Result SetAppletResourceEntry(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< アプレットリソースのエントリを削除します。
    void ResetAppletResourceEntry(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< HID 入力の提供を行うか否かを設定します。
    void EnableAppletToGetInput(
        ::nn::applet::AppletResourceUserId aruid,
        bool enablesInput) NN_NOEXCEPT;

    //!< 6 軸センサーの提供を行うか否かを設定します。
    void EnableAppletToGetSixAxisSensor(
        ::nn::applet::AppletResourceUserId aruid,
        bool enablesInput) NN_NOEXCEPT;

    //!< アプレットリソースの共有メモリのハンドルを返します。
    ::nn::Result GetAppletResourceSharedMemoryHandle(
        ::nn::os::NativeHandle* pOutValue,
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 共有アプレットリソースをアクティブ化します。
    ::nn::Result ActivateSharedAppletResource() NN_NOEXCEPT;

    //!< 共有アプレットリソースを非アクティブ化します。
    ::nn::Result DeactivateSharedAppletResource() NN_NOEXCEPT;

    //!< SixAxisSensorProcessor を返します。
    const SixAxisSensorProcessor& GetSixAxisSensorProcessor(
        const SixAxisSensorHandle& handle) const NN_NOEXCEPT;

    //!< SixAxisSensorAppletSettingManager を返します。
    SixAxisSensorAppletSettingManager& GetSixAxisSensorAppletSettingManager() NN_NOEXCEPT;

    //!< ConsoleSixAxisSensorAppletSettingManager を返します。
    ConsoleSixAxisSensorAppletSettingManager&
    GetConsoleSixAxisSensorAppletSettingManager() NN_NOEXCEPT;

    //!< XcdDriver を返します。
    XcdDriver& GetXcdDriver() NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad のプレイヤー番号が有効であることを表明します。
    ::nn::Result AssertValidBasicXpadPlayerNumber(int playerNumber) NN_NOEXCEPT;

    //!< JoyXpad のプレイヤー番号が有効であることを表明します。
    ::nn::Result AssertValidJoyXpadPlayerNumber(int playerNumber) NN_NOEXCEPT;

    //!< Xpad ID の再解決イベントのハンドルを確保します。
    ::nn::Result AcquireXpadIdEventHandle(
        ::nn::os::NativeHandle* outHandle, uint64_t clientId) NN_NOEXCEPT;

    //!< Xpad ID の再解決イベントのハンドルを開放します。
    ::nn::Result ReleaseXpadIdEventHandle(uint64_t clientId) NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad の解決された ID を取得します。
    ::nn::Result GetXpadIds(int* outCount, BasicXpadId* outXpadIds, int count
                            ) NN_NOEXCEPT;

    //!< JoyXpad の解決された ID を取得します。
    ::nn::Result GetJoyXpadIds(
        int* outCount, JoyXpadId* outXpadIds, int count) NN_NOEXCEPT;

    //!< 通知送信サービスをアクティブにします。
    void ActivateSenderForOverlay() NN_NOEXCEPT;

    //!< 通知送信サービスを非アクティブにします。
    void DeactivateSenderForOverlay() NN_NOEXCEPT;

    //!< Bluetooth オフ中のジョイコン取り外し通知のハンドルを確保します
    ::nn::Result AcquireJoyDetachOnBluetoothOffEventHandle(
        ::nn::os::NativeHandle* pOutValue,
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Pad 共通のタイマーを初期化します
    ::nn::Result ActivatePadCommonTimer() NN_NOEXCEPT;

    //!< Pad 共通のタイマーを初期化します
    ::nn::Result DeactivatePadCommonTimer() NN_NOEXCEPT;

    //!< タッチパネルの Reset をディアサートする。
    ::nn::Result DeassertTouchIcReset() NN_NOEXCEPT;

    // VibrationDeviceHandle が有効であることを表明します。
    ::nn::Result AssertValidVibrationDeviceHandle(const VibrationDeviceHandle& handle) NN_NOEXCEPT;

    //!< VibrationDeviceHandle から VibrationDeviceInfo を取得します。
    ::nn::Result GetVibrationDeviceInfo(
        VibrationDeviceInfo* pOutValue,
        const VibrationDeviceHandle& handle) NN_NOEXCEPT;

    //!< IVibratorDriver を返します。
    IVibratorDriver& GetVibrator(const VibrationDeviceHandle& handle) NN_NOEXCEPT;

    //!< VibratorXcdDriver を返します。
    VibratorXcdDriver* GetVibratorXcd(const VibrationDeviceHandle& handle) NN_NOEXCEPT;

    //!< VibratorGcDriver を返します。
    VibratorGcDriver* GetVibratorGc(const VibrationDeviceHandle& handle) NN_NOEXCEPT;

    //!< 振動の制御権を持つアプレットリソースユーザ ID であるかを判定します。
    ::nn::Result IsAruidValidForVibration(
        ::nn::applet::AppletResourceUserId aruid, bool* pOutValue) NN_NOEXCEPT;

    //!< 振動の制御権を持つアプレットリソースユーザ ID を設定します。
    ::nn::Result SetAruidValidForVibration(
        ::nn::applet::AppletResourceUserId aruid, bool enable) NN_NOEXCEPT;

    //!< 一時的な振動強制ON期間を開始します。
    ::nn::Result BeginPermitVibrationSession(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 一時的な振動強制ON期間を停止します。
    ::nn::Result EndPermitVibrationSession() NN_NOEXCEPT;

    //!< 全ての振動子を一旦リセットします。
    ::nn::Result ResetAllVibrationDevices() NN_NOEXCEPT;

private:
    //!< アプレットリソースマネージャを注入します。
    template<typename ManagerT>
    void InjectAppletResourceManager(ManagerT* pManager) NN_NOEXCEPT;

    //!< InputDetector マネージャを注入します。
    template<typename ManagerT>
    void InjectInputDetectorManager(ManagerT* pManager) NN_NOEXCEPT;

    //!< DebugPad マネージャを初期化します。
    void InitializeDebugPadManager() NN_NOEXCEPT;

    //!< Keyboard マネージャを初期化します。
    void InitializeKeyboardManager() NN_NOEXCEPT;

    //!< Mouse マネージャを初期化します。
    void InitializeMouseManager() NN_NOEXCEPT;

    //!< ホームボタンマネージャを初期化します。
    void InitializeHomeButtonManager() NN_NOEXCEPT;

    //!< スリープボタンマネージャを初期化します。
    void InitializeSleepButtonManager() NN_NOEXCEPT;

    //!< 撮影ボタンマネージャを初期化します。
    void InitializeCaptureButtonManager() NN_NOEXCEPT;

    //!< コントローラーの登録情報を管理するためのマネージャーを初期化します。
    void InitializeRegisteredDeviceManager() NN_NOEXCEPT;

    //!< Xpad マネージャを初期化します。
    void InitializeXpadManagers() NN_NOEXCEPT;

    //!< Xcd の割り当てマネージャーを初期化します。
    void InitializeAbstractedPadXcdManager() NN_NOEXCEPT;

    //!< Palma の割り当てマネージャーを初期化します。
    void InitializeAbstractedPalmaPadManager() NN_NOEXCEPT;

    //!< Gc コントローラーアダプターを管理するマネージャーを初期化します。
    void InitializeAbstractedGcAdapterPadManager() NN_NOEXCEPT;

    //!< 自動入力の抽象化 Pad を管理するマネージャーを初期化します。
    void InitializeAbstractedAutoPilotPadManager() NN_NOEXCEPT;

    //!< 自動入力の抽象化 Pad を管理するマネージャー
    void InitializeAbstractedAutoPilotVirtualPadManager() NN_NOEXCEPT;

    //!< 共有タスクを初期化します。
    void InitializeSharedTask() NN_NOEXCEPT;

    //!< TouchScreen タスクを初期化します。
    void InitializeTouchScreenTask() NN_NOEXCEPT;

    //!< ConsoleSixAxisSensor タスクを初期化します。
    void InitializeConsoleSixAxisSensorTask() NN_NOEXCEPT;

    //!< ahid タスクを初期化します。
    void InitializeAhidTask() NN_NOEXCEPT;

    //!< UniquePad を初期化します。
    void InitializeUniquePad() NN_NOEXCEPT;

    //!< UniquePad を初期化します。
    void InitializeNpad() NN_NOEXCEPT;

    //!< AbstractedPadManager を初期化します。
    void InitializeAbstractedPadManager() NN_NOEXCEPT;
};

//!< リソースマネージャを返します。
ResourceManager& GetResourceManager() NN_NOEXCEPT;

}}} // namespace nn::hid::detail
