﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_TimerEventTypes.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Mutex.h>
#include <nn/hid/system/hid_RegisteredDevice.h>
#include <nn/xcd/xcd.h>

#include "hid_ActivationCount.h"
#include "hid_IAbstractedPad.h"
#include "hid_OvlnSenderManager.h"

namespace nn { namespace hid { namespace detail {

class RegisteredDeviceManager
{
    NN_DISALLOW_COPY(RegisteredDeviceManager);
    NN_DISALLOW_MOVE(RegisteredDeviceManager);

private:
    //!< RegisteredDeviceManager がアクティベートされた回数
    ActivationCount m_ActivationCount;

    //!< コントローラーの登録情報
    system::RegisteredDevice m_RegisteredDevices[system::RegisteredDeviceCountMax];

    //!< コントローラーの登録情報の数
    int m_RegisteredDeviceCount;

    //!< 最後に通知を行ったペアリング情報
    int64_t m_LastRegistrationDeviceCount;

    //!< オーバーレイ通知を送信するための型
    OvlnSenderManager* m_pSender;

    //!< コントローラーへの登録処理がタイムアウトしたときに通知するためのイベント
    ::nn::os::SystemEvent m_ConnectionTriggerTimeoutEvent;

    //!< 登録情報の排他制御のための mutex
    ::nn::os::Mutex m_Mutex;

    //!< コントローラーが新たに登録された際に通知をうけるイベント
    ::nn::os::SystemEvent m_DeviceRegisteredEvent;

    //!< プレイレポート用に変化があった場合に通知を行うイベント
    ::nn::os::SystemEvent m_PlayReportUpdateEvent;

    //!< IAbstractedPad
    IAbstractedPad* m_pAbstractedPads[IAbstractedPadCountMax];

    //!< IAbstractedPad の数
    int m_AbstractedPadCount;

public:
    RegisteredDeviceManager() NN_NOEXCEPT;

    ::nn::Result Activate() NN_NOEXCEPT;

    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< IAbstractedPad をセットします
    void AddIAbstractedPad(IAbstractedPad* pPads) NN_NOEXCEPT;

    //!< ペアリング更新時に通知を受けるイベントを設定します。
    void BindXcdRegistrationEvent(::nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

    //!< ペアリング更新時に通知を受けるイベントを解除します。
    void UnbindXcdRegistrationEvent(::nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

    //!< コントローラーへの接続通知がタイムアウトする際の通知を受けるイベントを設定します
    void BindXcdConnectionTriggerTimeoutEvent(::nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

    //!< ペアリング更新時に通知を受けるイベントを解除します。
    void UnindXcdConnectionTriggerTimeoutEvent(::nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

    //!< コントローラーへの接続通知がタイムアウトした場合に呼ばれる処理
    void SignalConnectionTriggerTimeout() NN_NOEXCEPT;

    //!< オーバーレイ通知用のマネージャーを設定します。
    void SetOvlnSenderManager(OvlnSenderManager* pSender) NN_NOEXCEPT;

    //!< プレイレポート用に定常的な更新処理を行います
    void UpdateRegisteredDevice() NN_NOEXCEPT;

    //!< 現在のコントローラーの登録情報の取得を行います。【互換性維持用】
    int GetRegisteredDevices(system::RegisteredDeviceOld* pOutValues, int count) NN_NOEXCEPT;

    //!< 現在のコントローラーの登録情報の取得を行います。
    int GetRegisteredDevices(system::RegisteredDevice* pOutValues, int count) NN_NOEXCEPT;

    //!< コントローラーの登録情報を追加します。
    void AddRegisteredDevice(const system::RegisteredDevice& device) NN_NOEXCEPT;

    //!< コントローラーへの接続処理がタイムアウトした際の通知をうけるイベントハンドルを取得します
    ::nn::Result AcquireConnectionTriggerTimeoutEvent(::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT;

    //!< コントローラーに対して接続を試みます
    ::nn::Result SendConnectionTrigger(nn::bluetooth::Address& address) NN_NOEXCEPT;

    //!< コントローラーが新たに登録された際に通知をうけるイベントハンドルを取得します
    ::nn::Result AcquireDeviceRegisteredEventForControllerSupport(::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT;

    //!< プレイレポート用のコントローラーの登録情報が変化したときに受ける通知をイベントオブジェクトにバインドします
    Result AcquireUpdateForPlayReportEventHandle(::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT;

private:
    //!< コントローラーが有線ペアリングされたことを通知する
    void NotifyPairingComplete(system::NpadDeviceTypeSet device,
                               const NpadControllerColor& color,
                               ::nn::xcd::InterfaceType interface) NN_NOEXCEPT;

};

}}} // namespace nn::hid::detail
