﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_TimerEventTypes.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Mutex.h>
#include <nn/hid/system/hid_PlayReport.h>

#include "hid_ActivationCount.h"
#include "hid_RegisteredDeviceManager.h"
#include "hid_SynchronizedTimer.h"

namespace nn { namespace hid { namespace detail {

//!< プレイレポートの更新管理用のフラグです
struct PlayReportUpdateFlag final
{
    typedef ::nn::util::BitFlagSet<32, PlayReportUpdateFlag>::Flag<0>
        Usage;                       //!< コントローラーの利用状態の更新
};

//!< プレイレポートの更新管理用のフラグ集合を扱う型です。
typedef ::nn::util::BitFlagSet<32, PlayReportUpdateFlag> PlayReportUpdateFlagSet;

class PlayReportManager
{
    NN_DISALLOW_COPY(PlayReportManager);
    NN_DISALLOW_MOVE(PlayReportManager);

public:
    //!< サンプリング間隔
    static const ::nn::TimeSpan SamplingInterval;

private:
    //!< タイマーイベント
    SynchronizedTimer* m_pTimer;

    //!< PlayReportManager がアクティベートされた回数
    ActivationCount m_ActivationCount;

    //!< プレイレポート用に成型されたコントローラーの操作状態
    system::PlayReportControllerUsage m_Usages[system::PlayReportControllerUsageCountMax];

    //!< コントローラーの操作状態の数
    int m_UsagesCount;

    //!< コントローラーの利用状態に更新があった場合に通知するためのイベント
    ::nn::os::SystemEvent m_UsageUpdateEvent;

    //!< プレイレポートの更新管理用のフラグの集合です
    PlayReportUpdateFlagSet m_UpdateFlags;

    //!< プレイレポート情報の排他制御のための mutex
    ::nn::os::Mutex m_Mutex;

public:
    PlayReportManager() NN_NOEXCEPT;

    ::nn::Result Activate() NN_NOEXCEPT;

    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< タイマーイベントを設定します。
    void SetTimer(SynchronizedTimer* timer) NN_NOEXCEPT;

    //!< プレイレポート用に定常的な更新処理を行います
    void CheckForUpdates() NN_NOEXCEPT;

    //!< コントローラーの利用状態が変化したときに受ける通知をイベントオブジェクトにバインドします
    Result AcquireUsageUpdateEventHandle(::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT;

    //!< 現在のコントローラーの利用状態の取得を行います。
    int GetControllerUsages(system::PlayReportControllerUsage* pOutValues, int count) NN_NOEXCEPT;

    //!< 利用状態にあるデバイスを追加します。登録済みデバイスであった場合は更新します
    void UpdateDeviceInUse(uint8_t controllerNumber, system::PlayReportControllerUsage* pValues, int count) NN_NOEXCEPT;

private:
    //!< 利用状態のデバイスリストに指定のデバイスが含まれるかを探索します
    bool GetIndexOfDevice(int* pOutIndex, uint8_t controllerNumber, system::PlayReportDeviceType device) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
